"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeAction = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * CodePipeline invoke Action that is provided by an AWS Lambda function.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-invoke-lambda-function.html
 */
class LambdaInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.lambda,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 5,
                minOutputs: 0,
                maxOutputs: 5,
            },
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the Lambda function this action points to.
     * Variables in Lambda invoke actions are defined by calling the PutJobSuccessResult CodePipeline API call
     * with the 'outputVariables' property filled.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'outputVariables' section of the PutJobSuccessResult
     *   request that the Lambda function calls when the action is invoked
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // allow pipeline to list functions
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*'],
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [this.props.lambda.functionArn],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        this.props.lambda.addToRolePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['codepipeline:PutJobSuccessResult', 'codepipeline:PutJobFailureResult'],
        }));
        return {
            configuration: {
                FunctionName: this.props.lambda.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString(this.props.userParameters),
            },
        };
    }
}
exports.LambdaInvokeAction = LambdaInvokeAction;
//# sourceMappingURL=data:application/json;base64,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