"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudFrontWebDistribution = exports.GeoRestriction = exports.ViewerCertificate = exports.LambdaEdgeEventType = exports.CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedMethods = exports.OriginProtocolPolicy = exports.OriginSslPolicy = exports.SecurityPolicyProtocol = exports.SSLMethod = exports.ViewerProtocolPolicy = exports.PriceClass = exports.HttpVersion = void 0;
const certificatemanager = require("../../aws-certificatemanager"); // Automatically re-written from '@aws-cdk/aws-certificatemanager'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
var HttpVersion;
(function (HttpVersion) {
    HttpVersion["HTTP1_1"] = "http1.1";
    HttpVersion["HTTP2"] = "http2";
})(HttpVersion = exports.HttpVersion || (exports.HttpVersion = {}));
/**
 * The price class determines how many edge locations CloudFront will use for your distribution.
 */
var PriceClass;
(function (PriceClass) {
    PriceClass["PRICE_CLASS_100"] = "PriceClass_100";
    PriceClass["PRICE_CLASS_200"] = "PriceClass_200";
    PriceClass["PRICE_CLASS_ALL"] = "PriceClass_All";
})(PriceClass = exports.PriceClass || (exports.PriceClass = {}));
/**
 * How HTTPs should be handled with your distribution.
 */
var ViewerProtocolPolicy;
(function (ViewerProtocolPolicy) {
    ViewerProtocolPolicy["HTTPS_ONLY"] = "https-only";
    ViewerProtocolPolicy["REDIRECT_TO_HTTPS"] = "redirect-to-https";
    ViewerProtocolPolicy["ALLOW_ALL"] = "allow-all";
})(ViewerProtocolPolicy = exports.ViewerProtocolPolicy || (exports.ViewerProtocolPolicy = {}));
/**
 * The SSL method CloudFront will use for your distribution.
 *
 * Server Name Indication (SNI) - is an extension to the TLS computer networking protocol by which a client indicates
 *  which hostname it is attempting to connect to at the start of the handshaking process. This allows a server to present
 *  multiple certificates on the same IP address and TCP port number and hence allows multiple secure (HTTPS) websites
 * (or any other service over TLS) to be served by the same IP address without requiring all those sites to use the same certificate.
 *
 * CloudFront can use SNI to host multiple distributions on the same IP - which a large majority of clients will support.
 *
 * If your clients cannot support SNI however - CloudFront can use dedicated IPs for your distribution - but there is a prorated monthly charge for
 * using this feature. By default, we use SNI - but you can optionally enable dedicated IPs (VIP).
 *
 * See the CloudFront SSL for more details about pricing : https://aws.amazon.com/cloudfront/custom-ssl-domains/
 *
 */
var SSLMethod;
(function (SSLMethod) {
    SSLMethod["SNI"] = "sni-only";
    SSLMethod["VIP"] = "vip";
})(SSLMethod = exports.SSLMethod || (exports.SSLMethod = {}));
/**
 * The minimum version of the SSL protocol that you want CloudFront to use for HTTPS connections.
 * CloudFront serves your objects only to browsers or devices that support at least the SSL version that you specify.
 */
var SecurityPolicyProtocol;
(function (SecurityPolicyProtocol) {
    SecurityPolicyProtocol["SSL_V3"] = "SSLv3";
    SecurityPolicyProtocol["TLS_V1"] = "TLSv1";
    SecurityPolicyProtocol["TLS_V1_2016"] = "TLSv1_2016";
    SecurityPolicyProtocol["TLS_V1_1_2016"] = "TLSv1.1_2016";
    SecurityPolicyProtocol["TLS_V1_2_2018"] = "TLSv1.2_2018";
})(SecurityPolicyProtocol = exports.SecurityPolicyProtocol || (exports.SecurityPolicyProtocol = {}));
var OriginSslPolicy;
(function (OriginSslPolicy) {
    OriginSslPolicy["SSL_V3"] = "SSLv3";
    OriginSslPolicy["TLS_V1"] = "TLSv1";
    OriginSslPolicy["TLS_V1_1"] = "TLSv1.1";
    OriginSslPolicy["TLS_V1_2"] = "TLSv1.2";
})(OriginSslPolicy = exports.OriginSslPolicy || (exports.OriginSslPolicy = {}));
var OriginProtocolPolicy;
(function (OriginProtocolPolicy) {
    OriginProtocolPolicy["HTTP_ONLY"] = "http-only";
    OriginProtocolPolicy["MATCH_VIEWER"] = "match-viewer";
    OriginProtocolPolicy["HTTPS_ONLY"] = "https-only";
})(OriginProtocolPolicy = exports.OriginProtocolPolicy || (exports.OriginProtocolPolicy = {}));
/**
 * An enum for the supported methods to a CloudFront distribution.
 */
var CloudFrontAllowedMethods;
(function (CloudFrontAllowedMethods) {
    CloudFrontAllowedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedMethods["GET_HEAD_OPTIONS"] = "GHO";
    CloudFrontAllowedMethods["ALL"] = "ALL";
})(CloudFrontAllowedMethods = exports.CloudFrontAllowedMethods || (exports.CloudFrontAllowedMethods = {}));
/**
 * Enums for the methods CloudFront can cache.
 */
var CloudFrontAllowedCachedMethods;
(function (CloudFrontAllowedCachedMethods) {
    CloudFrontAllowedCachedMethods["GET_HEAD"] = "GH";
    CloudFrontAllowedCachedMethods["GET_HEAD_OPTIONS"] = "GHO";
})(CloudFrontAllowedCachedMethods = exports.CloudFrontAllowedCachedMethods || (exports.CloudFrontAllowedCachedMethods = {}));
var LambdaEdgeEventType;
(function (LambdaEdgeEventType) {
    /**
     * The origin-request specifies the request to the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_REQUEST"] = "origin-request";
    /**
     * The origin-response specifies the response from the
     * origin location (e.g. S3)
     */
    LambdaEdgeEventType["ORIGIN_RESPONSE"] = "origin-response";
    /**
     * The viewer-request specifies the incoming request
     */
    LambdaEdgeEventType["VIEWER_REQUEST"] = "viewer-request";
    /**
     * The viewer-response specifies the outgoing reponse
     */
    LambdaEdgeEventType["VIEWER_RESPONSE"] = "viewer-response";
})(LambdaEdgeEventType = exports.LambdaEdgeEventType || (exports.LambdaEdgeEventType = {}));
/**
 * Viewer certificate configuration class
 */
class ViewerCertificate {
    constructor(props, aliases = []) {
        this.props = props;
        this.aliases = aliases;
    }
    /**
     * Generate an AWS Certificate Manager (ACM) viewer certificate configuration
     *
     * @param certificate AWS Certificate Manager (ACM) certificate.
     *                    Your certificate must be located in the us-east-1 (US East (N. Virginia)) region to be accessed by CloudFront
     * @param options certificate configuration options
     */
    static fromAcmCertificate(certificate, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            acmCertificateArn: certificate.certificateArn, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate an IAM viewer certificate configuration
     *
     * @param iamCertificateId Identifier of the IAM certificate
     * @param options certificate configuration options
     */
    static fromIamCertificate(iamCertificateId, options = {}) {
        const { sslMethod: sslSupportMethod = SSLMethod.SNI, securityPolicy: minimumProtocolVersion, aliases, } = options;
        return new ViewerCertificate({
            iamCertificateId, sslSupportMethod, minimumProtocolVersion,
        }, aliases);
    }
    /**
     * Generate a viewer certifcate configuration using
     * the CloudFront default certificate (e.g. d111111abcdef8.cloudfront.net)
     * and a {@link SecurityPolicyProtocol.TLS_V1} security policy.
     *
     * @param aliases Alternative CNAME aliases
     *                You also must create a CNAME record with your DNS service to route queries
     */
    static fromCloudFrontDefaultCertificate(...aliases) {
        return new ViewerCertificate({ cloudFrontDefaultCertificate: true }, aliases);
    }
}
exports.ViewerCertificate = ViewerCertificate;
/**
 * Controls the countries in which your content is distributed.
 */
class GeoRestriction {
    /**
     * Creates an instance of GeoRestriction for internal use
     *
     * @param restrictionType Specifies the restriction type to impose (whitelist or blacklist)
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist/blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    constructor(restrictionType, locations) {
        this.restrictionType = restrictionType;
        this.locations = locations;
    }
    /**
     * Whitelist specific countries which you want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to whitelist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static whitelist(...locations) {
        return new GeoRestriction('whitelist', GeoRestriction.validateLocations(locations));
    }
    /**
     * Blacklist specific countries which you don't want CloudFront to distribute your content.
     *
     * @param locations Two-letter, uppercase country code for a country
     * that you want to blacklist. Include one element for each country.
     * See ISO 3166-1-alpha-2 code on the *International Organization for Standardization* website
     */
    static blacklist(...locations) {
        return new GeoRestriction('blacklist', GeoRestriction.validateLocations(locations));
    }
    static validateLocations(locations) {
        if (locations.length === 0) {
            throw new Error('Should provide at least 1 location');
        }
        locations.forEach(location => {
            if (!GeoRestriction.LOCATION_REGEX.test(location)) {
                throw new Error(`Invalid location format for location: ${location}, location should be two-letter and uppercase country ISO 3166-1-alpha-2 code`);
            }
        });
        return locations;
    }
}
exports.GeoRestriction = GeoRestriction;
GeoRestriction.LOCATION_REGEX = /^[A-Z]{2}$/;
/**
 * Amazon CloudFront is a global content delivery network (CDN) service that securely delivers data, videos,
 * applications, and APIs to your viewers with low latency and high transfer speeds.
 * CloudFront fronts user provided content and caches it at edge locations across the world.
 *
 * Here's how you can use this construct:
 *
 * ```ts
 * import { CloudFrontWebDistribution } from '@aws-cdk/aws-cloudfront'
 *
 * const sourceBucket = new Bucket(this, 'Bucket');
 *
 * const distribution = new CloudFrontWebDistribution(this, 'MyDistribution', {
 *  originConfigs: [
 *    {
 *      s3OriginSource: {
 *      s3BucketSource: sourceBucket
 *      },
 *      behaviors : [ {isDefaultBehavior: true}]
 *    }
 *  ]
 * });
 * ```
 *
 * This will create a CloudFront distribution that uses your S3Bucket as it's origin.
 *
 * You can customize the distribution using additional properties from the CloudFrontWebDistributionProps interface.
 *
 *
 */
class CloudFrontWebDistribution extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Maps our methods to the string arrays they are
         */
        this.METHOD_LOOKUP_MAP = {
            GH: ['GET', 'HEAD'],
            GHO: ['GET', 'HEAD', 'OPTIONS'],
            ALL: ['DELETE', 'GET', 'HEAD', 'OPTIONS', 'PATCH', 'POST', 'PUT'],
        };
        /**
         * Maps for which SecurityPolicyProtocol are available to which SSLMethods
         */
        this.VALID_SSL_PROTOCOLS = {
            [SSLMethod.SNI]: [
                SecurityPolicyProtocol.TLS_V1, SecurityPolicyProtocol.TLS_V1_1_2016,
                SecurityPolicyProtocol.TLS_V1_2016, SecurityPolicyProtocol.TLS_V1_2_2018,
            ],
            [SSLMethod.VIP]: [SecurityPolicyProtocol.SSL_V3, SecurityPolicyProtocol.TLS_V1],
        };
        let distributionConfig = {
            comment: props.comment,
            enabled: true,
            defaultRootObject: props.defaultRootObject !== undefined ? props.defaultRootObject : 'index.html',
            httpVersion: props.httpVersion || HttpVersion.HTTP2,
            priceClass: props.priceClass || PriceClass.PRICE_CLASS_100,
            ipv6Enabled: (props.enableIpV6 !== undefined) ? props.enableIpV6 : true,
            // tslint:disable-next-line:max-line-length
            customErrorResponses: props.errorConfigurations,
            webAclId: props.webACLId,
        };
        const behaviors = [];
        const origins = [];
        let originIndex = 1;
        for (const originConfig of props.originConfigs) {
            const originId = `origin${originIndex}`;
            if (!originConfig.s3OriginSource && !originConfig.customOriginSource) {
                throw new Error('There must be at least one origin source - either an s3OriginSource or a customOriginSource');
            }
            if (originConfig.customOriginSource && originConfig.s3OriginSource) {
                throw new Error('There cannot be both an s3OriginSource and a customOriginSource in the same SourceConfiguration.');
            }
            const originHeaders = [];
            if (originConfig.originHeaders) {
                Object.keys(originConfig.originHeaders).forEach(key => {
                    const oHeader = {
                        headerName: key,
                        headerValue: originConfig.originHeaders[key],
                    };
                    originHeaders.push(oHeader);
                });
            }
            let s3OriginConfig;
            if (originConfig.s3OriginSource) {
                // first case for backwards compatibility
                if (originConfig.s3OriginSource.originAccessIdentity) {
                    // grant CloudFront OriginAccessIdentity read access to S3 bucket
                    originConfig.s3OriginSource.s3BucketSource.grantRead(originConfig.s3OriginSource.originAccessIdentity);
                    s3OriginConfig = {
                        originAccessIdentity: `origin-access-identity/cloudfront/${originConfig.s3OriginSource.originAccessIdentity.originAccessIdentityName}`,
                    };
                }
                else {
                    s3OriginConfig = {};
                }
            }
            const originProperty = {
                id: originId,
                domainName: originConfig.s3OriginSource
                    ? originConfig.s3OriginSource.s3BucketSource.bucketRegionalDomainName
                    : originConfig.customOriginSource.domainName,
                originPath: originConfig.originPath,
                originCustomHeaders: originHeaders.length > 0 ? originHeaders : undefined,
                s3OriginConfig,
                customOriginConfig: originConfig.customOriginSource
                    ? {
                        httpPort: originConfig.customOriginSource.httpPort || 80,
                        httpsPort: originConfig.customOriginSource.httpsPort || 443,
                        originKeepaliveTimeout: originConfig.customOriginSource.originKeepaliveTimeout
                            && originConfig.customOriginSource.originKeepaliveTimeout.toSeconds() || 5,
                        originReadTimeout: originConfig.customOriginSource.originReadTimeout
                            && originConfig.customOriginSource.originReadTimeout.toSeconds() || 30,
                        originProtocolPolicy: originConfig.customOriginSource.originProtocolPolicy || OriginProtocolPolicy.HTTPS_ONLY,
                        originSslProtocols: originConfig.customOriginSource.allowedOriginSSLVersions || [OriginSslPolicy.TLS_V1_2],
                    }
                    : undefined,
            };
            for (const behavior of originConfig.behaviors) {
                behaviors.push({ ...behavior, targetOriginId: originId });
            }
            origins.push(originProperty);
            originIndex++;
        }
        origins.forEach(origin => {
            if (!origin.s3OriginConfig && !origin.customOriginConfig) {
                throw new Error(`Origin ${origin.domainName} is missing either S3OriginConfig or CustomOriginConfig. At least 1 must be specified.`);
            }
        });
        distributionConfig = {
            ...distributionConfig,
            origins,
        };
        const defaultBehaviors = behaviors.filter(behavior => behavior.isDefaultBehavior);
        if (defaultBehaviors.length !== 1) {
            throw new Error('There can only be one default behavior across all sources. [ One default behavior per distribution ].');
        }
        distributionConfig = { ...distributionConfig, defaultCacheBehavior: this.toBehavior(defaultBehaviors[0], props.viewerProtocolPolicy) };
        const otherBehaviors = [];
        for (const behavior of behaviors.filter(b => !b.isDefaultBehavior)) {
            if (!behavior.pathPattern) {
                throw new Error('pathPattern is required for all non-default behaviors');
            }
            otherBehaviors.push(this.toBehavior(behavior, props.viewerProtocolPolicy));
        }
        distributionConfig = { ...distributionConfig, cacheBehaviors: otherBehaviors.length > 0 ? otherBehaviors : undefined };
        if (props.aliasConfiguration && props.viewerCertificate) {
            throw new Error([
                'You cannot set both aliasConfiguration and viewerCertificate properties.',
                'Please only use viewerCertificate, as aliasConfiguration is deprecated.',
            ].join(' '));
        }
        let _viewerCertificate = props.viewerCertificate;
        if (props.aliasConfiguration) {
            const { acmCertRef, securityPolicy, sslMethod, names: aliases } = props.aliasConfiguration;
            _viewerCertificate = ViewerCertificate.fromAcmCertificate(certificatemanager.Certificate.fromCertificateArn(this, 'AliasConfigurationCert', acmCertRef), { securityPolicy, sslMethod, aliases });
        }
        if (_viewerCertificate) {
            const { props: viewerCertificate, aliases } = _viewerCertificate;
            Object.assign(distributionConfig, { aliases, viewerCertificate });
            const { minimumProtocolVersion, sslSupportMethod } = viewerCertificate;
            if (minimumProtocolVersion != null && sslSupportMethod != null) {
                const validProtocols = this.VALID_SSL_PROTOCOLS[sslSupportMethod];
                if (validProtocols.indexOf(minimumProtocolVersion.toString()) === -1) {
                    // tslint:disable-next-line:max-line-length
                    throw new Error(`${minimumProtocolVersion} is not compabtible with sslMethod ${sslSupportMethod}.\n\tValid Protocols are: ${validProtocols.join(', ')}`);
                }
            }
        }
        else {
            distributionConfig = { ...distributionConfig,
                viewerCertificate: { cloudFrontDefaultCertificate: true },
            };
        }
        if (props.loggingConfig) {
            this.loggingBucket = props.loggingConfig.bucket || new s3.Bucket(this, 'LoggingBucket');
            distributionConfig = {
                ...distributionConfig,
                logging: {
                    bucket: this.loggingBucket.bucketRegionalDomainName,
                    includeCookies: props.loggingConfig.includeCookies || false,
                    prefix: props.loggingConfig.prefix,
                },
            };
        }
        if (props.geoRestriction) {
            distributionConfig = {
                ...distributionConfig,
                restrictions: {
                    geoRestriction: {
                        restrictionType: props.geoRestriction.restrictionType,
                        locations: props.geoRestriction.locations,
                    },
                },
            };
        }
        const distribution = new cloudfront_generated_1.CfnDistribution(this, 'CFDistribution', { distributionConfig });
        this.node.defaultChild = distribution;
        this.domainName = distribution.attrDomainName;
        this.distributionId = distribution.ref;
    }
    toBehavior(input, protoPolicy) {
        let toReturn = {
            allowedMethods: this.METHOD_LOOKUP_MAP[input.allowedMethods || CloudFrontAllowedMethods.GET_HEAD],
            cachedMethods: this.METHOD_LOOKUP_MAP[input.cachedMethods || CloudFrontAllowedCachedMethods.GET_HEAD],
            compress: input.compress !== false,
            defaultTtl: input.defaultTtl && input.defaultTtl.toSeconds(),
            forwardedValues: input.forwardedValues || { queryString: false, cookies: { forward: 'none' } },
            maxTtl: input.maxTtl && input.maxTtl.toSeconds(),
            minTtl: input.minTtl && input.minTtl.toSeconds(),
            trustedSigners: input.trustedSigners,
            targetOriginId: input.targetOriginId,
            viewerProtocolPolicy: protoPolicy || ViewerProtocolPolicy.REDIRECT_TO_HTTPS,
        };
        if (!input.isDefaultBehavior) {
            toReturn = Object.assign(toReturn, { pathPattern: input.pathPattern });
        }
        if (input.lambdaFunctionAssociations) {
            toReturn = Object.assign(toReturn, {
                lambdaFunctionAssociations: input.lambdaFunctionAssociations
                    .map(fna => ({
                    eventType: fna.eventType,
                    lambdaFunctionArn: fna.lambdaFunction && fna.lambdaFunction.functionArn,
                })),
            });
            // allow edgelambda.amazonaws.com to assume the functions' execution role.
            for (const a of input.lambdaFunctionAssociations) {
                if (a.lambdaFunction.role && a.lambdaFunction.role instanceof iam.Role && a.lambdaFunction.role.assumeRolePolicy) {
                    a.lambdaFunction.role.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                        actions: ['sts:AssumeRole'],
                        principals: [new iam.ServicePrincipal('edgelambda.amazonaws.com')],
                    }));
                }
            }
        }
        return toReturn;
    }
}
exports.CloudFrontWebDistribution = CloudFrontWebDistribution;
//# sourceMappingURL=data:application/json;base64,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