"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthCheck = exports.ScalingProcess = exports.ScalingEvents = exports.ScalingEvent = exports.UpdateType = exports.AutoScalingGroup = exports.Monitoring = void 0;
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const elbv2 = require("../../aws-elasticloadbalancingv2"); // Automatically re-written from '@aws-cdk/aws-elasticloadbalancingv2'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const autoscaling_generated_1 = require("./autoscaling.generated");
const lifecycle_hook_1 = require("./lifecycle-hook");
const scheduled_action_1 = require("./scheduled-action");
const step_scaling_policy_1 = require("./step-scaling-policy");
const target_tracking_scaling_policy_1 = require("./target-tracking-scaling-policy");
const volume_1 = require("./volume");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The monitoring mode for instances launched in an autoscaling group
 */
var Monitoring;
(function (Monitoring) {
    /**
     * Generates metrics every 5 minutes
     */
    Monitoring[Monitoring["BASIC"] = 0] = "BASIC";
    /**
     * Generates metrics every minute
     */
    Monitoring[Monitoring["DETAILED"] = 1] = "DETAILED";
})(Monitoring = exports.Monitoring || (exports.Monitoring = {}));
class AutoScalingGroupBase extends core_1.Resource {
    /**
     * Send a message to either an SQS queue or SNS topic when instances launch or terminate
     */
    addLifecycleHook(id, props) {
        return new lifecycle_hook_1.LifecycleHook(this, `LifecycleHook${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in based on time
     */
    scaleOnSchedule(id, props) {
        return new scheduled_action_1.ScheduledAction(this, `ScheduledAction${id}`, {
            autoScalingGroup: this,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target CPU utilization
     */
    scaleOnCpuUtilization(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_CPU_UTILIZATION,
            targetValue: props.targetUtilizationPercent,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network ingress rate
     */
    scaleOnIncomingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_IN,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target network egress rate
     */
    scaleOnOutgoingBytes(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ASG_AVERAGE_NETWORK_OUT,
            targetValue: props.targetBytesPerSecond,
            ...props,
        });
    }
    /**
     * Scale out or in to achieve a target request handling rate
     *
     * The AutoScalingGroup must have been attached to an Application Load Balancer
     * in order to be able to call this.
     */
    scaleOnRequestCount(id, props) {
        if (this.albTargetGroup === undefined) {
            throw new Error('Attach the AutoScalingGroup to a non-imported Application Load Balancer before calling scaleOnRequestCount()');
        }
        const resourceLabel = `${this.albTargetGroup.firstLoadBalancerFullName}/${this.albTargetGroup.targetGroupFullName}`;
        const policy = new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            predefinedMetric: target_tracking_scaling_policy_1.PredefinedMetric.ALB_REQUEST_COUNT_PER_TARGET,
            targetValue: props.targetRequestsPerSecond,
            resourceLabel,
            ...props,
        });
        policy.node.addDependency(this.albTargetGroup.loadBalancerAttached);
        return policy;
    }
    /**
     * Scale out or in in order to keep a metric around a target value
     */
    scaleToTrackMetric(id, props) {
        return new target_tracking_scaling_policy_1.TargetTrackingScalingPolicy(this, `ScalingPolicy${id}`, {
            autoScalingGroup: this,
            customMetric: props.metric,
            ...props,
        });
    }
    /**
     * Scale out or in, in response to a metric
     */
    scaleOnMetric(id, props) {
        return new step_scaling_policy_1.StepScalingPolicy(this, id, { ...props, autoScalingGroup: this });
    }
}
/**
 * A Fleet represents a managed set of EC2 instances
 *
 * The Fleet models a number of AutoScalingGroups, a launch configuration, a
 * security group and an instance role.
 *
 * It allows adding arbitrary commands to the startup scripts of the instances
 * in the fleet.
 *
 * The ASG spans the availability zones specified by vpcSubnets, falling back to
 * the Vpc default strategy if not specified.
 */
class AutoScalingGroup extends AutoScalingGroupBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.securityGroups = [];
        this.loadBalancerNames = [];
        this.targetGroupArns = [];
        this.notifications = [];
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'InstanceSecurityGroup', {
            vpc: props.vpc,
            allowAllOutbound: props.allowAllOutbound !== false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            roleName: core_1.PhysicalName.GENERATE_IF_NEEDED,
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const launchConfig = new autoscaling_generated_1.CfnLaunchConfiguration(this, 'LaunchConfig', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            instanceMonitoring: (props.instanceMonitoring !== undefined ? (props.instanceMonitoring === Monitoring.DETAILED) : undefined),
            securityGroups: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            associatePublicIpAddress: props.associatePublicIpAddress,
            spotPrice: props.spotPrice,
            blockDeviceMappings: (props.blockDevices !== undefined ?
                synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined),
        });
        launchConfig.node.addDependency(this.role);
        // desiredCapacity just reflects what the user has supplied.
        const desiredCapacity = props.desiredCapacity;
        const minCapacity = props.minCapacity !== undefined ? props.minCapacity : 1;
        const maxCapacity = props.maxCapacity !== undefined ? props.maxCapacity :
            desiredCapacity !== undefined ? desiredCapacity : Math.max(minCapacity, 1);
        core_1.withResolved(minCapacity, maxCapacity, (min, max) => {
            if (min > max) {
                throw new Error(`minCapacity (${min}) should be <= maxCapacity (${max})`);
            }
        });
        core_1.withResolved(desiredCapacity, minCapacity, (desired, min) => {
            if (desired === undefined) {
                return;
            }
            if (desired < min) {
                throw new Error(`Should have minCapacity (${min}) <= desiredCapacity (${desired})`);
            }
        });
        core_1.withResolved(desiredCapacity, maxCapacity, (desired, max) => {
            if (desired === undefined) {
                return;
            }
            if (max < desired) {
                throw new Error(`Should have desiredCapacity (${desired}) <= maxCapacity (${max})`);
            }
        });
        if (desiredCapacity !== undefined) {
            this.node.addWarning('desiredCapacity has been configured. Be aware this will reset the size of your AutoScalingGroup on every deployment. See https://github.com/aws/aws-cdk/issues/5215');
        }
        this.maxInstanceLifetime = props.maxInstanceLifetime;
        if (this.maxInstanceLifetime &&
            (this.maxInstanceLifetime.toSeconds() < 604800 || this.maxInstanceLifetime.toSeconds() > 31536000)) {
            throw new Error('maxInstanceLifetime must be between 7 and 365 days (inclusive)');
        }
        if (props.notificationsTopic && props.notifications) {
            throw new Error('Cannot set \'notificationsTopic\' and \'notifications\', \'notificationsTopic\' is deprecated use \'notifications\' instead');
        }
        if (props.notificationsTopic) {
            this.notifications = [{
                    topic: props.notificationsTopic,
                }];
        }
        if (props.notifications) {
            this.notifications = props.notifications.map(nc => {
                var _a;
                return ({
                    topic: nc.topic,
                    scalingEvents: (_a = nc.scalingEvents) !== null && _a !== void 0 ? _a : ScalingEvents.ALL,
                });
            });
        }
        const { subnetIds, hasPublic } = props.vpc.selectSubnets(props.vpcSubnets);
        const asgProps = {
            cooldown: props.cooldown !== undefined ? props.cooldown.toSeconds().toString() : undefined,
            minSize: core_1.Tokenization.stringifyNumber(minCapacity),
            maxSize: core_1.Tokenization.stringifyNumber(maxCapacity),
            desiredCapacity: desiredCapacity !== undefined ? core_1.Tokenization.stringifyNumber(desiredCapacity) : undefined,
            launchConfigurationName: launchConfig.ref,
            loadBalancerNames: core_1.Lazy.listValue({ produce: () => this.loadBalancerNames }, { omitEmpty: true }),
            targetGroupArns: core_1.Lazy.listValue({ produce: () => this.targetGroupArns }, { omitEmpty: true }),
            notificationConfigurations: this.renderNotificationConfiguration(),
            vpcZoneIdentifier: subnetIds,
            healthCheckType: props.healthCheck && props.healthCheck.type,
            healthCheckGracePeriod: props.healthCheck && props.healthCheck.gracePeriod && props.healthCheck.gracePeriod.toSeconds(),
            maxInstanceLifetime: this.maxInstanceLifetime ? this.maxInstanceLifetime.toSeconds() : undefined,
        };
        if (!hasPublic && props.associatePublicIpAddress) {
            throw new Error("To set 'associatePublicIpAddress: true' you must select Public subnets (vpcSubnets: { subnetType: SubnetType.PUBLIC })");
        }
        this.autoScalingGroup = new autoscaling_generated_1.CfnAutoScalingGroup(this, 'ASG', asgProps);
        this.osType = imageConfig.osType;
        this.autoScalingGroupName = this.autoScalingGroup.ref;
        this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
            service: 'autoscaling',
            resource: 'autoScalingGroup:*:autoScalingGroupName',
            resourceName: this.autoScalingGroupName,
        });
        this.node.defaultChild = this.autoScalingGroup;
        this.applyUpdatePolicies(props);
        this.spotPrice = props.spotPrice;
    }
    static fromAutoScalingGroupName(scope, id, autoScalingGroupName) {
        class Import extends AutoScalingGroupBase {
            constructor() {
                super(...arguments);
                this.autoScalingGroupName = autoScalingGroupName;
                this.autoScalingGroupArn = core_1.Stack.of(this).formatArn({
                    service: 'autoscaling',
                    resource: 'autoScalingGroup:*:autoScalingGroupName',
                    resourceName: this.autoScalingGroupName,
                });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Add the security group to all instances via the launch configuration
     * security groups array.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Attach to a classic load balancer
     */
    attachToClassicLB(loadBalancer) {
        this.loadBalancerNames.push(loadBalancer.loadBalancerName);
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToApplicationTargetGroup(targetGroup) {
        if (this.albTargetGroup !== undefined) {
            throw new Error('Cannot add AutoScalingGroup to 2nd Target Group');
        }
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        if (targetGroup instanceof elbv2.ApplicationTargetGroup) {
            // Copy onto self if it's a concrete type. We need this for autoscaling
            // based on request count, which we cannot do with an imported TargetGroup.
            this.albTargetGroup = targetGroup;
        }
        targetGroup.registerConnectable(this);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Attach to ELBv2 Application Target Group
     */
    attachToNetworkTargetGroup(targetGroup) {
        this.targetGroupArns.push(targetGroup.targetGroupArn);
        return { targetType: elbv2.TargetType.INSTANCE };
    }
    /**
     * Add command to the startup script of fleet instances.
     * The command must be in the scripting language supported by the fleet's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by instances of this fleet.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Apply CloudFormation update policies for the AutoScalingGroup
     */
    applyUpdatePolicies(props) {
        if (props.updateType === UpdateType.REPLACING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingReplacingUpdate: {
                    willReplace: true,
                },
            };
            if (props.replacingUpdateMinSuccessfulInstancesPercent !== undefined) {
                // Yes, this goes on CreationPolicy, not as a process parameter to ReplacingUpdate.
                // It's a little confusing, but the docs seem to explicitly state it will only be used
                // during the update?
                //
                // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-creationpolicy.html
                this.autoScalingGroup.cfnOptions.creationPolicy = {
                    ...this.autoScalingGroup.cfnOptions.creationPolicy,
                    autoScalingCreationPolicy: {
                        minSuccessfulInstancesPercent: validatePercentage(props.replacingUpdateMinSuccessfulInstancesPercent),
                    },
                };
            }
        }
        else if (props.updateType === UpdateType.ROLLING_UPDATE) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingRollingUpdate: renderRollingUpdateConfig(props.rollingUpdateConfiguration),
            };
        }
        // undefined is treated as 'true'
        if (props.ignoreUnmodifiedSizeProperties !== false) {
            this.autoScalingGroup.cfnOptions.updatePolicy = {
                ...this.autoScalingGroup.cfnOptions.updatePolicy,
                autoScalingScheduledAction: { ignoreUnmodifiedGroupSizeProperties: true },
            };
        }
        if (props.resourceSignalCount !== undefined || props.resourceSignalTimeout !== undefined) {
            this.autoScalingGroup.cfnOptions.creationPolicy = {
                ...this.autoScalingGroup.cfnOptions.creationPolicy,
                resourceSignal: {
                    count: props.resourceSignalCount,
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
    renderNotificationConfiguration() {
        if (this.notifications.length === 0) {
            return undefined;
        }
        return this.notifications.map(notification => ({
            topicArn: notification.topic.topicArn,
            notificationTypes: notification.scalingEvents ? notification.scalingEvents._types : ScalingEvents.ALL._types,
        }));
    }
}
exports.AutoScalingGroup = AutoScalingGroup;
/**
 * The type of update to perform on instances in this AutoScalingGroup
 */
var UpdateType;
(function (UpdateType) {
    /**
     * Don't do anything
     */
    UpdateType["NONE"] = "None";
    /**
     * Replace the entire AutoScalingGroup
     *
     * Builds a new AutoScalingGroup first, then delete the old one.
     */
    UpdateType["REPLACING_UPDATE"] = "Replace";
    /**
     * Replace the instances in the AutoScalingGroup.
     */
    UpdateType["ROLLING_UPDATE"] = "RollingUpdate";
})(UpdateType = exports.UpdateType || (exports.UpdateType = {}));
/**
 * Fleet scaling events
 */
var ScalingEvent;
(function (ScalingEvent) {
    /**
     * Notify when an instance was launced
     */
    ScalingEvent["INSTANCE_LAUNCH"] = "autoscaling:EC2_INSTANCE_LAUNCH";
    /**
     * Notify when an instance was terminated
     */
    ScalingEvent["INSTANCE_TERMINATE"] = "autoscaling:EC2_INSTANCE_TERMINATE";
    /**
     * Notify when an instance failed to terminate
     */
    ScalingEvent["INSTANCE_TERMINATE_ERROR"] = "autoscaling:EC2_INSTANCE_TERMINATE_ERROR";
    /**
     * Notify when an instance failed to launch
     */
    ScalingEvent["INSTANCE_LAUNCH_ERROR"] = "autoscaling:EC2_INSTANCE_LAUNCH_ERROR";
    /**
     * Send a test notification to the topic
     */
    ScalingEvent["TEST_NOTIFICATION"] = "autoscaling:TEST_NOTIFICATION";
})(ScalingEvent = exports.ScalingEvent || (exports.ScalingEvent = {}));
/**
 * A list of ScalingEvents, you can use one of the predefined lists, such as ScalingEvents.ERRORS
 * or create a custome group by instantiating a `NotificationTypes` object, e.g: `new NotificationTypes(`NotificationType.INSTANCE_LAUNCH`)`.
 */
class ScalingEvents {
    constructor(...types) {
        this._types = types;
    }
}
exports.ScalingEvents = ScalingEvents;
/**
 * Fleet scaling errors
 */
ScalingEvents.ERRORS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * All fleet scaling events
 */
ScalingEvents.ALL = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR, ScalingEvent.INSTANCE_TERMINATE, ScalingEvent.INSTANCE_TERMINATE_ERROR);
/**
 * Fleet scaling launch events
 */
ScalingEvents.LAUNCH_EVENTS = new ScalingEvents(ScalingEvent.INSTANCE_LAUNCH, ScalingEvent.INSTANCE_LAUNCH_ERROR);
var ScalingProcess;
(function (ScalingProcess) {
    ScalingProcess["LAUNCH"] = "Launch";
    ScalingProcess["TERMINATE"] = "Terminate";
    ScalingProcess["HEALTH_CHECK"] = "HealthCheck";
    ScalingProcess["REPLACE_UNHEALTHY"] = "ReplaceUnhealthy";
    ScalingProcess["AZ_REBALANCE"] = "AZRebalance";
    ScalingProcess["ALARM_NOTIFICATION"] = "AlarmNotification";
    ScalingProcess["SCHEDULED_ACTIONS"] = "ScheduledActions";
    ScalingProcess["ADD_TO_LOAD_BALANCER"] = "AddToLoadBalancer";
})(ScalingProcess = exports.ScalingProcess || (exports.ScalingProcess = {}));
/**
 * Health check settings
 */
class HealthCheck {
    constructor(type, gracePeriod) {
        this.type = type;
        this.gracePeriod = gracePeriod;
    }
    /**
     * Use EC2 for health checks
     *
     * @param options EC2 health check options
     */
    static ec2(options = {}) {
        return new HealthCheck(HealthCheckType.EC2, options.grace);
    }
    /**
     * Use ELB for health checks.
     * It considers the instance unhealthy if it fails either the EC2 status checks or the load balancer health checks.
     *
     * @param options ELB health check options
     */
    static elb(options) {
        return new HealthCheck(HealthCheckType.ELB, options.grace);
    }
}
exports.HealthCheck = HealthCheck;
var HealthCheckType;
(function (HealthCheckType) {
    HealthCheckType["EC2"] = "EC2";
    HealthCheckType["ELB"] = "ELB";
})(HealthCheckType || (HealthCheckType = {}));
/**
 * Render the rolling update configuration into the appropriate object
 */
function renderRollingUpdateConfig(config = {}) {
    const waitOnResourceSignals = config.minSuccessfulInstancesPercent !== undefined ? true : false;
    const pauseTime = config.pauseTime || (waitOnResourceSignals ? core_1.Duration.minutes(5) : core_1.Duration.seconds(0));
    return {
        maxBatchSize: config.maxBatchSize,
        minInstancesInService: config.minInstancesInService,
        minSuccessfulInstancesPercent: validatePercentage(config.minSuccessfulInstancesPercent),
        waitOnResourceSignals,
        pauseTime: pauseTime && pauseTime.toISOString(),
        suspendProcesses: config.suspendProcesses !== undefined ? config.suspendProcesses :
            // Recommended list of processes to suspend from here:
            // https://aws.amazon.com/premiumsupport/knowledge-center/auto-scaling-group-rolling-updates/
            [ScalingProcess.HEALTH_CHECK, ScalingProcess.REPLACE_UNHEALTHY, ScalingProcess.AZ_REBALANCE,
                ScalingProcess.ALARM_NOTIFICATION, ScalingProcess.SCHEDULED_ACTIONS],
    };
}
function validatePercentage(x) {
    if (x === undefined || (0 <= x && x <= 100)) {
        return x;
    }
    throw new Error(`Expected: a percentage 0..100, got: ${x}`);
}
/**
 * Synthesize an array of block device mappings from a list of block device
 *
 * @param construct the instance/asg construct, used to host any warning
 * @param blockDevices list of block devices
 */
function synthesizeBlockDeviceMappings(construct, blockDevices) {
    return blockDevices.map(({ deviceName, volume, mappingEnabled }) => {
        const { virtualName, ebsDevice: ebs } = volume;
        if (volume === volume_1.BlockDeviceVolume._NO_DEVICE || mappingEnabled === false) {
            return {
                deviceName,
                noDevice: true,
            };
        }
        if (ebs) {
            const { iops, volumeType } = ebs;
            if (!iops) {
                if (volumeType === volume_1.EbsDeviceVolumeType.IO1) {
                    throw new Error('iops property is required with volumeType: EbsDeviceVolumeType.IO1');
                }
            }
            else if (volumeType !== volume_1.EbsDeviceVolumeType.IO1) {
                construct.node.addWarning('iops will be ignored without volumeType: EbsDeviceVolumeType.IO1');
            }
        }
        return {
            deviceName, ebs, virtualName,
        };
    });
}
//# sourceMappingURL=data:application/json;base64,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