"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthorizationType = exports.Method = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const apigateway_generated_1 = require("./apigateway.generated");
const authorizer_1 = require("./authorizer");
const integration_1 = require("./integration");
const mock_1 = require("./integrations/mock");
const restapi_1 = require("./restapi");
const util_1 = require("./util");
class Method extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.resource = props.resource;
        this.api = props.resource.api;
        this.httpMethod = props.httpMethod.toUpperCase();
        util_1.validateHttpMethod(this.httpMethod);
        const options = props.options || {};
        const defaultMethodOptions = props.resource.defaultMethodOptions || {};
        const authorizer = options.authorizer || defaultMethodOptions.authorizer;
        const authorizerId = authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizerId;
        const authorizationTypeOption = options.authorizationType || defaultMethodOptions.authorizationType;
        const authorizationType = (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) || authorizationTypeOption || AuthorizationType.NONE;
        // if the authorizer defines an authorization type and we also have an explicit option set, check that they are the same
        if ((authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) && authorizationTypeOption && (authorizer === null || authorizer === void 0 ? void 0 : authorizer.authorizationType) !== authorizationTypeOption) {
            throw new Error(`${this.resource}/${this.httpMethod} - Authorization type is set to ${authorizationTypeOption} ` +
                `which is different from what is required by the authorizer [${authorizer.authorizationType}]`);
        }
        if (authorizer_1.Authorizer.isAuthorizer(authorizer)) {
            authorizer._attachToApi(this.api);
        }
        const integration = (_b = (_a = props.integration) !== null && _a !== void 0 ? _a : this.resource.defaultIntegration) !== null && _b !== void 0 ? _b : new mock_1.MockIntegration();
        const bindResult = integration.bind(this);
        const methodProps = {
            resourceId: props.resource.resourceId,
            restApiId: this.api.restApiId,
            httpMethod: this.httpMethod,
            operationName: options.operationName || defaultMethodOptions.operationName,
            apiKeyRequired: options.apiKeyRequired || defaultMethodOptions.apiKeyRequired,
            authorizationType,
            authorizerId,
            requestParameters: options.requestParameters || defaultMethodOptions.requestParameters,
            integration: this.renderIntegration(integration),
            methodResponses: this.renderMethodResponses(options.methodResponses),
            requestModels: this.renderRequestModels(options.requestModels),
            requestValidatorId: this.requestValidatorId(options),
            authorizationScopes: (_c = options.authorizationScopes) !== null && _c !== void 0 ? _c : defaultMethodOptions.authorizationScopes,
        };
        const resource = new apigateway_generated_1.CfnMethod(this, 'Resource', methodProps);
        this.methodId = resource.ref;
        if (restapi_1.RestApiBase._isRestApiBase(props.resource.api)) {
            props.resource.api._attachMethod(this);
        }
        const deployment = props.resource.api.latestDeployment;
        if (deployment) {
            deployment.node.addDependency(resource);
            deployment.addToLogicalId({
                method: {
                    ...methodProps,
                    integrationToken: bindResult === null || bindResult === void 0 ? void 0 : bindResult.deploymentToken,
                },
            });
        }
    }
    /**
     * The RestApi associated with this Method
     * @deprecated - Throws an error if this Resource is not associated with an instance of `RestApi`. Use `api` instead.
     */
    get restApi() {
        return this.resource.restApi;
    }
    /**
     * Returns an execute-api ARN for this method:
     *
     *   arn:aws:execute-api:{region}:{account}:{restApiId}/{stage}/{method}/{path}
     *
     * NOTE: {stage} will refer to the `restApi.deploymentStage`, which will
     * automatically set if auto-deploy is enabled, or can be explicitly assigned.
     * When not configured, {stage} will be set to '*', as a shorthand for 'all stages'.
     *
     * @attribute
     */
    get methodArn() {
        var _a;
        const stage = (_a = this.api.deploymentStage) === null || _a === void 0 ? void 0 : _a.stageName;
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), stage);
    }
    /**
     * Returns an execute-api ARN for this method's "test-invoke-stage" stage.
     * This stage is used by the AWS Console UI when testing the method.
     */
    get testMethodArn() {
        return this.api.arnForExecuteApi(this.httpMethod, pathForArn(this.resource.path), 'test-invoke-stage');
    }
    renderIntegration(integration) {
        const options = integration._props.options || {};
        let credentials;
        if (options.credentialsPassthrough !== undefined && options.credentialsRole !== undefined) {
            throw new Error('\'credentialsPassthrough\' and \'credentialsRole\' are mutually exclusive');
        }
        if (options.connectionType === integration_1.ConnectionType.VPC_LINK && options.vpcLink === undefined) {
            throw new Error('\'connectionType\' of VPC_LINK requires \'vpcLink\' prop to be set');
        }
        if (options.connectionType === integration_1.ConnectionType.INTERNET && options.vpcLink !== undefined) {
            throw new Error('cannot set \'vpcLink\' where \'connectionType\' is INTERNET');
        }
        if (options.credentialsRole) {
            credentials = options.credentialsRole.roleArn;
        }
        else if (options.credentialsPassthrough) {
            // arn:aws:iam::*:user/*
            // tslint:disable-next-line:max-line-length
            credentials = core_1.Stack.of(this).formatArn({ service: 'iam', region: '', account: '*', resource: 'user', sep: '/', resourceName: '*' });
        }
        return {
            type: integration._props.type,
            uri: integration._props.uri,
            cacheKeyParameters: options.cacheKeyParameters,
            cacheNamespace: options.cacheNamespace,
            contentHandling: options.contentHandling,
            integrationHttpMethod: integration._props.integrationHttpMethod,
            requestParameters: options.requestParameters,
            requestTemplates: options.requestTemplates,
            passthroughBehavior: options.passthroughBehavior,
            integrationResponses: options.integrationResponses,
            connectionType: options.connectionType,
            connectionId: options.vpcLink ? options.vpcLink.vpcLinkId : undefined,
            credentials,
        };
    }
    renderMethodResponses(methodResponses) {
        if (!methodResponses) {
            // Fall back to nothing
            return undefined;
        }
        return methodResponses.map(mr => {
            let responseModels;
            if (mr.responseModels) {
                responseModels = {};
                for (const contentType in mr.responseModels) {
                    if (mr.responseModels.hasOwnProperty(contentType)) {
                        responseModels[contentType] = mr.responseModels[contentType].modelId;
                    }
                }
            }
            const methodResponseProp = {
                statusCode: mr.statusCode,
                responseParameters: mr.responseParameters,
                responseModels,
            };
            return methodResponseProp;
        });
    }
    renderRequestModels(requestModels) {
        if (!requestModels) {
            // Fall back to nothing
            return undefined;
        }
        const models = {};
        for (const contentType in requestModels) {
            if (requestModels.hasOwnProperty(contentType)) {
                models[contentType] = requestModels[contentType].modelId;
            }
        }
        return models;
    }
    requestValidatorId(options) {
        var _a;
        if (options.requestValidator && options.requestValidatorOptions) {
            throw new Error('Only one of \'requestValidator\' or \'requestValidatorOptions\' must be specified.');
        }
        if (options.requestValidatorOptions) {
            const validator = this.restApi.addRequestValidator('validator', options.requestValidatorOptions);
            return validator.requestValidatorId;
        }
        // For backward compatibility
        return (_a = options.requestValidator) === null || _a === void 0 ? void 0 : _a.requestValidatorId;
    }
}
exports.Method = Method;
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * Open access.
     */
    AuthorizationType["NONE"] = "NONE";
    /**
     * Use AWS IAM permissions.
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Use a custom authorizer.
     */
    AuthorizationType["CUSTOM"] = "CUSTOM";
    /**
     * Use an AWS Cognito user pool.
     */
    AuthorizationType["COGNITO"] = "COGNITO_USER_POOLS";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
function pathForArn(path) {
    return path.replace(/\{[^\}]*\}/g, '*'); // replace path parameters (like '{bookId}') with asterisk
}
//# sourceMappingURL=data:application/json;base64,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