"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaIntegration = void 0;
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const aws_1 = require("./aws");
/**
 * Integrates an AWS Lambda function to an API Gateway method.
 *
 * @example
 *
 *    const handler = new lambda.Function(this, 'MyFunction', ...);
 *    api.addMethod('GET', new LambdaIntegration(handler));
 *
 */
class LambdaIntegration extends aws_1.AwsIntegration {
    constructor(handler, options = {}) {
        const proxy = options.proxy === undefined ? true : options.proxy;
        super({
            proxy,
            service: 'lambda',
            path: `2015-03-31/functions/${handler.functionArn}/invocations`,
            options,
        });
        this.handler = handler;
        this.enableTest = options.allowTestInvoke === undefined ? true : false;
    }
    bind(method) {
        super.bind(method);
        const principal = new iam.ServicePrincipal('apigateway.amazonaws.com');
        const desc = `${method.api.node.uniqueId}.${method.httpMethod}.${method.resource.path.replace(/\//g, '.')}`;
        this.handler.addPermission(`ApiPermission.${desc}`, {
            principal,
            scope: method,
            sourceArn: core_1.Lazy.stringValue({ produce: () => method.methodArn }),
        });
        // add permission to invoke from the console
        if (this.enableTest) {
            this.handler.addPermission(`ApiPermission.Test.${desc}`, {
                principal,
                scope: method,
                sourceArn: method.testMethodArn,
            });
        }
        let functionName;
        if (this.handler instanceof lambda.Function) {
            // if not imported, extract the name from the CFN layer to reach
            // the literal value if it is given (rather than a token)
            functionName = this.handler.node.defaultChild.functionName;
        }
        else {
            // imported, just take the function name.
            functionName = this.handler.functionName;
        }
        let deploymentToken;
        if (!core_1.Token.isUnresolved(functionName)) {
            deploymentToken = JSON.stringify({ functionName });
        }
        return {
            deploymentToken,
        };
    }
}
exports.LambdaIntegration = LambdaIntegration;
//# sourceMappingURL=data:application/json;base64,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