#!/usr/bin/env python3
"""
Script to delete Identifier files, when using WSL2 with Docker on Windows.
"""

import os
import sys
from pathlib import Path

def find_and_delete_identifiers(root_dir, dry_run=False):
    """
    Search for and delete all files ending in .Identifier.
    
    Args:
        root_dir: Root directory to search.
        dry_run: If True, only shows which files would be deleted without actually deleting them.
    """
    root_path = Path(root_dir).resolve()
    
    if not root_path.exists():
        print(f"❌ Error: The directory '{root_dir}' does not exist")
        return
    
    if not root_path.is_dir():
        print(f"❌ Error: '{root_dir}' is not a directory")
        return

    print(f"🔍 Searching for .Identifier files in: {root_path}")
    if dry_run:
        print("⚠️  DRY RUN MODE - No files will be deleted\n")
    else:
        print()
    
    deleted_count = 0
    errors = []

    # Recursively search for all .Identifier files
    for identifier_file in root_path.rglob('*.Identifier'):
        try:
            relative_path = identifier_file.relative_to(root_path)
            
            if dry_run:
                print(f"  🗑️  Would delete: {relative_path}")
                deleted_count += 1
            else:
                identifier_file.unlink()
                print(f"  ✅ Deleted: {relative_path}")
                deleted_count += 1
                
        except Exception as e:
            error_msg = f"Error processing {identifier_file}: {e}"
            errors.append(error_msg)
            print(f"  ❌ {error_msg}")

    # Summary
    print(f"\n{'='*60}")
    if dry_run:
        print(f"📊 Found {deleted_count} .Identifier files")
        print("   Run without --dry-run to delete them")
    else:
        print(f"✨ Total deleted: {deleted_count} files")

    if errors:
        print(f"⚠️  Errors found: {len(errors)}")
    
    print('='*60)

def main():
    import argparse
    
    parser = argparse.ArgumentParser(
        description='Delete .Identifier files generated by WSL2',
        epilog='Example: python script.py /path/to/directory'
    )
    parser.add_argument(
        'directory',
        nargs='?',
        default='.',
        help='Directory to search (default: current directory)'
    )
    parser.add_argument(
        '--dry-run',
        action='store_true',
        help='Simulate deletion without removing files'
    )
    
    args = parser.parse_args()
    
    try:
        find_and_delete_identifiers(args.directory, args.dry_run)
    except KeyboardInterrupt:
        print("\n\n⚠️  Operation canceled by user")
        sys.exit(1)

if __name__ == "__main__":
    main()