"use strict";
/*
 * Unless explicitly stated otherwise all files in this repository are licensed
 * under the Apache License Version 2.0.
 *
 * This product includes software developed at Datadog (https://www.datadoghq.com/).
 * Copyright 2021 Datadog, Inc.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.generateExtensionLayerId = exports.generateLambdaLayerId = exports.getMissingLayerVersionErrorMsg = exports.getExtensionLayerArn = exports.getLambdaLayerArn = exports.applyLayers = exports.runtimeLookup = exports.RuntimeType = exports.DD_GOV_ACCOUNT_ID = exports.DD_ACCOUNT_ID = void 0;
const crypto = require("crypto");
const lambda = require("@aws-cdk/aws-lambda");
const loglevel_1 = require("loglevel");
exports.DD_ACCOUNT_ID = "464622532012";
exports.DD_GOV_ACCOUNT_ID = "002406178527";
/**
 * @stability stable
 */
var RuntimeType;
(function (RuntimeType) {
    RuntimeType[RuntimeType["NODE"] = 0] = "NODE";
    RuntimeType[RuntimeType["PYTHON"] = 1] = "PYTHON";
    RuntimeType[RuntimeType["UNSUPPORTED"] = 2] = "UNSUPPORTED";
})(RuntimeType = exports.RuntimeType || (exports.RuntimeType = {}));
const layerPrefix = "DatadogLayer";
const extensionLayerPrefix = "DatadogExtension";
exports.runtimeLookup = {
    "nodejs10.x": RuntimeType.NODE,
    "nodejs12.x": RuntimeType.NODE,
    "nodejs8.10": RuntimeType.NODE,
    "nodejs14.x": RuntimeType.NODE,
    "python2.7": RuntimeType.PYTHON,
    "python3.6": RuntimeType.PYTHON,
    "python3.7": RuntimeType.PYTHON,
    "python3.8": RuntimeType.PYTHON,
};
const runtimeToLayerName = {
    "nodejs8.10": "Datadog-Node8-10",
    "nodejs10.x": "Datadog-Node10-x",
    "nodejs12.x": "Datadog-Node12-x",
    "nodejs14.x": "Datadog-Node14-x",
    "python2.7": "Datadog-Python27",
    "python3.6": "Datadog-Python36",
    "python3.7": "Datadog-Python37",
    "python3.8": "Datadog-Python38",
};
const layers = new Map();
function applyLayers(scope, region, lambdas, pythonLayerVersion, nodeLayerVersion, extensionLayerVersion) {
    // TODO: check region availability
    const errors = [];
    loglevel_1.default.debug("Applying layers to Lambda functions...");
    lambdas.forEach((lam) => {
        const runtime = lam.runtime.name;
        const lambdaRuntimeType = exports.runtimeLookup[runtime];
        if (lambdaRuntimeType === RuntimeType.UNSUPPORTED) {
            loglevel_1.default.debug(`Unsupported runtime: ${runtime}`);
            return;
        }
        let lambdaLayerArn;
        let extensionLayerArn;
        if (lambdaRuntimeType === RuntimeType.PYTHON) {
            if (pythonLayerVersion === undefined) {
                errors.push(getMissingLayerVersionErrorMsg(lam.node.id, "Python", "python"));
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, pythonLayerVersion, runtime);
            loglevel_1.default.debug(`Using Python Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (lambdaRuntimeType === RuntimeType.NODE) {
            if (nodeLayerVersion === undefined) {
                errors.push(getMissingLayerVersionErrorMsg(lam.node.id, "Node.js", "node"));
                return;
            }
            lambdaLayerArn = getLambdaLayerArn(region, nodeLayerVersion, runtime);
            loglevel_1.default.debug(`Using Node Lambda layer: ${lambdaLayerArn}`);
            addLayer(lambdaLayerArn, false, scope, lam, runtime);
        }
        if (extensionLayerVersion !== undefined) {
            extensionLayerArn = getExtensionLayerArn(region, extensionLayerVersion);
            loglevel_1.default.debug(`Using extension layer: ${extensionLayerArn}`);
            addLayer(extensionLayerArn, true, scope, lam, runtime);
        }
    });
    return errors;
}
exports.applyLayers = applyLayers;
function addLayer(layerArn, isExtensionLayer, scope, lam, runtime) {
    let layerId;
    if (isExtensionLayer) {
        layerId = generateExtensionLayerId(lam.functionArn);
    }
    else {
        layerId = generateLambdaLayerId(lam.functionArn, runtime);
    }
    if (layerArn !== undefined) {
        let lambdaLayer = layers.get(layerArn);
        if (lambdaLayer === undefined) {
            lambdaLayer = lambda.LayerVersion.fromLayerVersionArn(scope, layerId, layerArn);
            layers.set(layerArn, lambdaLayer);
        }
        loglevel_1.default.debug(`Adding layer ${lambdaLayer} to Lambda: ${lam.functionName}`);
        lam.addLayers(lambdaLayer);
    }
}
function getLambdaLayerArn(region, version, runtime) {
    const layerName = runtimeToLayerName[runtime];
    // TODO: edge case where gov cloud is the region, but they are using a token so we can't resolve it.
    const isGovCloud = region === "us-gov-east-1" || region === "us-gov-west-1";
    // if this is a GovCloud region, use the GovCloud lambda layer
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud lambda layer");
        return `arn:aws-us-gov:lambda:${region}:${exports.DD_GOV_ACCOUNT_ID}:layer:${layerName}:${version}`;
    }
    return `arn:aws:lambda:${region}:${exports.DD_ACCOUNT_ID}:layer:${layerName}:${version}`;
}
exports.getLambdaLayerArn = getLambdaLayerArn;
function getExtensionLayerArn(region, version) {
    const isGovCloud = region === "us-gov-east-1" || region === "us-gov-west-1";
    if (isGovCloud) {
        loglevel_1.default.debug("GovCloud region detected, using the GovCloud extension layer");
        return `arn:aws-us-gov:lambda:${region}:${exports.DD_GOV_ACCOUNT_ID}:layer:Datadog-Extension:${version}`;
    }
    return `arn:aws:lambda:${region}:${exports.DD_ACCOUNT_ID}:layer:Datadog-Extension:${version}`;
}
exports.getExtensionLayerArn = getExtensionLayerArn;
function getMissingLayerVersionErrorMsg(functionKey, formalRuntime, paramRuntime) {
    return (`Resource ${functionKey} has a ${formalRuntime} runtime, but no ${formalRuntime} Lambda Library version was provided. ` +
        `Please add the '${paramRuntime}LayerVersion' parameter for the Datadog serverless macro.`);
}
exports.getMissingLayerVersionErrorMsg = getMissingLayerVersionErrorMsg;
function generateLambdaLayerId(lambdaFunctionArn, runtime) {
    loglevel_1.default.debug("Generating construct Id for Datadog Lambda layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return layerPrefix + "-" + runtime + "-" + layerValue;
}
exports.generateLambdaLayerId = generateLambdaLayerId;
function generateExtensionLayerId(lambdaFunctionArn) {
    loglevel_1.default.debug("Generating construct Id for Datadog Extension layer");
    const layerValue = crypto.createHash("sha256").update(lambdaFunctionArn).digest("hex");
    return extensionLayerPrefix + "-" + layerValue;
}
exports.generateExtensionLayerId = generateExtensionLayerId;
//# sourceMappingURL=data:application/json;base64,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