"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NewRelicStack = exports.EndpointUrlMetrics = exports.EndpointUrlLogs = exports.EndpointType = exports.NEW_RELIC_AWS_ACCOUNT_ID = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const iam = require("aws-cdk-lib/aws-iam");
const firehose = require("aws-cdk-lib/aws-kinesisfirehose");
const s3 = require("aws-cdk-lib/aws-s3");
const secretsmanager = require("aws-cdk-lib/aws-secretsmanager");
const utils_1 = require("../../common/utils");
const base_1 = require("../base");
exports.NEW_RELIC_AWS_ACCOUNT_ID = '754728514883';
var EndpointType;
(function (EndpointType) {
    EndpointType["METRICS"] = "metrics";
    EndpointType["LOGS"] = "logs";
})(EndpointType = exports.EndpointType || (exports.EndpointType = {}));
var EndpointUrlLogs;
(function (EndpointUrlLogs) {
    EndpointUrlLogs["EU_LOGS"] = "https://aws-api.eu.newrelic.com/firehose/v1";
    EndpointUrlLogs["US_LOGS"] = "https://aws-api.newrelic.com/firehose/v1";
})(EndpointUrlLogs = exports.EndpointUrlLogs || (exports.EndpointUrlLogs = {}));
var EndpointUrlMetrics;
(function (EndpointUrlMetrics) {
    EndpointUrlMetrics["EU_METRICS"] = "https://aws-api.eu01.nr-data.net/cloudwatch-metrics/v1";
    EndpointUrlMetrics["US_METRICS"] = "https://aws-api.newrelic.com/cloudwatch-metrics/v1";
})(EndpointUrlMetrics = exports.EndpointUrlMetrics || (exports.EndpointUrlMetrics = {}));
class NewRelicStack extends base_1.BaseStack {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.newRelicIntegrationRole = this.createNewRelicRole(props.newRelicAccountId);
        this.newRelicSecret = this.createSecrets(props.newRelicAccountId, props.newRelicLicenseKey);
        this.newRelicBucket = this.createFirehoseBucket(props.newRelicBucketName);
        this.newRelicFirehoseRole = this.createFirehoseRole(this.newRelicBucket);
        if (props.newRelicApiUrlLogs) {
            this.newRelicFirehoseLogs = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.LOGS, props.newRelicApiUrlLogs, props.newRelicLicenseKey);
        }
        if (props.newRelicApiUrlMetrics) {
            this.newRelicFirehoseMetrics = this.createFirehoseStream(this.newRelicBucket, this.newRelicFirehoseRole, EndpointType.METRICS, props.newRelicApiUrlMetrics, props.newRelicLicenseKey);
        }
    }
    createNewRelicRole(newRelicAccountId) {
        let role = new iam.Role(this, 'newrelic-role', {
            roleName: 'NewRelicInfrastructure-Integrations',
            assumedBy: new iam.AccountPrincipal(exports.NEW_RELIC_AWS_ACCOUNT_ID),
            externalIds: [
                newRelicAccountId,
            ],
        });
        role.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('ReadOnlyAccess'));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'budgets:ViewBudget',
            ],
            resources: ['*'],
        }));
        utils_1.addBaseTags(role);
        new aws_cdk_lib_1.CfnOutput(this, 'newrelic-role-output', {
            value: role.roleArn,
            description: 'New Relic role arn',
            exportName: 'newRelicRole',
        });
        return role;
    }
    createFirehoseStream(newRelicBucket, role, endpointType, endpointUrl, newRelicLicenseLey) {
        if (this.stage == 'production') {
            // Minute in one day: 1440
            // Interval: 5min
            // Sends per day: 1440/5 = 288
            // Usage per day: 288*5mb = 1.5gb
            var bufferingHints = {
                intervalInSeconds: 300,
                sizeInMBs: 5,
            };
        }
        else {
            // Usage per day: 144*3mb = 0.432gb
            var bufferingHints = {
                intervalInSeconds: 600,
                sizeInMBs: 3,
            };
        }
        let httpEndpointMetrics = {
            bufferingHints: bufferingHints,
            endpointConfiguration: {
                url: endpointUrl,
                accessKey: newRelicLicenseLey,
                name: endpointType,
            },
            s3Configuration: {
                bucketArn: newRelicBucket.bucketArn,
                roleArn: role.roleArn,
            },
            requestConfiguration: {
                contentEncoding: 'GZIP',
            },
            roleArn: role.roleArn,
        };
        let firehoseStream = new firehose.CfnDeliveryStream(this, `newrelic-firehose-${endpointType}`, {
            deliveryStreamName: `NewRelic-stream-${endpointType}`,
            deliveryStreamType: 'DirectPut',
            httpEndpointDestinationConfiguration: httpEndpointMetrics,
        });
        utils_1.addBaseTags(firehoseStream);
        return firehoseStream;
    }
    createSecrets(newRelicAccountId, newRelicLicenseLey) {
        let secret = new secretsmanager.Secret(this, 'newrelic-secret', {
            secretName: 'NEW_RELIC_LICENSE_KEY',
            secretObjectValue: {
                NrAccountId: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicAccountId),
                LicenseKey: aws_cdk_lib_1.SecretValue.unsafePlainText(newRelicLicenseLey),
            },
        });
        utils_1.addBaseTags(secret);
        return secret;
    }
    createFirehoseBucket(newRelicBucketName) {
        let bucket = new s3.Bucket(this, 'newrelic-bucket', {
            bucketName: newRelicBucketName,
            versioned: true,
            lifecycleRules: [
                {
                    id: 'raw-bucket-lifecycle-rule',
                    enabled: true,
                    noncurrentVersionExpiration: aws_cdk_lib_1.Duration.days(30),
                    transitions: [
                        {
                            storageClass: s3.StorageClass.INFREQUENT_ACCESS,
                            transitionAfter: aws_cdk_lib_1.Duration.days(30),
                        },
                    ],
                },
            ],
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
            autoDeleteObjects: true,
        });
        utils_1.addBaseTags(bucket);
        return bucket;
    }
    createFirehoseRole(newRelicFirehoseBucket) {
        let role = new iam.Role(this, 'newrelic-firehose-role', {
            assumedBy: new iam.ServicePrincipal('firehose.amazonaws.com'),
        });
        utils_1.addBaseTags(role);
        // TODO: create more restrictive policy
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                'glue:GetTable',
                'glue:GetTableVersion',
                'glue:GetTableVersions',
                'glue:GetSchemaByDefinition',
                'glue:GetSchemaVersion',
                'logs:PutLogEvents',
                'kinesis:DescribeStream',
                'kinesis:GetShardIterator',
                'kinesis:GetRecords',
                'kinesis:ListShards',
                'kms:Decrypt',
                'kms:GenerateDataKey',
                'kms:Decrypt',
                'kafka:GetBootstrapBrokers',
                'kafka:DescribeCluster',
                'kafka:DescribeClusterV2',
                'kafka-cluster:Connect',
                'kafka-cluster:DescribeTopic',
                'kafka-cluster:DescribeTopicDynamicConfiguration',
                'kafka-cluster:ReadData',
                'kafka-cluster:DescribeGroup',
                'lambda:InvokeFunction',
                'lambda:GetFunctionConfiguration',
            ],
            resources: ['*'],
        }));
        role.addToPolicy(new iam.PolicyStatement({
            actions: [
                's3:AbortMultipartUpload',
                's3:GetBucketLocation',
                's3:GetObject',
                's3:ListBucket',
                's3:ListBucketMultipartUploads',
                's3:PutObject',
                'S3:PutObjectAcl',
            ],
            resources: [
                `${newRelicFirehoseBucket.bucketArn}`,
                `${newRelicFirehoseBucket.bucketArn}/*`,
            ],
        }));
        return role;
    }
}
exports.NewRelicStack = NewRelicStack;
_a = JSII_RTTI_SYMBOL_1;
NewRelicStack[_a] = { fqn: "neulabs-cdk-constructs.stacks.newrelic.NewRelicStack", version: "0.1.0" };
//# sourceMappingURL=data:application/json;base64,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