# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['git_vuln_finder']

package_data = \
{'': ['*'], 'git_vuln_finder': ['patterns/en/medium/*']}

install_requires = \
['gitpython>=3.0.5,<4.0.0',
 'langdetect>=1.0.7,<2.0.0',
 'pymisp>=2.4.128,<3.0.0']

entry_points = \
{'console_scripts': ['git-vuln-finder = bin.finder:main']}

setup_kwargs = {
    'name': 'git-vuln-finder',
    'version': '1.4',
    'description': 'Finding potential software vulnerabilities from git commit messages.',
    'long_description': '# git-vuln-finder\n\n![git-vuln-finder logo](https://raw.githubusercontent.com/cve-search/git-vuln-finder/f22077452c37e110bff0564e1f7b34637dc726c3/doc/logos/git-vuln-finder-small.png)\n\n[![Workflow](https://github.com/cve-search/git-vuln-finder/workflows/Python%20application/badge.svg)](https://github.com/cve-search/git-vuln-finder/actions?query=workflow%3A%22Python+application%22)\n\nFinding potential software vulnerabilities from git commit messages.\nThe output format is a JSON with the associated commit which could contain a\nfix regarding a software vulnerability. The search is based on a set of regular\nexpressions against the commit messages only. If CVE IDs are present, those are\nadded automatically in the output. The input can be any git repositories or\na [GH archive source](https://www.gharchive.org/).\n\n# Requirements\n\n- jq (``sudo apt install jq``)\n\n\n# Installation\n\n## Use it as a library\n\ngit-vuln-finder can be install with poetry. If you don\'t have poetry installed, you can do the following `curl -sSL https://raw.githubusercontent.com/python-poetry/poetry/master/get-poetry.py | python`.\n\n~~~bash\n$ poetry install\n$ poetry shell\n$ git-vuln-finder -h\n~~~\n\nYou can also use ``pip``. Then just import it:\n\n~~~python\nPython 3.8.0 (default, Dec 11 2019, 21:43:13)\n[GCC 9.2.1 20191008] on linux\nType "help", "copyright", "credits" or "license" for more information.\n>>> from git_vuln_finder import find\n>>> all_potential_vulnerabilities, all_cve_found, found = find("~/git/curl")\n\n>>> [commit for commit, summary in all_potential_vulnerabilities.items() if summary[\'state\'] == \'cve-assigned\']\n[\'9069838b30fb3b48af0123e39f664cea683254a5\', \'facb0e4662415b5f28163e853dc6742ac5fafb3d\',\n... snap ...\n \'8a75dbeb2305297640453029b7905ef51b87e8dd\', \'1dc43de0dccc2ea7da6dddb7b98f8d7dcf323914\', \'192c4f788d48f82c03e9cef40013f34370e90737\', \'2eb8dcf26cb37f09cffe26909a646e702dbcab66\', \'fa1ae0abcde5df8d0b3283299e3f246bedf7692c\', \'c11c30a8c8d727dcf5634fa0cc6ee0b4b77ddc3d\', \'75ca568fa1c19de4c5358fed246686de8467c238\', \'a20daf90e358c1476a325ea665d533f7a27e3364\', \'042cc1f69ec0878f542667cb684378869f859911\']\n\n >>> print(json.dumps(all_potential_vulnerabilities[\'9069838b30fb3b48af0123e39f664cea683254a5\'], sort_keys=True, indent=4, separators=(",", ": ")))\n {\n     "author": "Daniel Stenberg",\n     "author-email": "daniel@haxx.se",\n     "authored_date": 1567544372,\n     "branches": [\n         "master"\n     ],\n     "commit-id": "9069838b30fb3b48af0123e39f664cea683254a5",\n     "committed_date": 1568009674,\n     "cve": [\n         "CVE-2019-5481",\n         "CVE-2019-5481"\n     ],\n     "language": "en",\n     "message": "security:read_data fix bad realloc()\\n\\n... that could end up a double-free\\n\\nCVE-2019-5481\\nBug: https://curl.haxx.se/docs/CVE-2019-5481.html\\n",\n     "origin": "https://github.com/curl/curl.git",\n     "origin-github-api": "https://api.github.com/repos///github.com/curl/curl/commits/9069838b30fb3b48af0123e39f664cea683254a5",\n     "pattern-matches": [\n         "double-free"\n     ],\n     "pattern-selected": "(?i)(double[-| ]free|buffer overflow|double free|race[-| ]condition)",\n     "state": "cve-assigned",\n     "stats": {\n         "deletions": 4,\n         "files": 1,\n         "insertions": 2,\n         "lines": 6\n     },\n     "summary": "security:read_data fix bad realloc()",\n     "tags": []\n }\n~~~\n\n\n## Use it as a command line tool\n\n~~~bash\n$ git clone https://github.com/cve-search/git-vuln-finder.git\n$ cd https://github.com/cve-search/git-vuln-finder.git\n$ pip install .\n$ git-vuln-finder --help\n~~~\n\nYou can also use pip.\n``pipx`` installs scripts (system wide available) provided by Python packages\ninto separate virtualenvs to shield them from your system and each other.\n\n\n### Usage\n\n~~~bash\nusage: git-vuln-finder [-h] [-v] [-r R] [-o O] [-s S] [-p P] [-c] [-t] [-gh GH]\n\nFinding potential software vulnerabilities from git commit messages.\n\noptional arguments:\n  -h, --help  show this help message and exit\n  -v          increase output verbosity\n  -r R        git repository to analyse\n  -o O        Output format: [json]\n  -s S        State of the commit found\n  -p P        Matching pattern to use: [vulnpatterns, cryptopatterns, cpatterns] - the pattern \'all\' is used to match all the patterns at once.\n  -c          output only a list of the CVE pattern found in commit messages (disable by default)\n  -t          Include tags matching a specific commit\n  -gh GH      special option for gharchive, pass a file containing a PushEvent in JSON format\n\nMore info: https://github.com/cve-search/git-vuln-finder\n~~~\n\n\n# Patterns\n\ngit-vuln-finder comes with 3 default patterns which can be selected to find the potential vulnerabilities described in the commit messages such as:\n\n- [`vulnpatterns`](git_vuln_finder/patterns/en/medium/vuln) is a generic vulnerability pattern especially targeting web application and generic security commit message. Based on an academic paper.\n- [`cryptopatterns`](git_vuln_finder/patterns/en/medium/crypto) is a vulnerability pattern for cryptographic errors mentioned in commit messages.\n- [`cpatterns`](git_vuln_finder/patterns/en/medium/c) is a set of standard vulnerability patterns see for C/C++-like languages.\n\n\n## A sample partial output from Curl git repository\n\n~~~bash\n$ git-vuln-finder -r ~/git/curl | jq .\n...\n "6df916d751e72fc9a1febc07bb59c4ddd886c043": {\n    "message": "loadlibrary: Only load system DLLs from the system directory\\n\\nInspiration provided by: Daniel Stenberg and Ray Satiro\\n\\nBug: https://curl.haxx.se/docs/adv_20160530.html\\n\\nRef: Windows DLL hijacking with curl, CVE-2016-4802\\n",\n    "language": "en",\n    "commit-id": "6df916d751e72fc9a1febc07bb59c4ddd886c043",\n    "summary": "loadlibrary: Only load system DLLs from the system directory",\n    "stats": {\n      "insertions": 180,\n      "deletions": 8,\n      "lines": 188,\n      "files": 7\n    },\n    "author": "Steve Holme",\n    "author-email": "steve_holme@hotmail.com",\n    "authored_date": 1464555460,\n    "committed_date": 1464588867,\n    "branches": [\n      "master"\n    ],\n    "pattern-selected": "(?i)(denial of service |\\bXXE\\b|remote code execution|\\bopen redirect|OSVDB|\\bvuln|\\bCVE\\b |\\bXSS\\b|\\bReDoS\\b|\\bNVD\\b|malicious|x−frame−options|attack|cross site |exploit|malicious|directory traversal |\\bRCE\\b|\\bdos\\b|\\bXSRF \\b|\\bXSS\\b|clickjack|session.fixation|hijack|\\badvisory|\\binsecure |security |\\bcross−origin\\b|unauthori[z|s]ed |infinite loop)",\n    "pattern-matches": [\n      "hijack"\n    ],\n    "origin": "git@github.com:curl/curl.git",\n    "origin-github-api": "https://api.github.com/repos/curl/curl/commits/6df916d751e72fc9a1febc07bb59c4ddd886c043",\n    "tags": [],\n    "cve": [\n      "CVE-2016-4802"\n    ],\n    "state": "cve-assigned"\n  },\n  "c2b3f264cb5210f82bdc84a3b89250a611b68dd3": {\n    "message": "CONNECT_ONLY: don\'t close connection on GSS 401/407 reponses\\n\\nPreviously, connections were closed immediately before the user had a\\nchance to extract the socket when the proxy required Negotiate\\nauthentication.\\n\\nThis regression was brought in with the security fix in commit\\n79b9d5f1a42578f\\n\\nCloses #655\\n",\n    "language": "en",\n    "commit-id": "c2b3f264cb5210f82bdc84a3b89250a611b68dd3",\n    "summary": "CONNECT_ONLY: don\'t close connection on GSS 401/407 reponses",\n    "stats": {\n      "insertions": 4,\n      "deletions": 2,\n      "lines": 6,\n      "files": 1\n    },\n    "author": "Marcel Raad",\n    "author-email": "raad@teamviewer.com",\n    "authored_date": 1455523116,\n    "committed_date": 1461704516,\n    "branches": [\n      "master"\n    ],\n    "pattern-selected": "(?i)(denial of service |\\bXXE\\b|remote code execution|\\bopen redirect|OSVDB|\\bvuln|\\bCVE\\b |\\bXSS\\b|\\bReDoS\\b|\\bNVD\\b|malicious|x−frame−options|attack|cross site |exploit|malicious|directory traversal |\\bRCE\\b|\\bdos\\b|\\bXSRF \\b|\\bXSS\\b|clickjack|session.fixation|hijack|\\badvisory|\\binsecure |security |\\bcross−origin\\b|unauthori[z|s]ed |infinite loop)",\n    "pattern-matches": [\n      "security "\n    ],\n    "origin": "git@github.com:curl/curl.git",\n    "origin-github-api": "https://api.github.com/repos/curl/curl/commits/c2b3f264cb5210f82bdc84a3b89250a611b68dd3",\n    "tags": [],\n    "state": "under-review"\n  },\n...\n~~~\n\n- Extracting CVE id(s) from git messages\n\n~~~json\n  "98d132cf6a879faf0147aa83ea0c07ff326260ed": {\n    "message": "Add a macro for testing assertion in both debug and production builds\\n\\nIf we have an assert then in a debug build we want an abort() to occur.\\nIn a production build we wan\nt the function to return an error.\\n\\nThis introduces a new macro to assist with that. The idea is to replace\\nexisting use of OPENSSL_assert() with this new macro. The problem with\\nOPENSSL\n_assert() is that it aborts() on an assertion failure in both debug\\nand production builds. It should never be a library\'s decision to abort a\\nprocess (we don\'t get to decide when to kill t\nhe life support machine or\\nthe nuclear reactor control system). Additionally if an attacker can\\ncause a reachable assert to be hit then this can be a source of DoS attacks\\ne.g. see CVE-20\n17-3733, CVE-2015-0293, CVE-2011-4577 and CVE-2002-1568.\\n\\nReviewed-by: Tim Hudson <tjh@openssl.org>\\n(Merged from https://github.com/openssl/openssl/pull/3496)",\n    "commit-id": "98d132cf6a879faf0147aa83ea0c07ff326260ed",\n    "summary": "Add a macro for testing assertion in both debug and production builds",\n    "stats": {\n      "insertions": 18,\n      "deletions": 0,\n      "lines": 18,\n      "files": 1\n    },\n    "author": "Matt Caswell",\n    "author-email": "matt@openssl.org",\n    "authored_date": 1495182637,\n    "committed_date": 1495457671,\n    "branches": [\n      "master"\n    ],\n    "pattern-selected": "(?i)(denial of service |\\bXXE\\b|remote code execution|\\bopen redirect|OSVDB|\\bvuln|\\bCVE\\b |\\bXSS\\b|\\bReDoS\\b|\\bNVD\\b|malicious|x−frame−options|attack|cross site |ex\nploit|malicious|directory traversal |\\bRCE\\b|\\bdos\\b|\\bXSRF \\b|\\bXSS\\b|clickjack|session.fixation|hijack|\\badvisory|\\binsecure |security |\\bcross−origin\\b|unauthori[z|s]ed |infinite loop)",\n    "pattern-matches": [\n      "attack"\n    ],\n    "cve": [\n      "CVE-2017-3733",\n      "CVE-2015-0293",\n      "CVE-2011-4577",\n      "CVE-2002-1568"\n    ],\n    "state": "cve-assigned"\n  }\n~~~\n\n\n\n## Usage for the special gharchive option\n\n~~~bash\n$ git-vuln-finder -gh ../tests/gharchive_test.json\n~~~\n\n\n\nthe value for the `gh` parameters need to be a json file, containing an array of each PushEvent you want to test.\n\n~~~json\n[\n   {\n    "id": "19351512310",\n    "type": "PushEvent",\n    "actor": {\n      "id": 32466128,\n      "login": "DavidCruciani",\n      "display_login": "DavidCruciani",\n      "gravatar_id": "",\n      "url": "https://api.github.com/users/DavidCruciani",\n      "avatar_url": "https://avatars.githubusercontent.com/u/32466128?"\n    },\n    "repo": {\n      "id": 424660123,\n      "name": "ail-project/ail-feeder-gharchive",\n      "url": "https://api.github.com/repos/ail-project/ail-feeder-gharchive"\n    },\n    "payload": {\n      "push_id": 8628652926,\n      "size": 1,\n      "distinct_size": 1,\n      "ref": "refs/heads/main",\n      "head": "910ed71a2819546a3f3bcce1ebb9e3984a8c8d86",\n      "before": "40a9ef5dc6b2add5184a0a58401bfe9058faa8df",\n      "commits": [\n        {\n          "sha": "910ed71a2819546a3f3bcce1ebb9e3984a8c8d86",\n          "author": {\n            "email": "da.cruciani@laposte.net",\n            "name": "David Cruciani"\n          },\n          "message": "chg: [feeder] case sensitive",\n          "distinct": true,\n          "url": "https://api.github.com/repos/ail-project/ail-feeder-gharchive/commits/910ed71a2819546a3f3bcce1ebb9e3984a8c8d86"\n        }\n      ]\n    },\n    "public": true,\n    "created_at": "2021-12-15T16:06:43Z",\n    "org": {\n      "id": 62389074,\n      "login": "ail-project",\n      "gravatar_id": "",\n      "url": "https://api.github.com/orgs/ail-project",\n      "avatar_url": "https://avatars.githubusercontent.com/u/62389074?"\n    }\n  }\n]\n~~~\n\n\n\n## Usage for import \n\nIf the goal is to import the module to use it, the method to call is `find_event`\n\n~~~python\nfrom git_vuln_finder import find_event\n\nfor element in event:\n    for i in range(0,len(element["payload"]["commits"])):\n        all_potential_vulnerabilities, all_cve_found, found = find_event(element["payload"]["commits"][i], element)\n~~~\n\n\n\n## Output with gharchive option\n\n~~~json\n{\n    "repo_name": "LeandroFChaves/gerenciador-alunos",\n    "message": "[UI] - Ajustes no css da aplica\\u00e7\\u00e3o\\n\\n- Adicionado padding para a exibi\\u00e7\\u00e3o do conte\\u00fado das p\\u00e1ginas;\\n- Alinhado os bot\\u00f5es de a\\u00e7\\u00f5es dos forms a direita da table",\n    "language": "pt",\n    "commit-id": "73a1c68b520853198eaac199a41d141ee96dc64d",\n    "author": "LeandroFChaves",\n    "author-email": "bbf3d4347c6affed0d9692115680849e2ace4d62@gmail.com",\n    "authored_date": "2021-10-01T03:00:07Z",\n    "branches": "refs/heads/master",\n    "pattern-selected": "(?i)(denial of service|\\\\bXXE\\\\b|remote code execution|\\\\bopen redirect|OSVDB|\\\\bvuln|\\\\bCVE\\\\b|\\\\bXSS\\\\b|\\\\bReDoS\\\\b|\\\\bNVD\\\\b|malicious|x\\u2212frame\\u2212options|attack|cross site|exploit|malicious|directory traversal|\\\\bRCE\\\\b|\\\\bdos\\\\b|\\\\bXSRF \\\\b|\\\\bXSS\\\\b|clickjack|session.fixation|hijack|\\\\badvisory|\\\\binsecure|security|\\\\bcross\\u2212origin\\\\b|unauthori[z|s]ed|infinite loop)",\n    "pattern-matches": [\n        "dos"\n    ],\n    "origin-github-api": "https://api.github.com/repos/LeandroFChaves/gerenciador-alunos/commits/73a1c68b520853198eaac199a41d141ee96dc64d",\n    "state": "under-review"\n}\n~~~\n\n\n\n\n\n# Running the tests\n\n~~~bash\n$ pytest\n~~~\n\n\n# License and author(s)\n\nThis software is free software and licensed under the AGPL version 3.\n\n- Copyright (c) 2019-2021 Alexandre Dulaunoy - https://github.com/adulau/\n- Copyright (c) 2019-2021 All contributors to the project\n\n# Acknowledgment\n\n- Thanks to [Jean-Louis Huynen](https://github.com/gallypette) for the discussions about the crypto vulnerability patterns.\n- Thanks to [Sebastien Tricaud](https://github.com/stricaud) for the discussions regarding native language, commit messages and external patterns.\n- Thanks to [Cedric Bonhomme](https://github.com/cedricbonhomme) to make git-vuln-finder a Python library, add tests and improve the overall installation process.\n- Thanks to [David Cruciani](https://github.com/DavidCruciani) for the support of [gharchive](https://www.gharchive.org/).\n\n\n# Contributing\n\nWe welcome contributions for the software and especially additional vulnerability patterns. Every contributors will be added in the [AUTHORS file](./AUTHORS) and\ncollectively own this open source software. The contributors acknowledge the [Developer Certificate of Origin](https://developercertificate.org/).\n\n\n# References\n\n- [Notes](https://gist.github.com/adulau/dce5a6ca5c65017869bb01dfee576303#file-finding-vuln-git-commit-messages-md)\n- https://csce.ucmss.com/cr/books/2017/LFS/CSREA2017/ICA2077.pdf (mainly using CVE referenced in the commit message) - archive (http://archive.is/xep9o)\n- https://asankhaya.github.io/pdf/automated-identification-of-security-issues-from-commit-messages-and-bug-reports.pdf (2 main regexps)\n',
    'author': 'Alexandre Dulaunoy',
    'author_email': 'a@foo.be',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/cve-search/git-vuln-finder',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
