"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvocationType = exports.LambdaInvoke = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const iam = require("@aws-cdk/aws-iam");
const sfn = require("@aws-cdk/aws-stepfunctions");
const cdk = require("@aws-cdk/core");
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @stability stable
 */
class LambdaInvoke extends sfn.TaskStateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_b = props.integrationPattern) !== null && _b !== void 0 ? _b : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (props.payloadResponseOnly &&
            (props.integrationPattern || props.invocationType || props.clientContext || props.qualifier)) {
            throw new Error("The 'payloadResponseOnly' property cannot be used if 'integrationPattern', 'invocationType', 'clientContext', or 'qualifier' are specified.");
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
        if ((_c = props.retryOnServiceExceptions) !== null && _c !== void 0 ? _c : true) {
            // Best practice from https://docs.aws.amazon.com/step-functions/latest/dg/bp-lambda-serviceexception.html
            this.addRetry({
                errors: ['Lambda.ServiceException', 'Lambda.AWSLambdaException', 'Lambda.SdkClientException'],
                interval: cdk.Duration.seconds(2),
                maxAttempts: 6,
                backoffRate: 2,
            });
        }
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    /**
     * @internal
     */
    _renderTask() {
        if (this.props.payloadResponseOnly) {
            return {
                Resource: this.props.lambdaFunction.functionArn,
                ...this.props.payload && { Parameters: sfn.FieldUtils.renderObject(this.props.payload.value) },
            };
        }
        else {
            return {
                Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
                Parameters: sfn.FieldUtils.renderObject({
                    FunctionName: this.props.lambdaFunction.functionArn,
                    Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                    InvocationType: this.props.invocationType,
                    ClientContext: this.props.clientContext,
                    Qualifier: this.props.qualifier,
                }),
            };
        }
    }
}
exports.LambdaInvoke = LambdaInvoke;
_a = JSII_RTTI_SYMBOL_1;
LambdaInvoke[_a] = { fqn: "@aws-cdk/aws-stepfunctions-tasks.LambdaInvoke", version: "1.107.0" };
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda.
 *
 * @stability stable
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    LambdaInvocationType["EVENT"] = "Event";
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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