# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['ptera']

package_data = \
{'': ['*']}

install_requires = \
['Rx>=3.2.0,<4.0.0']

extras_require = \
{':python_version >= "3.6" and python_version < "3.7"': ['contextvars>=2.4,<3.0']}

setup_kwargs = {
    'name': 'ptera',
    'version': '0.3.5',
    'description': 'Call graph addressing library.',
    'long_description': '\n# Ptera\n\n**Note**: This is super alpha. A lot of the features are implemented very inefficiently and the error reporting is not very good. That will be fixed in due time, and then this note will disappear into the mists of git history.\n\n\n## What is Ptera?\n\nPtera is a way to probe arbitrary variables in arbitrary functions in your program, for instance to plot their values over time, to get the maximum or minimum value during execution.\n\n* **Keep your program clean**: Queries can be defined outside of your main function, so there is no need to pollute your code with logging or debug code.\n* **Debug and analyze across scopes**: Easily write queries that collect variables at various points in the call stack, or even across different calls. Then, you can analyze them all together.\n* **Tag variables and functions**: Categorize parts of your program to make more general queries.\n\n```python\nfrom ptera import probing, op\n\ndef fact(n):\n    if n <= 1:\n        return n\n    else:\n        return n * fact(n - 1)\n\nwith probing("fact(n) as v") as probe:\n    probe.pipe(op.keymap(lambda n, v: f"fact({n}) = {v}")).subscribe(print)\n    fact(3)\n    # prints fact(1) = 1; fact(2) = 2; fact(3) = 6\n```\n\n\n## probing\n\nUsage: `with ptera.probing(selector) as probe: ...`\n\nThe **selector** is a specification of which variables in which functions we want to stream through the probe. One of the variables must be the *focus* of the selector, meaning that the probe is triggered when *that* variable is set. The focus may be indicated either as `f(!x)` or `f > x`.\n\nThe **probe** is an instance of [rx.Observable](https://github.com/ReactiveX/RxPY). All of the `rx` [operators](https://rxpy.readthedocs.io/en/latest/reference_operators.html) should therefore work with ptera\'s probes (map, reduce, min, max, debounce, etc.)\n\n\n### Example 1: intermediate variables\n\nPtera is capable of capturing any variable in a function, not just inputs and return values:\n\n```python\ndef fact2(n):\n    curr = 1\n    for i in range(n):\n        curr = curr * (i + 1)\n    return curr\n\nwith probing("fact2(i, !curr)") as probe:\n    probe.subscribe(print)\n    fact2(3)\n    # {\'curr\': 1}\n    # {\'curr\': 1, \'i\': 0}\n    # {\'curr\': 2, \'i\': 1}\n    # {\'curr\': 6, \'i\': 2}\n```\n\nThe "!" in the selector above means that the focus is `curr`. This means it is triggered when `curr` is set. This is why the first result does not have a value for `i`. You can use the selector `fact2(!i, curr)` to focus on `i` instead:\n\n```python\nwith probing("fact2(!i, curr)") as probe:\n    probe.subscribe(print)\n    fact2(3)\n    # {\'i\': 0, \'curr\': 1}\n    # {\'i\': 1, \'curr\': 1}\n    # {\'i\': 2, \'curr\': 2}\n```\n\nYou can see that the associations are different (curr is 2 when i is 2, whereas it was 6 with the other selector), but this is simply because they are now triggered when `i` is set.\n\n### Example 2: multiple scopes\n\nA selector may act on several nested scopes in a call graph. For example, the selector `f(x) >> g(y) >> h > z` would capture variables `x`, `y` and `z` from the scopes of three different functions, but only when `f` calls `g` and `g` calls `h` (either directly or indirectly). (Note: `f(x) > g(y) > h > z` is also legal and is supposed to represent direct calls, but it may behave in confusing ways depending on which functions are instrumented globally, so avoid it for the time being).\n\n```python\ndef f(x):\n    return g(x + 1) * g(-x - 1)\n\ndef g(x):\n    return x * 2\n\n# Use "as" to rename a variable if there is a name conflict\nwith probing("f(x) >> g > x as gx") as probe:\n    probe.subscribe(print)\n    f(5)\n    # {\'gx\': 6, \'x\': 5}\n    # {\'gx\': -6, \'x\': 5}\n    g(10)\n    # Prints nothing\n```\n\n### Example 3: sibling calls\n\nSelectors can also specify variables on different paths in the call graph. For example:\n\n```python\ndef f(x):\n    v = g(x + 1) * h(-x - 1)\n    return v\n\ndef g(y):\n    return y * 2\n\ndef h(z):\n    return z * 3\n\nwith probing("f(x, g(y), h(!z))") as probe:\n    probe.subscribe(print)\n    f(10)\n    # {\'z\': -11, \'x\': 10, \'y\': 11}\n```\n\nRemember to set the focus with `!`. It should ideally be on the last variable to be set.\n\nThere is currently no error if you don\'t set a focus, it will simply do nothing, so beware of that for the time being.\n\n### Example 4: tagging variables\n\nUsing annotations, variables can be given various tags, and probes can use these tags instead of variable names.\n\n```python\ndef fishy(x):\n    a: "@fish" = x + 1\n    b: "@fish & @trout" = x + 2\n    return a * b\n\nwith probing("fishy > $x:@trout") as probe:\n    probe.subscribe(print)\n    fishy(10)\n    # {\'x\': 12}\n\nwith probing("fishy > $x:@fish") as probe:\n    probe.subscribe(print)\n    fishy(10)\n    # {\'x\': 11}\n    # {\'x\': 12}\n```\n\n\n## Probe\n\n`Probe` works more or less the same way as `probing`, but it is not a context manager: it just works globally from the moment of its creation. This means that streams created with `Probe` never actually end, so operators that wait for the full stream before triggering, such as `ptera.op.min`, will not work.\n\n```python\nProbe("fact() as result").subscribe(print)\nfact(2)\n# {\'result\': 1}\n# {\'result\': 2}\nfact(3)\n# {\'result\': 1}\n# {\'result\': 2}\n# {\'result\': 6}\n```\n\n## Absolute probes\n\nHere is a notation to probe a function using an "absolute path" in the module system:\n\n```python\nProbe("/xyz.submodule/Klass/method > x")\n\n# is mostly equivalent to:\n\nfrom xyz.submodule import Klass\nProbe("Klass.method > x")\n```\n\nThe slashes represent a physical nesting rather than object attributes. For example, `/module.submodule/x/y` means:\n\n* Go in the file that defines `module.submodule`\n* Enter `def x` or `class x` (it will *not* work if `x` is imported from elsewhere)\n* Within that definition, enter `def y` or `class y`\n\nNote:\n\n* Unlike the normal notation, the absolute notation bypasses decorators: `/module/function` will probe the function inside the `def function(): ...` in `module.py`, so it will work even if the function was wrapped by a decorator (unless the decorator does not actually call the function).\n* Although the `/module/function/closure` notation can theoretically point to closures, **this does not work yet.** (It will, eventually.)\n* Use `/module.submodule/func`, *not* `/module/submodule/func`. The former roughly corresponds to `from module.submodule import func` and the latter to `from module import submodule; func = submodule.func`, which can be different in Python. It\'s a bit odd, but it works that way to properly address Python quirks.\n\n\n## Operators\n\nAll the existing [operators](https://rxpy.readthedocs.io/en/latest/reference_operators.html) defined in the `rx` package should be compatible with `Probe` and `probing`. They may be imported as `ptera.operators` or `ptera.op` *In addition to this*, `ptera.operators` defines the following operators:\n\n### Utility\n\n* **`getitem(name)`**: extract an item from a stream of dicts\n* **`keymap(fn)`**: calls a function using kwargs from a stream of dicts\n* **`throttle(duration)`**: alias for `rx.operators.throttle_first`\n\n### Arithmetic\n\n* **`roll(n, reduce=None, key_mapper=None, seed=None)`**: transform a stream into rolling windows of size at most n. Successive windows overlap completely except for the first and last elements.\n  * If reduce is provided, it is called with arguments `(last, add, drop, last_size, current_size)`\n  * If transform is provided, it is called on each element\n* **`rolling_average(n, key_mapper=None)`**: efficient implementation of a rolling average (mean of the last n elements)\n* **`rolling_average_and_variance(n, key_mapper=None)`**: efficient implementation of a rolling average and (sample) variance of the last n elements, returned as a tuple.\n\n\n## Query language\n\nHere is some code annotated with queries that will match various variables. The queries are not exhaustive, just examples.\n\n* The semicolon ";" is used to separate queries and it is not part of any query.\n* The hash character "#" *is* part of the query if there is no space after it, otherwise it starts a comment.\n\n```python\nfrom ptera import ptera, tag\n\nAnimal = tag.Animal\nThing = tag.Thing\n\n@tooled\ndef art(a, b):               # art > a ; art > b ; art(!a, b) ; art(a, !b)\n\n    a1: Animal = bee(a)      # a1 ; art > a1 ; art(!a1) ; art > $x\n                             # a1:Animal ; $x:Animal\n                             # art(!a1) > bee > d  # Focus on a1, also includes d\n                             # art > bee  # This refers to the bee function\n                             # * > a1 ; *(!a1)\n\n    a2: Thing = cap(b)       # a2 ; art > a2 ; art(!a2) ; art > $x\n                             # a2:Thing ; $x:Thing\n\n    return a1 + a2           # art > #value ; art(#value as art_result)\n                             # art() as art_result\n                             # art > $x\n\n\n@tooled\ndef bee(c):\n    c1 = c + 1               # bee > c1 ; art >> c1 ; art(a2) > bee > c1\n                             # bee > c1 as xyz\n\n    return c1                # bee > #value ; bee(c) as bee_value\n\n\n@tooled\ndef cap(d: Thing & int):     # cap > d ; $x:Thing ; cap > $x\n                             # art(bee(c)) > cap > d\n    return d * d\n```\n\n* The `!` operator marks the **focus** of the query. There will be one result for each time the focus is triggered, and when using `tweak` or `rewrite` the focus is what is being tweaked or rewritten.\n  * Other variables are supplemental information, available along with the focus in query results. They can also be used to compute a value for the focus *if* they are available by the time the focus is reached.\n  * The nesting operators `>` and `>>` automatically set the focus to the right hand side if the rhs is a single variable and the operator is not inside `(...)`.\n* The wildcard `*` stands in for any function.\n* The `>>` operator represents **deep nesting**. For example, `art >> c1` encompasses the pattern `art > bee > c1`.\n  * In general, `a >> z` encompasses `a > z`, `a > b > z`, `a > b > c > z`, `a > * > z`, and so on.\n* A function\'s return value corresponds to a special variable named `#value`.\n* `$x` will match any variable name. Getting the variable name for the capture is possible but requires the `map_full` method. For example:\n  * Query: `art > $x`\n  * Getting the names: `results.map_full(lambda x: x.name) == ["a1", "a2", "#value"]`\n  * Other fields accessible from `map_full` are `value`, `names` and `values`, the latter two being needed if multiple results are captured together.\n* Variable annotations are preserved and can be filtered on, using the `:` operator. However, *Ptera only recognizes tags* created using `ptera.Tag("XYZ")` or `ptera.tag.XYZ`. It will not filter over types.\n* `art(bee(c)) > cap > d` triggers on the variable `d` in calls to `cap`, but it will *also* include the value of `c` for all calls to `bee` inside `art`.\n  * If there are multiple calls to `bee`, all values of `c` will be pooled together, and it will be necessary to use `map_all` to retrieve the values (or `map_full`).\n',
    'author': 'Olivier Breuleux',
    'author_email': 'breuleux@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/breuleux/ptera',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
