"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxEcrProvider = void 0;
const fs = require("fs");
const https = require("https");
/**
 * The version provider parses a JSON file containing references to ECRs that serve Thinkbox's Deadline Docker images.
 * It can be downloaded or loaded from local file and returns the ECR ARN prefix.
 */
class ThinkboxEcrProvider {
    constructor(indexPath) {
        this.indexPath = indexPath;
    }
    get indexJson() {
        if (!this.indexJsonPromise) {
            this.indexJsonPromise = new Promise((resolve, reject) => {
                try {
                    if (this.indexPath) {
                        return resolve(this.readEcrIndex(this.indexPath));
                    }
                    else {
                        // return resolve(this.getMockEcrIndex());
                        resolve(this.getEcrIndex());
                    }
                }
                catch (e) {
                    return reject(e);
                }
            }).then((json) => {
                // convert the response to a json object and return.
                let data;
                try {
                    data = JSON.parse(json);
                }
                catch (e) {
                    throw new Error(`ECR index file contains invalid JSON: "${e}"`);
                }
                if (this.verifyThinkboxDockerImageIndex(data)) {
                    return data;
                }
                else {
                    throw new Error('This should be a dead code path');
                }
            });
        }
        return this.indexJsonPromise;
    }
    verifyThinkboxDockerImageIndex(index) {
        function expectObject(key, value) {
            const valueType = typeof value;
            if (valueType != 'object') {
                throw new Error(`expected ${key} to be an object but got ${valueType}`);
            }
            if (Array.isArray(index)) {
                throw new Error(`expected ${key} to be an object but got array`);
            }
        }
        function expectString(key, value) {
            const valueType = typeof value;
            if (valueType != 'string') {
                throw new Error(`expected ${key} to be a string but got ${valueType}`);
            }
        }
        expectObject('index', index);
        expectObject('index.registry', index.registry);
        expectString('index.registry.uri', index.registry.uri);
        expectObject('index.products', index.products);
        expectObject('index.products.deadline', index.products.deadline);
        expectString('index.products.deadline.namespace', index.products.deadline.namespace);
        return true;
    }
    /**
     * Gets the global ECR base URI for Thinkbox published Deadline Docker images.
     */
    async getGlobalEcrBaseURI() {
        const indexJson = await this.indexJson;
        const globalEcrBaseURI = `${indexJson.registry.uri}/${indexJson.products.deadline.namespace}`;
        if (globalEcrBaseURI === undefined) {
            throw new Error('No global ECR');
        }
        if (typeof (globalEcrBaseURI) != 'string') {
            throw new Error(`Unexpected type for global base ECR URI: "${typeof (globalEcrBaseURI)}`);
        }
        return globalEcrBaseURI;
    }
    /**
     * Downloads and parses the ECR index.
     *
     * Returns a promise that is resolved with a JSON-parsed object containing the index.
     */
    async getEcrIndex() {
        return new Promise((resolve, reject) => {
            const request = https.get(ThinkboxEcrProvider.ECR_INDEX_URL, (res) => {
                let json = '';
                res.on('data', (chunk) => {
                    // keep appending the response chunks until we get 'end' event.
                    json += chunk;
                });
                res.on('end', () => {
                    // complete response is available here:
                    if (res.statusCode === 200) {
                        resolve(json);
                    }
                    else {
                        reject(new Error(`Expected status code 200, but got ${res.statusCode}`));
                    }
                });
            });
            request.on('error', (err) => {
                reject(err);
            });
        });
    }
    /**
     * This method reads the ECR index from a file and returns a parsed JSON object.
     *
     * @param indexFilePath The path to the ECR index file
     */
    readEcrIndex(indexFilePath) {
        if (!fs.existsSync(indexFilePath)) {
            throw new Error(`File "${indexFilePath}" was not found`);
        }
        const json = fs.readFileSync(indexFilePath, 'utf8');
        return json;
    }
}
exports.ThinkboxEcrProvider = ThinkboxEcrProvider;
/**
 * The URL to obtain the ECR index from.
 */
ThinkboxEcrProvider.ECR_INDEX_URL = 'https://downloads.thinkboxsoftware.com/thinkbox_ecr.json';
//# sourceMappingURL=data:application/json;base64,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