/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { DynamoDB } from 'aws-sdk';
export declare class CompositeStringIndexTable {
    static readonly API_VERSION = "2012-08-10";
    static fromExisting(client: DynamoDB, tableName: string): Promise<CompositeStringIndexTable>;
    /**
     * A simplified interface to create a new DynamoDB Table with a composite index
     * consisting of a pair of string attributes.
     * @param args
     */
    static createNew(args: {
        client: DynamoDB;
        name: string;
        primaryKeyName: string;
        sortKeyName: string;
        region?: string;
        tags?: Array<{
            Key: string;
            Value: string;
        }>;
    }): Promise<CompositeStringIndexTable>;
    readonly primaryKey: string;
    readonly sortKey: string;
    protected readonly client: DynamoDB;
    protected tableName: string | undefined;
    protected constructor(client: DynamoDB, name: string, primaryKey: string, sortKey: string);
    /**
     * Delete this table from DynamoDB.
     */
    deleteTable(): Promise<void>;
    /**
     * Puts an item into the Table. The item it put into the table index with the given
     * primary and sort key attribute values. If any attributes are given, then they are
     * stored in the item.
     *
     * @param props
     * @throws Error if the request fails.
     * @returns True if the item was stored in the table; false otherwise
     */
    putItem(props: {
        primaryKeyValue: string;
        sortKeyValue: string;
        /**
         * Additional attribute values to store in the table. This must
         * not contain values for the primary & sort key attributes.
         * Property key is the attribute name.
         */
        attributes?: object;
        /**
         * If true, then allow dynamo to overwrite an existing value at the index
         * if one exists.
         * @default false
         */
        allow_overwrite?: boolean;
    }): Promise<boolean>;
    /**
     * Does a consistent read to get the item from the Table with the given primary and sort key, if one exists.
     *
     * @param props
     * @throws Error if the request fails.
     * @returns The attributes of the item **excluding** the primary and sort key, or undefined if there was no item
     *         found.
     */
    getItem(props: {
        primaryKeyValue: string;
        sortKeyValue: string;
    }): Promise<{
        [key: string]: any;
    } | undefined>;
    /**
     * Deletes the item from the table that is indexed by the given primary and sort key value
     * @param props
     * @throws Error if the request fails
     * @returns True if the item was deleted; false if there was no matching item to delete.
     */
    deleteItem(props: {
        primaryKeyValue: string;
        sortKeyValue: string;
    }): Promise<boolean>;
    /**
     * Query the table for all items with the given primary key value.
     * @param primaryKeyValue
     * @param pageLimit Maximum number of table items to evaluate per request.
     * @throws Error if the request fails.
     * @returns All of the found items, keyed by their unique sort key values. i.e.
     *         {
     *             'sort key value 1': {
     *                 # attributes other than sort & primary key for this item
     *             },
     *             'sort key value 2': {
     *                 # attributes other than sort & primary key for this item
     *             },
     *             ... etc
     *         }
     */
    query(primaryKeyValue: string, pageLimit?: number): Promise<{
        [key: string]: {
            [key: string]: any;
        };
    }>;
}
