"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const http_1 = require("http");
const net_1 = require("net");
const deadline_client_1 = require("../../deadline-client");
const spot_event_plugin_client_1 = require("../spot-event-plugin-client");
describe('SpotEventPluginClient', () => {
    let spotEventPluginClient;
    let describeDataResponse;
    let successfulResponse;
    let consoleLogMock;
    let consoleErrorMock;
    beforeEach(() => {
        consoleLogMock = jest.spyOn(console, 'log').mockReturnValue(undefined);
        consoleErrorMock = jest.spyOn(console, 'error').mockReturnValue(undefined);
        describeDataResponse = {
            data: {
                ServerData: [{
                        ID: 'event.plugin.spot',
                        ConcurrencyToken: 'token',
                    }],
            },
            fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
        };
        successfulResponse = {
            data: {},
            fullResponse: new http_1.IncomingMessage(new net_1.Socket()),
        };
        spotEventPluginClient = new spot_event_plugin_client_1.SpotEventPluginClient(new deadline_client_1.DeadlineClient({
            host: 'test',
            port: 0,
            protocol: 'HTTP',
        }));
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn();
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].GetRequest = jest.fn();
    });
    afterEach(() => {
        jest.clearAllMocks();
    });
    test('successful saveServerData', async () => {
        // GIVEN
        const configuration = 'configuration';
        const mockSuccessfulPostRequest = jest.fn((_a) => Promise.resolve(successfulResponse));
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['concurrencyToken'] = jest.fn().mockResolvedValue('token');
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = mockSuccessfulPostRequest;
        const result = await spotEventPluginClient.saveServerData(configuration);
        // THEN
        expect(result).toBeTruthy();
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(1);
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/Saving server data configuration:/);
        expect(consoleLogMock.mock.calls[1][0]).toMatch(configuration);
    });
    test('failed saveServerData on post request', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['concurrencyToken'] = jest.fn().mockResolvedValue('token');
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(statusMessage);
        const result = await spotEventPluginClient.saveServerData('configuration');
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock.mock.calls.length).toBe(1);
        expect(consoleErrorMock.mock.calls[0][0]).toMatch(`Failed to save server data. Reason: ${statusMessage}`);
    });
    test('failed saveServerData on concurrency token', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['concurrencyToken'] = jest.fn().mockRejectedValue(statusMessage);
        const result = await spotEventPluginClient.saveServerData('configuration');
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock.mock.calls.length).toBe(1);
        expect(consoleErrorMock.mock.calls[0][0]).toMatch(`Failed to save server data. Reason: ${statusMessage}`);
    });
    test('successful configureSpotEventPlugin', async () => {
        // GIVEN
        const configs = [
            {
                Key: 'testkey',
                Value: 'testValue',
            },
            {
                Key: 'testkey2',
                Value: 'testValue2',
            },
        ];
        const mockConfigureSpotEventPlugin = jest.fn((_a) => Promise.resolve(successfulResponse));
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = mockConfigureSpotEventPlugin;
        const result = await spotEventPluginClient.configureSpotEventPlugin(configs);
        // THEN
        expect(result).toBeTruthy();
        // eslint-disable-next-line dot-notation
        expect(spotEventPluginClient['deadlineClient'].PostRequest).toBeCalledTimes(1);
        expect(consoleLogMock.mock.calls.length).toBe(2);
        expect(consoleLogMock.mock.calls[0][0]).toMatch(/Saving plugin configuration:/);
        expect(consoleLogMock.mock.calls[1][0]).toEqual(configs);
    });
    test('failed configureSpotEventPlugin', async () => {
        // GIVEN
        const configs = [
            {
                Key: 'testkey',
                Value: 'testValue',
            },
            {
                Key: 'testkey2',
                Value: 'testValue2',
            },
        ];
        const statusMessage = 'error message';
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(statusMessage);
        const result = await spotEventPluginClient.configureSpotEventPlugin(configs);
        // THEN
        expect(result).toBeFalsy();
        expect(consoleErrorMock.mock.calls.length).toBe(1);
        expect(consoleErrorMock.mock.calls[0][0]).toMatch(`Failed to save plugin configuration. Reason: ${statusMessage}`);
    });
    test('valid concurrency token', async () => {
        // GIVEN
        const concurrencyToken = 'TOKEN';
        const validResponse = {
            data: {
                ServerData: [{
                        ID: 'event.plugin.spot',
                        ConcurrencyToken: concurrencyToken,
                    }],
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['describeServerData'] = jest.fn().mockResolvedValue(validResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['concurrencyToken']();
        // THEN
        expect(result).toBe(concurrencyToken);
    });
    test('returns empty token if no event plugin id entry', async () => {
        // GIVEN
        const noSpotEventOluginResponse = {
            data: {
                ServerData: [{
                        ID: 'NOT.event.plugin.spot',
                        ConcurrencyToken: 'token',
                    }],
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['describeServerData'] = jest.fn().mockResolvedValue(noSpotEventOluginResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['concurrencyToken']();
        // THEN
        expect(result).toBe('');
    });
    test('throws if invalid server data', async () => {
        // GIVEN
        const invalidDescribeDataResponse = {
            data: {
                NotServerData: {},
            },
        };
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['describeServerData'] = jest.fn().mockResolvedValue(invalidDescribeDataResponse);
        // eslint-disable-next-line dot-notation
        const promise = spotEventPluginClient['concurrencyToken']();
        // THEN
        await expect(promise).rejects.toThrowError(`Failed to receive a ConcurrencyToken. Invalid response: ${invalidDescribeDataResponse.data}.`);
    });
    test('successful describeServerData', async () => {
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockResolvedValue(describeDataResponse);
        // eslint-disable-next-line dot-notation
        const result = await spotEventPluginClient['describeServerData']();
        // THEN
        expect(result).toEqual(describeDataResponse);
    });
    test('failed describeServerData', async () => {
        // GIVEN
        const statusMessage = 'error message';
        // WHEN
        // eslint-disable-next-line dot-notation
        spotEventPluginClient['deadlineClient'].PostRequest = jest.fn().mockRejectedValue(statusMessage);
        // eslint-disable-next-line dot-notation
        const promise = spotEventPluginClient['describeServerData']();
        // THEN
        await expect(promise).rejects.toEqual(statusMessage);
    });
});
//# sourceMappingURL=data:application/json;base64,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