"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.validateStringOptional = exports.validateString = exports.convertToBooleanOptional = exports.convertToBoolean = exports.convertToIntOptional = exports.convertToInt = exports.convertLaunchSpecifications = exports.convertInstanceProfile = exports.isValidInstanceProfile = exports.convertBlockDeviceMapping = exports.convertEbs = exports.isValidDeviceMapping = exports.convertTagSpecifications = exports.isValidTagSpecification = exports.convertSecurityGroups = exports.isValidSecurityGroup = exports.validateProperty = exports.validateArray = exports.convertSpotEventPluginSettings = exports.convertSpotFleetRequestConfiguration = void 0;
/**
 * Convert the configuration we received from ConfigureSpotEventPlugin construct to the fromat expected by the Spot Event Plugin.
 * boolean and number properties get converted into strings when passed to the Lambda,
 * so we need to restore the original types.
 */
function convertSpotFleetRequestConfiguration(spotFleetRequestConfigs) {
    const convertedSpotFleetRequestConfigs = {};
    for (const [group_name, sfrConfigs] of Object.entries(spotFleetRequestConfigs)) {
        const convertedSpotFleetRequestProps = {
            AllocationStrategy: validateString(sfrConfigs.AllocationStrategy, `${group_name}.AllocationStrategy`),
            IamFleetRole: validateString(sfrConfigs.IamFleetRole, `${group_name}.IamFleetRole`),
            LaunchSpecifications: convertLaunchSpecifications(sfrConfigs.LaunchSpecifications, `${group_name}.LaunchSpecifications`),
            ReplaceUnhealthyInstances: convertToBoolean(sfrConfigs.ReplaceUnhealthyInstances, `${group_name}.ReplaceUnhealthyInstances`),
            TargetCapacity: convertToInt(sfrConfigs.TargetCapacity, `${group_name}.TargetCapacity`),
            TerminateInstancesWithExpiration: convertToBoolean(sfrConfigs.TerminateInstancesWithExpiration, `${group_name}.TerminateInstancesWithExpiration`),
            Type: validateString(sfrConfigs.Type, `${group_name}.Type`),
            ValidUntil: validateStringOptional(sfrConfigs.ValidUntil, `${group_name}.ValidUntil`),
            TagSpecifications: convertTagSpecifications(sfrConfigs.TagSpecifications, `${group_name}.TagSpecifications`),
        };
        convertedSpotFleetRequestConfigs[group_name] = convertedSpotFleetRequestProps;
    }
    return convertedSpotFleetRequestConfigs;
}
exports.convertSpotFleetRequestConfiguration = convertSpotFleetRequestConfiguration;
/**
 * Convert the configuration we received from ConfigureSpotEventPlugin construct to the fromat expected by the Spot Event Plugin.
 * boolean and number properties get converted into strings when passed to the Lambda,
 * so we need to restore the original types.
 */
function convertSpotEventPluginSettings(pluginOptions) {
    return {
        AWSInstanceStatus: validateString(pluginOptions.AWSInstanceStatus, 'AWSInstanceStatus'),
        DeleteInterruptedSlaves: convertToBoolean(pluginOptions.DeleteInterruptedSlaves, 'DeleteInterruptedSlaves'),
        DeleteTerminatedSlaves: convertToBoolean(pluginOptions.DeleteTerminatedSlaves, 'DeleteTerminatedSlaves'),
        IdleShutdown: convertToInt(pluginOptions.IdleShutdown, 'IdleShutdown'),
        Logging: validateString(pluginOptions.Logging, 'Logging'),
        PreJobTaskMode: validateString(pluginOptions.PreJobTaskMode, 'PreJobTaskMode'),
        Region: validateString(pluginOptions.Region, 'Region'),
        ResourceTracker: convertToBoolean(pluginOptions.ResourceTracker, 'ResourceTracker'),
        StaggerInstances: convertToInt(pluginOptions.StaggerInstances, 'StaggerInstances'),
        State: validateString(pluginOptions.State, 'State'),
        StrictHardCap: convertToBoolean(pluginOptions.StrictHardCap, 'StrictHardCap'),
    };
}
exports.convertSpotEventPluginSettings = convertSpotEventPluginSettings;
function validateArray(input, propertyName) {
    if (!input || !Array.isArray(input) || input.length === 0) {
        throw new Error(`${propertyName} should be an array with at least one element.`);
    }
}
exports.validateArray = validateArray;
function validateProperty(isValid, property, propertyName) {
    if (!isValid(property)) {
        throw new Error(`${propertyName} type is not valid.`);
    }
}
exports.validateProperty = validateProperty;
function isValidSecurityGroup(securityGroup) {
    if (!securityGroup || typeof (securityGroup) !== 'object' || Array.isArray(securityGroup)) {
        return false;
    }
    // We also verify groupId with validateString later
    if (!securityGroup.GroupId || typeof (securityGroup.GroupId) !== 'string') {
        return false;
    }
    return true;
}
exports.isValidSecurityGroup = isValidSecurityGroup;
function convertSecurityGroups(securityGroups, propertyName) {
    validateArray(securityGroups, propertyName);
    const convertedSecurityGroups = securityGroups.map(securityGroup => {
        validateProperty(isValidSecurityGroup, securityGroup, propertyName);
        const convertedSecurityGroup = {
            GroupId: validateString(securityGroup.GroupId, `${propertyName}.GroupId`),
        };
        return convertedSecurityGroup;
    });
    return convertedSecurityGroups;
}
exports.convertSecurityGroups = convertSecurityGroups;
function isValidTagSpecification(tagSpecification) {
    if (!tagSpecification || typeof (tagSpecification) !== 'object' || Array.isArray(tagSpecification)) {
        return false;
    }
    // We also verify resourceType with validateString later
    if (!tagSpecification.ResourceType || typeof (tagSpecification.ResourceType) !== 'string') {
        return false;
    }
    if (!tagSpecification.Tags || !Array.isArray(tagSpecification.Tags)) {
        return false;
    }
    for (let element of tagSpecification.Tags) {
        if (!element || typeof (element) !== 'object') {
            return false;
        }
        ;
        if (!element.Key || typeof (element.Key) !== 'string') {
            return false;
        }
        if (!element.Value || typeof (element.Value) !== 'string') {
            return false;
        }
    }
    return true;
}
exports.isValidTagSpecification = isValidTagSpecification;
function convertTagSpecifications(tagSpecifications, propertyName) {
    validateArray(tagSpecifications, propertyName);
    const convertedTagSpecifications = tagSpecifications.map(tagSpecification => {
        validateProperty(isValidTagSpecification, tagSpecification, propertyName);
        const convertedTagSpecification = {
            ResourceType: validateString(tagSpecification.ResourceType, `${propertyName}.ResourceType`),
            Tags: tagSpecification.Tags,
        };
        return convertedTagSpecification;
    });
    return convertedTagSpecifications;
}
exports.convertTagSpecifications = convertTagSpecifications;
function isValidDeviceMapping(deviceMapping) {
    if (!deviceMapping || typeof (deviceMapping) !== 'object' || Array.isArray(deviceMapping)) {
        return false;
    }
    // We validate the rest properties when convert them.
    return true;
}
exports.isValidDeviceMapping = isValidDeviceMapping;
function convertEbs(ebs, propertyName) {
    const convertedEbs = {
        DeleteOnTermination: convertToBooleanOptional(ebs.DeleteOnTermination, `${propertyName}.DeleteOnTermination`),
        Encrypted: convertToBooleanOptional(ebs.Encrypted, `${propertyName}.Encrypted`),
        Iops: convertToIntOptional(ebs.Iops, `${propertyName}.Iops`),
        SnapshotId: validateStringOptional(ebs.SnapshotId, `${propertyName}.SnapshotId`),
        VolumeSize: convertToIntOptional(ebs.VolumeSize, `${propertyName}.VolumeSize`),
        VolumeType: validateStringOptional(ebs.VolumeType, `${propertyName}.VolumeType`),
    };
    return convertedEbs;
}
exports.convertEbs = convertEbs;
function convertBlockDeviceMapping(blockDeviceMappings, propertyName) {
    validateArray(blockDeviceMappings, propertyName);
    const convertedBlockDeviceMappings = blockDeviceMappings.map(deviceMapping => {
        validateProperty(isValidDeviceMapping, deviceMapping, propertyName);
        const convertedDeviceMapping = {
            DeviceName: validateString(deviceMapping.DeviceName, `${propertyName}.DeviceName`),
            Ebs: deviceMapping.Ebs ? convertEbs(deviceMapping.Ebs, `${propertyName}.Ebs`) : undefined,
            NoDevice: validateStringOptional(deviceMapping.NoDevice, `${propertyName}.NoDevice`),
            VirtualName: validateStringOptional(deviceMapping.VirtualName, `${propertyName}.VirtualName`),
        };
        return convertedDeviceMapping;
    });
    return convertedBlockDeviceMappings;
}
exports.convertBlockDeviceMapping = convertBlockDeviceMapping;
function isValidInstanceProfile(instanceProfile) {
    if (!instanceProfile || typeof (instanceProfile) !== 'object' || Array.isArray(instanceProfile)) {
        return false;
    }
    // We also verify arn with validateString later
    if (!instanceProfile.Arn || typeof (instanceProfile.Arn) !== 'string') {
        return false;
    }
    return true;
}
exports.isValidInstanceProfile = isValidInstanceProfile;
function convertInstanceProfile(instanceProfile, propertyName) {
    validateProperty(isValidInstanceProfile, instanceProfile, propertyName);
    const convertedInstanceProfile = {
        Arn: validateString(instanceProfile.Arn, `${propertyName}.Arn`),
    };
    return convertedInstanceProfile;
}
exports.convertInstanceProfile = convertInstanceProfile;
function convertLaunchSpecifications(launchSpecifications, propertyName) {
    validateArray(launchSpecifications, propertyName);
    const convertedLaunchSpecifications = [];
    launchSpecifications.map(launchSpecification => {
        const SecurityGroups = convertSecurityGroups(launchSpecification.SecurityGroups, `${propertyName}.SecurityGroups`);
        const TagSpecifications = convertTagSpecifications(launchSpecification.TagSpecifications, `${propertyName}.TagSpecifications`);
        const BlockDeviceMappings = launchSpecification.BlockDeviceMappings ?
            convertBlockDeviceMapping(launchSpecification.BlockDeviceMappings, `${propertyName}.BlockDeviceMappings`) : undefined;
        const convertedLaunchSpecification = {
            BlockDeviceMappings,
            IamInstanceProfile: convertInstanceProfile(launchSpecification.IamInstanceProfile, `${propertyName}.IamInstanceProfile`),
            ImageId: validateString(launchSpecification.ImageId, `${propertyName}.ImageId`),
            KeyName: validateStringOptional(launchSpecification.KeyName, `${propertyName}.KeyName`),
            SecurityGroups,
            SubnetId: validateStringOptional(launchSpecification.SubnetId, `${propertyName}.SubnetId`),
            TagSpecifications,
            UserData: validateString(launchSpecification.UserData, `${propertyName}.UserData`),
            InstanceType: validateString(launchSpecification.InstanceType, `${propertyName}.InstanceType`),
        };
        convertedLaunchSpecifications.push(convertedLaunchSpecification);
    });
    return convertedLaunchSpecifications;
}
exports.convertLaunchSpecifications = convertLaunchSpecifications;
function convertToInt(value, propertyName) {
    if (typeof (value) === 'number') {
        if (Number.isInteger(value)) {
            return value;
        }
    }
    if (typeof (value) === 'string') {
        const result = Number.parseFloat(value);
        if (Number.isInteger(result)) {
            return result;
        }
    }
    throw new Error(`The value of ${propertyName} should be an integer. Received: ${value} of type ${typeof (value)}`);
}
exports.convertToInt = convertToInt;
function convertToIntOptional(value, propertyName) {
    if (value === undefined) {
        return undefined;
    }
    return convertToInt(value, propertyName);
}
exports.convertToIntOptional = convertToIntOptional;
function convertToBoolean(value, propertyName) {
    if (typeof (value) === 'boolean') {
        return value;
    }
    if (typeof (value) === 'string') {
        if (value === 'true') {
            return true;
        }
        if (value === 'false') {
            return false;
        }
    }
    throw new Error(`The value of ${propertyName} should be a boolean. Received: ${value} of type ${typeof (value)}`);
}
exports.convertToBoolean = convertToBoolean;
function convertToBooleanOptional(value, propertyName) {
    if (value === undefined) {
        return undefined;
    }
    return convertToBoolean(value, propertyName);
}
exports.convertToBooleanOptional = convertToBooleanOptional;
function validateString(value, propertyName) {
    if (typeof (value) === 'string') {
        return value;
    }
    throw new Error(`The value of ${propertyName} should be a string. Received: ${value} of type ${typeof (value)}`);
}
exports.validateString = validateString;
function validateStringOptional(value, propertyName) {
    if (value === undefined) {
        return undefined;
    }
    return validateString(value, propertyName);
}
exports.validateStringOptional = validateStringOptional;
//# sourceMappingURL=data:application/json;base64,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