"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable dot-notation */
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('Test WorkerInstanceConfiguration for Linux', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        const userData = stack.resolve(instance.userData.render());
        // // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`),
                ],
            ],
        });
    });
    test('custom listener port', () => {
        const otherListenerPort = 55555;
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                listenerPort: otherListenerPort,
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${otherListenerPort} /tmp/`),
                ],
            ],
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'g1,g2\' \'p1,p2\' \'r1\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`),
                ],
            ],
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudwatchLogSettings: logGroupProps,
        });
        const ssmParam = config.node.findChild('StringParameter');
        const logGroup = config.node.findChild('ConfigLogGroupWrapper');
        const ssmParamName = stack.resolve(ssmParam.parameterName);
        const logGroupName = stack.resolve(logGroup.logGroupName);
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '#!/bin/bash\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxCloudWatchScriptBoilerplate(),
                    '\' ',
                    ssmParamName,
                    '\nmkdir -p $(dirname \'/tmp/',
                    ...asset_constants_1.linuxConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} /tmp/`),
                ],
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"cloud-init-output-{instance_id}\",\"file_path\":\"/var/log/cloud-init-output.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"/var/log/Thinkbox/Deadline10/deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        }));
    });
});
describe('Test WorkerInstanceConfiguration for Windows', () => {
    let stack;
    let vpc;
    let instance;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('groups, pools, region setup', () => {
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                groups: ['g1', 'g2'],
                pools: ['p1', 'p2'],
                region: 'r1',
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'g1,g2\' \'p1,p2\' \'r1\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('custom listner port', () => {
        const otherListenerPort = 55555;
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            workerSettings: {
                listenerPort: otherListenerPort,
            },
        });
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${otherListenerPort} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
    });
    test('log setup', () => {
        // GIVEN
        const logGroupProps = {
            logGroupPrefix: '/test-prefix/',
        };
        // WHEN
        const config = new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            cloudwatchLogSettings: logGroupProps,
        });
        const ssmParam = config.node.findChild('StringParameter');
        const logGroup = config.node.findChild('ConfigLogGroupWrapper');
        const ssmParamName = stack.resolve(ssmParam.parameterName);
        const logGroupName = stack.resolve(logGroup.logGroupName);
        const userData = stack.resolve(instance.userData.render());
        // THEN
        expect(userData).toStrictEqual({
            'Fn::Join': [
                '',
                [
                    '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsCloudWatchScriptBoilerplate(),
                    '\' ',
                    ssmParamName,
                    '\nif (!$?) { Write-Error \'Failed to execute the file \"C:/temp/',
                    {
                        'Fn::Select': [
                            0,
                            {
                                'Fn::Split': [
                                    '||',
                                    {
                                        Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key,
                                    },
                                ],
                            },
                        ],
                    },
                    {
                        'Fn::Select': [
                            1,
                            {
                                'Fn::Split': [
                                    '||',
                                    { Ref: asset_constants_1.CWA_ASSET_WINDOWS.Key },
                                ],
                            },
                        ],
                    },
                    '\"\' -ErrorAction Stop }' +
                        '\nmkdir (Split-Path -Path \'C:/temp/',
                    ...asset_constants_1.windowsConfigureWorkerScriptBoilerplate(`\' \'\' \'\' \'\' \'${lib_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}\' ${lib_1.WorkerInstanceConfiguration['DEFAULT_LISTENER_PORT']} C:/temp/`),
                    '\"\' -ErrorAction Stop }</powershell>',
                ],
            ],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::SSM::Parameter', {
            Value: {
                'Fn::Join': [
                    '',
                    [
                        '{\"logs\":{\"logs_collected\":{\"files\":{\"collect_list\":[{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"UserdataExecution-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Amazon\\\\EC2-Windows\\\\Launch\\\\Log\\\\UserdataExecution.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"WorkerLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlineslave*.log\",\"timezone\":\"Local\"},{\"log_group_name\":\"',
                        logGroupName,
                        '\",\"log_stream_name\":\"LauncherLogs-{instance_id}\",\"file_path\":\"C:\\\\ProgramData\\\\Thinkbox\\\\Deadline10\\\\logs\\\\deadlinelauncher*.log\",\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}',
                    ],
                ],
            },
        }));
    });
});
describe('Test WorkerInstanceConfiguration connect to RenderQueue', () => {
    let stack;
    let vpc;
    let renderQueue;
    let renderQueueSGId;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        const version = new lib_1.VersionQuery(stack, 'Version');
        renderQueue = new lib_1.RenderQueue(stack, 'RQ', {
            version,
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(stack, 'Repository', {
                vpc,
                version,
            }),
        });
        const rqSecGrp = renderQueue.connections.securityGroups[0];
        renderQueueSGId = stack.resolve(rqSecGrp.securityGroupId);
    });
    test('For Linux', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        }));
    });
    test('For Windows', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE),
        });
        // WHEN
        new lib_1.WorkerInstanceConfiguration(stack, 'Config', {
            worker: instance,
            renderQueue,
        });
        const instanceSG = instance.connections.securityGroups[0];
        const instanceSGId = stack.resolve(instanceSG.securityGroupId);
        // THEN
        // Open-box testing. We know that we invoked the connection method on the
        // render queue if the security group for the instance has an ingress rule to the RQ.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
            IpProtocol: 'tcp',
            ToPort: 8080,
            SourceSecurityGroupId: instanceSGId,
            GroupId: renderQueueSGId,
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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