"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceConfiguration = exports.InstanceUserDataProvider = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const version_1 = require("./version");
/**
 * Implementation of {@link IInstanceUserDataProvider}.
 *
 * Can be used as sub-class with override the desired methods
 * to add custom user data commands for WorkerInstanceFleet or WorkerInstanceConfiguration.
 *
 * @stability stable
 */
class InstanceUserDataProvider extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id) {
        super(scope, id);
    }
    /**
     * Method that is invoked before configuring the Cloud Watch Agent.
     *
     * @stability stable
     * @inheritdoc true
     */
    preCloudWatchAgent(_host) {
    }
    /**
     * Method that is invoked before the render queue configuration.
     *
     * @stability stable
     * @inheritdoc true
     */
    preRenderQueueConfiguration(_host) {
    }
    /**
     * Method that is invoked after configuring the connection to the render queue and before configuring the Deadline Worker.
     *
     * @stability stable
     * @inheritdoc true
     */
    preWorkerConfiguration(_host) {
    }
    /**
     * Method that is invoked after all configuration is done and worker started.
     *
     * @stability stable
     * @inheritdoc true
     */
    postWorkerLaunch(_host) {
    }
}
exports.InstanceUserDataProvider = InstanceUserDataProvider;
_a = JSII_RTTI_SYMBOL_1;
InstanceUserDataProvider[_a] = { fqn: "aws-rfdk.deadline.InstanceUserDataProvider", version: "0.26.0" };
/**
 * This construct can be used to configure Deadline Workers on an instance to connect to a RenderQueue, stream their log files to CloudWatch, and configure various settings of the Deadline Worker.
 *
 * The configuration happens on instance start-up using user data scripting.
 *
 * This configuration performs the following steps in order:
 * 1) Configure Cloud Watch Agent
 * 2) Configure Deadline Worker RenderQueue connection
 * 3) Configure Deadline Worker settings
 *
 * A `userDataProvider` can be specified that defines callback functions.
 * These callbacks can be used to inject user data commands at different points during the Worker instance configuration.
 *
 * Security Considerations
 * ------------------------
 * - The instances configured by this construct will download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 *
 * @stability stable
 */
class WorkerInstanceConfiguration extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j;
        super(scope, id);
        (_c = props.userDataProvider) === null || _c === void 0 ? void 0 : _c.preCloudWatchAgent(props.worker);
        if (props.cloudwatchLogSettings) {
            this.configureCloudWatchLogStream(props.worker, id, props.cloudwatchLogSettings);
        }
        (_d = props.userDataProvider) === null || _d === void 0 ? void 0 : _d.preRenderQueueConfiguration(props.worker);
        (_e = props.renderQueue) === null || _e === void 0 ? void 0 : _e.configureClientInstance({ host: props.worker });
        (_f = props.userDataProvider) === null || _f === void 0 ? void 0 : _f.preWorkerConfiguration(props.worker);
        this.listenerPort = (_h = (_g = props.workerSettings) === null || _g === void 0 ? void 0 : _g.listenerPort) !== null && _h !== void 0 ? _h : WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT;
        this.configureWorkerSettings(props.worker, id, props.workerSettings);
        (_j = props.userDataProvider) === null || _j === void 0 ? void 0 : _j.postWorkerLaunch(props.worker);
    }
    /**
     * This method can be used to configure a Deadline Worker instance to stream its logs to the AWS CloudWatch service.
     *
     * The logs that this configures to stream are:
     * - EC2 Instance UserData execution; this is the startup scripting that is run when the instance launches
     *    for the first time.
     * - Deadline Worker logs.
     * - Deadline Launcher logs.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param logGroupProps Configuration for the log group in CloudWatch.
     * @stability stable
     */
    configureCloudWatchLogStream(worker, id, logGroupProps) {
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, id, logGroupProps);
        logGroup.grantWrite(worker);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        if (worker.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        }
        else {
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
            cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        }
        new core_2.CloudWatchAgent(this, `${id}LogsConfig`, {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: worker,
        });
    }
    /**
     * This method can be used to set up the Deadline Worker application on an EC2 instance.
     *
     * From a practical
     * perspective, this is executing the script found in aws-rfdk/lib/deadline/scripts/[bash,powershell]/configureWorker.[sh,ps1]
     * to configure the Deadline Worker application.
     *
     * @param worker The worker to configure.
     * @param id Identifier to disambiguate the resources that are created.
     * @param settings The Deadline Worker settings to apply.
     * @stability stable
     */
    configureWorkerSettings(worker, id, settings) {
        var _c, _d, _e, _f, _g;
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, `${id}ConfigScript`, {
            osType: worker.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        const configureWorkerPortAsset = new aws_s3_assets_1.Asset(this, `${id}WorkerListenerScript`, {
            path: path.join(__dirname, '..', 'scripts', 'python', 'worker-listening-port.py'),
        });
        const configWorkerPortLocalPath = worker.userData.addS3DownloadCommand({
            bucket: configureWorkerPortAsset.bucket,
            bucketKey: configureWorkerPortAsset.s3ObjectKey,
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = (_d = (_c = settings === null || settings === void 0 ? void 0 : settings.groups) === null || _c === void 0 ? void 0 : _c.map(val => val.toLowerCase()).join(',')) !== null && _d !== void 0 ? _d : '';
        const pools = (_f = (_e = settings === null || settings === void 0 ? void 0 : settings.pools) === null || _e === void 0 ? void 0 : _e.map(val => val.toLowerCase()).join(',')) !== null && _f !== void 0 ? _f : '';
        configureWorkerScriptAsset.executeOn({
            host: worker,
            args: [
                `'${groups}'`,
                `'${pools}'`,
                `'${(_g = settings === null || settings === void 0 ? void 0 : settings.region) !== null && _g !== void 0 ? _g : ''}'`,
                `'${version_1.Version.MINIMUM_SUPPORTED_DEADLINE_VERSION.toString()}'`,
                this.listenerPort.toString(),
                configWorkerPortLocalPath,
            ],
        });
    }
}
exports.WorkerInstanceConfiguration = WorkerInstanceConfiguration;
_b = JSII_RTTI_SYMBOL_1;
WorkerInstanceConfiguration[_b] = { fqn: "aws-rfdk.deadline.WorkerInstanceConfiguration", version: "0.26.0" };
/**
 * The default port to use for a worker to listen on for remote commands.
 */
WorkerInstanceConfiguration.DEFAULT_LISTENER_PORT = 56032;
//# sourceMappingURL=data:application/json;base64,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