/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { DockerImageAsset } from '@aws-cdk/aws-ecr-assets';
import { Construct } from '@aws-cdk/core';
import { IVersion, RenderQueueImages, Stage, UsageBasedLicensingImages } from '.';
/**
 * An enum that is associated with AWS Thinkbox managed recipes that are available in the stage manifest.
 *
 * @stability stable
 */
export declare enum ThinkboxManagedDeadlineDockerRecipes {
    /**
     * The Docker Image Asset for the Remote Connection Server.
     *
     * @stability stable
     */
    REMOTE_CONNECTION_SERVER = "rcs",
    /**
     * The Docker Image Asset for the License Forwarder.
     *
     * @stability stable
     */
    LICENSE_FORWARDER = "license-forwarder"
}
/**
 * Interface to specify the properties when instantiating a {@link ThinkboxDockerRecipes} instnace.
 *
 * @stability stable
 */
export interface ThinkboxDockerRecipesProps {
    /**
     * The place where Deadline is staged.
     *
     * @stability stable
     */
    readonly stage: Stage;
}
/**
 * An API for interacting with staged Deadline Docker recipes provided by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * @stability stable
 * @example
 *
 * Construct a RenderQueue
 *
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { RenderQueue, Repository, ThinkboxDockerRecipes } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(app, stack);
 * const recipes = new ThinkboxDockerRecipes(stack, 'Recipes', {
 * path: '/path/to/staged/recipes'
 * });
 * const repository = new Repository(stack, 'Repository', {
 * vpc,
 * version: recipes.version
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 * images: recipes.renderQueueImages,
 * // ...
 * });
 */
export declare class ThinkboxDockerRecipes extends Construct {
    /**
     * A {@link DockerImageAsset} that can be used to build Thinkbox's Deadline RCS Docker Recipe into a container image that can be deployed in CDK.
     *
     * @stability stable
     */
    readonly remoteConnectionServer: DockerImageAsset;
    /**
     * A {@link DockerImageAsset} that can be used to build Thinkbox's Deadline License Forwarder Docker Recipe into a container image that can be deployed in CDK.
     *
     * @stability stable
     */
    readonly licenseForwarder: DockerImageAsset;
    /**
     * Docker images staged locally for use with the {@link RenderQueue} construct.
     *
     * @stability stable
     */
    readonly renderQueueImages: RenderQueueImages;
    /**
     * Docker images staged locally for use with the {@link UsageBasedLicensing} construct.
     *
     * @stability stable
     */
    readonly ublImages: UsageBasedLicensingImages;
    /**
     * The version of Deadline in the stage directory.
     *
     * @stability stable
     */
    readonly version: IVersion;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: ThinkboxDockerRecipesProps);
}
