"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const runtime_info_1 = require("../../core/lib/runtime-info");
const database_connection_1 = require("./database-connection");
const version_query_1 = require("./version-query");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository. If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * - Encrypted Amazon Elastic File System (EFS) - If no file system is provided.
 * - An Amazon DocumentDB - If no database connection is provided.
 * - Auto Scaling Group (ASG) with min & max capacity of 1 instance.
 * - Instance Role and corresponding IAM Policy.
 * - An Amazon CloudWatch log group that contains the Deadline Repository installation logs.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The file system that is created by, or provided to, this construct contains the data for Deadline's Repository file
 *    system. This file system contains information about your submitted jobs, and the plugin scripts that are run by the
 *    Deadline applications in your render farm. An actor that can modify the contents of this file system can cause your
 *    Deadline applications to run code of their choosing. You should restrict access to this file system to only those who
 *    require it.
 * - The database that is created by, or provided to, this construct is used by Deadline to store data about its configuration,
 *    submitted jobs, machine information and status, and so on. An actor with access to this database can read any information
 *    that is entered into Deadline, and modify the bevavior of your render farm. You should restrict access to this database
 *    to only those who require it.
 *
 * @stability stable
 */
class Repository extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p, _q, _r, _s, _t;
        super(scope, id);
        if (props.database && ((_b = props.backupOptions) === null || _b === void 0 ? void 0 : _b.databaseRetention)) {
            this.node.addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.filesystem)) {
            this.node.addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_d = props.removalPolicy) === null || _d === void 0 ? void 0 : _d.database)) {
            this.node.addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        this.version = props.version;
        // Set up the Filesystem of the repository
        if (props.fileSystem !== undefined) {
            this.fileSystem = props.fileSystem;
        }
        else {
            this.efs = new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_g = (_f = props.removalPolicy) === null || _f === void 0 ? void 0 : _f.filesystem) !== null && _g !== void 0 ? _g : core_1.RemovalPolicy.RETAIN,
                securityGroup: (_h = props.securityGroupsOptions) === null || _h === void 0 ? void 0 : _h.fileSystem,
            });
            this.fileSystem = new core_2.MountableEfs(this, {
                filesystem: this.efs,
            });
        }
        // Set up the Database of the repository
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                const warningMsg = 'The parameter databaseAuditLogging only has an effect when the Repository is creating its own database.\n' +
                    'Please ensure that the Database provided is configured correctly.';
                this.node.addWarning(warningMsg);
            }
        }
        else {
            const databaseAuditLogging = (_j = props.databaseAuditLogging) !== null && _j !== void 0 ? _j : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_k = props.documentDbInstanceCount) !== null && _k !== void 0 ? _k : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                engineVersion: '3.6.0',
                instanceProps: {
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                    vpc: props.vpc,
                    vpcSubnets: (_l = props.vpcSubnets) !== null && _l !== void 0 ? _l : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                    securityGroup: (_m = props.securityGroupsOptions) === null || _m === void 0 ? void 0 : _m.database,
                },
                instances,
                backup: {
                    retention: (_p = (_o = props.backupOptions) === null || _o === void 0 ? void 0 : _o.databaseRetention) !== null && _p !== void 0 ? _p : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_r = (_q = props.removalPolicy) === null || _q === void 0 ? void 0 : _q.database) !== null && _r !== void 0 ? _r : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: (_s = props.vpcSubnets) !== null && _s !== void 0 ? _s : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            resourceSignalTimeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            updateType: aws_autoscaling_1.UpdateType.REPLACING_UPDATE,
            replacingUpdateMinSuccessfulInstancesPercent: 100,
            securityGroup: (_t = props.securityGroupsOptions) === null || _t === void 0 ? void 0 : _t.installer,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    onValidate() {
        const validationErrors = [];
        // Using the output of VersionQuery across stacks can cause issues. CloudFormation stack outputs cannot change if
        // a resource in another stack is referencing it.
        if (this.version instanceof version_query_1.VersionQuery) {
            const versionStack = core_1.Stack.of(this.version);
            const thisStack = core_1.Stack.of(this);
            if (versionStack != thisStack) {
                validationErrors.push('A VersionQuery can not be supplied from a different stack');
            }
        }
        return validationErrors;
    }
    /**
     * Configures an ECS Container Instance and Task Definition for deploying a Deadline Client that directly connects to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(props) {
        var _b, _c;
        const hostMountPoint = (_b = props.containerInstances.filesystemMountPoint) !== null && _b !== void 0 ? _b : '/mnt/repo';
        const majorVersion = core_1.Token.isUnresolved(this.version.majorVersion) ?
            core_1.Token.asString(this.version.majorVersion) : this.version.majorVersion.toString();
        const containerMountPoint = (_c = props.containers.filesystemMountPoint) !== null && _c !== void 0 ? _c : `/opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Note: pathToFileURL messes up CDK Tokens like the one in majorVersion
        const containerMountPointURL = props.containers.filesystemMountPoint ?
            url_1.pathToFileURL(props.containers.filesystemMountPoint).toString() :
            `file:///opt/Thinkbox/DeadlineRepository${majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: containerMountPointURL,
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * Configure a Deadline Client, that is running in an Amazon EC2 instance, for direct connection to this repository.
     *
     * This includes:
     *    - Ingress to database & filesystem Security Groups, as required.
     *    - IAM Permissions for database & filesystem, as required.
     *    - Mounts the Repository File System via UserData
     *    - Configures Deadline to direct-connect to the Repository.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(props) {
        var _b;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        var _b;
        const prefix = (_b = logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) !== null && _b !== void 0 ? _b : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = this.node.uniqueId;
        core_1.Tags.of(this.installerGroup).add(tagKey, this.node.uniqueId);
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl http://169.254.169.254/latest/meta-data/instance-id)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version) {
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: [
                `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
                `"${installPath}"`,
                version.linuxFullVersionString(),
            ],
        });
    }
}
exports.Repository = Repository;
_a = JSII_RTTI_SYMBOL_1;
Repository[_a] = { fqn: "aws-rfdk.deadline.Repository", version: "0.26.0" };
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
//# sourceMappingURL=data:application/json;base64,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