/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup } from '@aws-cdk/aws-autoscaling';
import { Connections, IConnectable, ISecurityGroup } from '@aws-cdk/aws-ec2';
import { Cluster } from '@aws-cdk/aws-ecs';
import { ApplicationLoadBalancer } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IGrantable, IPrincipal } from '@aws-cdk/aws-iam';
import { ISecret } from '@aws-cdk/aws-secretsmanager';
import { Construct, IConstruct } from '@aws-cdk/core';
import { ECSConnectOptions, InstanceConnectOptions, IVersion, RenderQueueProps } from '.';
import { ConnectableApplicationEndpoint } from '../../core';
/**
 * Interface for Deadline Render Queue.
 *
 * @stability stable
 */
export interface IRenderQueue extends IConstruct, IConnectable {
    /**
     * The endpoint used to connect to the Render Queue.
     *
     * @stability stable
     */
    readonly endpoint: ConnectableApplicationEndpoint;
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @returns An environment mapping that is used to configure the Docker Images
     * @stability stable
     */
    configureClientECS(params: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     */
    configureClientInstance(params: InstanceConnectOptions): void;
}
/**
 * Base class for Render Queue providers
 */
declare abstract class RenderQueueBase extends Construct implements IRenderQueue {
    /**
     * The endpoint that Deadline clients can use to connect to the Render Queue
     */
    abstract readonly endpoint: ConnectableApplicationEndpoint;
    /**
     * Allows specifying security group connections for the Render Queue.
     */
    abstract readonly connections: Connections;
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue
     * @returns An environment mapping that is used to configure the Docker Images
     */
    abstract configureClientECS(params: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue
     */
    abstract configureClientInstance(params: InstanceConnectOptions): void;
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * - An Amazon Elastic Container Service (ECS) cluster.
 * - An AWS EC2 auto-scaling group that provides the instances that host the ECS service.
 * - An ECS service with a task definition that deploys the Deadline Remote Connetion Server (RCS) in a container.
 * - A Amazon CloudWatch log group for streaming logs from the Deadline RCS.
 * - An application load balancer, listener and target group that balance incoming traffic among the RCS containers.
 *
 * Security Considerations
 * ------------------------
 * - The instances deployed by this construct download and run scripts from your CDK bootstrap bucket when that instance
 *    is launched. You must limit write access to your CDK bootstrap bucket to prevent an attacker from modifying the actions
 *    performed by these scripts. We strongly recommend that you either enable Amazon S3 server access logging on your CDK
 *    bootstrap bucket, or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Care must be taken to secure what can connect to the RenderQueue. The RenderQueue does not authenticate API
 *    requests made against it. You must limit access to the RenderQueue endpoint to only trusted hosts. Those hosts
 *    should be governed carefully, as malicious software could use the API to remotely execute code across the entire render farm.
 * - The RenderQueue can be deployed with network encryption through Transport Layer Security (TLS) or without it. Unencrypted
 *    network communications can be eavesdropped upon or modified in transit. We strongly recommend deploying the RenderQueue
 *    with TLS enabled in production environments.
 *
 * @stability stable
 */
export declare class RenderQueue extends RenderQueueBase implements IGrantable {
    /**
     * Container listening ports for each protocol.
     */
    private static readonly RCS_PROTO_PORTS;
    /**
     * Regular expression that validates a hostname (portion in front of the subdomain).
     */
    private static readonly RE_VALID_HOSTNAME;
    /**
     * The principal to grant permissions to.
     *
     * @stability stable
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The Amazon ECS cluster that is hosting the fleet of Deadline RCS applications.
     *
     * @stability stable
     */
    readonly cluster: Cluster;
    /**
     * Allows specifying security group connections for the Render Queue.
     *
     * @stability stable
     * @inheritdoc true
     */
    readonly connections: Connections;
    /**
     * The endpoint that Deadline clients can use to connect to the Render Queue.
     *
     * @stability stable
     * @inheritdoc true
     */
    readonly endpoint: ConnectableApplicationEndpoint;
    /**
     * The application load balancer that serves the traffic.
     *
     * @stability stable
     */
    readonly loadBalancer: ApplicationLoadBalancer;
    /**
     * The Amazon EC2 Auto Scaling Group within the {@link RenderQueue.cluster} that contains the Deadline RCS's instances.
     *
     * @stability stable
     */
    readonly asg: AutoScalingGroup;
    /**
     * The version of Deadline that the RenderQueue uses.
     *
     * @stability stable
     */
    readonly version: IVersion;
    /**
     * The secret containing the cert chain for external connections.
     *
     * @stability stable
     */
    readonly certChain?: ISecret;
    /**
     * Whether SEP policies have been added
     */
    private haveAddedSEPPolicies;
    /**
     * Whether Resource Tracker policies have been added
     */
    private haveAddedResourceTrackerPolicies;
    /**
     * The log group where the RCS container will log to
     */
    private readonly logGroup;
    /**
     * Instance of the Application Load Balanced EC2 service pattern.
     */
    private readonly pattern;
    /**
     * The certificate used by the ALB for external Traffic
     */
    private readonly clientCert?;
    /**
     * The connection object that contains the logic for how clients can connect to the Render Queue.
     */
    private readonly rqConnection;
    /**
     * The listener on the ALB that is redirecting traffic to the RCS.
     */
    private readonly listener;
    /**
     * The ECS task for the RCS.
     */
    private readonly taskDefinition;
    /**
     * Depend on this to ensure that ECS Service is stable.
     */
    private ecsServiceStabilized;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: RenderQueueProps);
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @stability stable
     */
    protected onValidate(): string[];
    /**
     * Configures an ECS cluster to be able to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientECS(param: ECSConnectOptions): {
        [name: string]: string;
    };
    /**
     * Configure an Instance/Autoscaling group to connect to a RenderQueue.
     *
     * @stability stable
     * @inheritdoc true
     */
    configureClientInstance(param: InstanceConnectOptions): void;
    /**
     * Adds AWS Managed Policies to the Render Queue so it is able to control Deadline's Spot Event Plugin.
     *
     * See: https://docs.thinkboxsoftware.com/products/deadline/10.1/1_User%20Manual/manual/event-spot.html for additonal information.
     *
     * @param includeResourceTracker Whether or not the Resource tracker admin policy should also be added (Default: True).
     * @stability stable
     */
    addSEPPolicies(includeResourceTracker?: boolean): void;
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     * @stability stable
     */
    addChildDependency(child: IConstruct): void;
    /**
     * Adds security groups to the frontend of the Render Queue, which is its load balancer.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addFrontendSecurityGroups(...securityGroups: ISecurityGroup[]): void;
    /**
     * Adds security groups to the backend components of the Render Queue, which consists of the AutoScalingGroup for the Deadline RCS.
     *
     * @param securityGroups The security groups to add.
     * @stability stable
     */
    addBackendSecurityGroups(...securityGroups: ISecurityGroup[]): void;
    private createTaskDefinition;
}
export {};
