"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const tag_helpers_1 = require("./tag-helpers");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MongoDbInstance', () => {
    let app;
    let stack;
    let vpc;
    let dnsZone;
    let caCert;
    let serverCert;
    const hostname = 'hostname';
    const zoneName = 'testZone';
    const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
    const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
    beforeEach(() => {
        app = new core_1.App();
        stack = new core_1.Stack(app, 'Stack');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
    });
    test('default mongodb instance is created correctly', () => {
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'r5.large',
            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                Ebs: assert_1.objectLike({
                    Encrypted: true,
                }),
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Route53::RecordSet', {
            Name: hostname + '.' + zoneName + '.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
            Description: `Admin credentials for the MongoDB database ${instance.node.uniqueId}`,
            GenerateSecretString: {
                ExcludeCharacters: '\"()$\'',
                ExcludePunctuation: true,
                GenerateStringKey: 'password',
                IncludeSpace: false,
                PasswordLength: 24,
                RequireEachIncludedType: true,
                SecretStringTemplate: '{\"username\":\"admin\"}',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            Tags: assert_1.arrayWith(assert_1.objectLike({
                Key: 'VolumeGrantAttach-dff922f1fb3c0287b3759d461a48c4b8',
                Value: '6238d22b121db8094cb816e2a49d2b61',
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: assert_1.objectLike({
                Statement: assert_1.arrayWith({
                    Action: [
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'MongoDbInstanceMongoDbInstanceLogGroupWrapperEAF733BB',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                }, {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.MONGODB_INSTANCE_3_6_SCRIPT.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.MONGODB_INSTANCE_3_6_SCRIPT.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Cert',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Key',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ServerCertPassphraseE4C3CB38',
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                }),
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/MongoDbInstance',
        }));
        const cloudInitLogPath = '/var/log/cloud-init-output.log';
        const cloudInitLogPrefix = 'cloud-init-output';
        const mongoLogPath = '/var/log/mongodb/mongod.log';
        const mongoLogPrefix = 'MongoDB';
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
            Description: 'config file for Repository logs config',
            Value: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('\",\"log_stream_name\":\"' + cloudInitLogPrefix + '-{instance_id}\",\"file_path\":\"' + cloudInitLogPath + '\",' +
                    '\"timezone\":\"Local\"},{\"log_group_name\":\"', '\",\"log_stream_name\":\"' + mongoLogPrefix + '-{instance_id}\",\"file_path\":\"' + mongoLogPath + '\"' +
                    ',\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}')),
            }),
        }));
        const userData = instance.userData.render();
        const token = '${Token[TOKEN.\\d+]}';
        // Make sure we add signal on exit
        const exitTrap = '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource MongoDbInstanceServerAsgASG2643AD1D --region ' + token +
            ' -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            'test \"${MONGO_SETUP_DIR} != \"\" && sudo umount \"${MONGO_SETUP_DIR}\n' +
            '}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(exitTrap)));
        const callExitTrap = 'trap exitTrap EXIT';
        expect(userData).toMatch(new RegExp(callExitTrap));
        const settings = 'set -xefuo pipefail';
        expect(userData).toMatch(new RegExp(settings));
        const createTempDir = 'mkdir -p $\\(dirname \'/tmp/' + token + token + '\'\\)\n';
        const s3Copy = 'aws s3 cp \'s3://' + token + '/' + token + token + '\' \'/tmp/' + token + token + '\'\n';
        // CloudWatch Agent
        const setE = 'set -e\n';
        const setChmod = 'chmod \\+x \'/tmp/' + token + token + '\'\n';
        const execute = '\'/tmp/' + token + token + '\' ' + token + '\n';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + setE + setChmod + execute)));
        // Make sure we mount EBS volume
        const mount = 'TMPDIR=$\\(mktemp -d\\)\n' +
            'pushd \"$TMPDIR\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'bash ./mountEbsBlockVolume.sh ' + token + ' xfs /var/lib/mongo rw \"\"\n' +
            'popd\n' +
            'rm -f /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + mount)));
        // install mongodb
        const bashCmd = 'bash /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + bashCmd)));
        // configureMongoDb
        const monogdbConfig = 'which mongod && test -f /etc/mongod.conf\n' +
            'sudo service mongod stop\n' +
            'MONGO_SETUP_DIR=$\\(mktemp -d\\)\n' +
            'mkdir -p \"${MONGO_SETUP_DIR}\"\n' +
            'sudo mount -t tmpfs -o size=50M tmpfs \"${MONGO_SETUP_DIR}\"\n' +
            'pushd \"${MONGO_SETUP_DIR}\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'cp /etc/mongod.conf .';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + monogdbConfig)));
        // Getting the server certificate
        const serverCertCmd = 'bash serverCertFromSecrets.sh \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(serverCertCmd)));
        // set mongodb certificates and credentials
        const monogdbCredentials = 'sudo mkdir -p /etc/mongod_certs\n' +
            'sudo mv ./ca.crt ./key.pem /etc/mongod_certs\n' +
            'sudo chown root.mongod -R /etc/mongod_certs/\n' +
            'sudo chmod 640 -R /etc/mongod_certs/\n' +
            'sudo chmod 750 /etc/mongod_certs/\n' +
            'sudo chown mongod.mongod -R /var/lib/mongo\n' +
            'bash ./setMongoLimits.sh\n' +
            'bash ./setStoragePath.sh \"/var/lib/mongo\"\n' +
            'bash ./setMongoNoAuth.sh\n' +
            'sudo service mongod start\n' +
            'bash ./setAdminCredentials.sh \"' + token + '\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(monogdbCredentials)));
        // Setup for live deployment, and start mongod
        const startMongo = 'sudo service mongod stop\n' +
            'bash ./setLiveConfiguration.sh\n' +
            'sudo systemctl enable mongod\n' +
            'sudo service mongod start\n' +
            'popd';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(startMongo)));
        // Make sure all the required public members are set
        expect(instance.version).toBe(version);
        expect(instance.connections).toBeDefined();
        expect(instance.connections).toBe(instance.server.connections);
        expect(instance.grantPrincipal).toBeDefined();
        expect(instance.grantPrincipal).toBe(instance.server.grantPrincipal);
        expect(instance.port).toBeDefined();
        expect(instance.role).toBeDefined();
        expect(instance.role).toBe(instance.server.role);
        expect(instance.userData).toBeDefined();
        expect(instance.userData).toBe(instance.server.userData);
        expect(instance.fullHostname).toBeDefined();
    });
    test('throw exception when no available subnets', () => {
        // GIVEN
        const invalidSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
            availabilityZones: ['dummy zone'],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
                vpcSubnets: invalidSubnets,
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('changing instance type works correctly', () => {
        // GIVEN
        const expectedInstanceType = 'm4.micro';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            instanceType: new aws_ec2_1.InstanceType(expectedInstanceType),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: expectedInstanceType,
        }));
    });
    test('allowing ssh connection with key name', () => {
        // GIVEN
        const expectedKeyName = 'someKeyName';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            keyName: expectedKeyName,
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            KeyName: expectedKeyName,
        }));
    });
    test('using custom admin user works correctly', () => {
        // GIVEN
        const expectedAdminUser = new aws_secretsmanager_1.Secret(stack, 'AdminUser', {
            description: 'Custom admin credentials for the MongoDB database',
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'test_password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                adminUser: expectedAdminUser,
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        expect(instance.adminUser).toBe(expectedAdminUser);
    });
    test('setting security group works correctly', () => {
        // GIVEN
        const actualSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'SecurityGroup', {
            securityGroupName: 'CustomSecurityGroup',
            vpc,
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            securityGroup: actualSecurityGroup,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
    });
    test('setting role works correctly', () => {
        // GIVEN
        const expectedRole = new aws_iam_1.Role(stack, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            roleName: 'CustomRole',
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            role: expectedRole,
        });
        // THEN
        expect(instance.server.role).toBe(expectedRole);
        expect(instance.role).toBe(expectedRole);
    });
    test('setting custom data volume works correctly', () => {
        // GIVEN
        const actualVolume = new aws_ec2_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: core_1.Size.gibibytes(50),
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volume: actualVolume,
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Volume', 1));
    });
    test('setting custom encryption key for data volume works correctly', () => {
        // GIVEN
        // KmsKeyId is Key961B73FD
        const actualEncryptionKey = new aws_kms_1.Key(stack, 'Key', {
            description: 'Key for testing',
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        encryptionKey: actualEncryptionKey,
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: assert_1.objectLike({
                'Fn::GetAtt': assert_1.arrayWith('Key961B73FD'),
            }),
        }));
    });
    test('setting custom size for data volume works correctly', () => {
        // GIVEN
        const volumeSize = 123;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        size: core_1.Size.gibibytes(volumeSize),
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Size: volumeSize,
        }));
    });
    test('setting LogGroup bucket name enables export to S3', () => {
        // GIVEN
        const bucketName = 'test-bucket';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                bucketName,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: assert_1.arrayWith(assert_1.objectLike({
                Input: '{\"BucketName\":\"' + bucketName + '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"/renderfarm/MongoDbInstance\",\"RetentionInHours\":72}',
            })),
        }));
    });
    test.each([
        'test-prefix/',
        '',
    ])('is created with correct LogGroup prefix %s', (testPrefix) => {
        // GIVEN
        const id = 'MongoDbInstance';
        // WHEN
        new lib_1.MongoDbInstance(stack, id, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix: testPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: testPrefix + id,
        }));
    });
    test('is created with correct LogGroup retention', () => {
        // GIVEN
        const retention = aws_logs_1.RetentionDays.ONE_DAY;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                retention,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: retention,
        }));
    });
    test('adds security group', () => {
        // GIVEN
        const securityGroup = new aws_ec2_1.SecurityGroup(stack, 'NewSecurityGroup', {
            vpc,
        });
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // WHEN
        instance.addSecurityGroup(securityGroup);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            SecurityGroups: assert_1.arrayWith(stack.resolve(securityGroup.securityGroupId)),
        }));
    });
    tag_helpers_1.testConstructTags({
        constructName: 'MongoDbInstance',
        createConstruct: () => {
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            new lib_1.MongoDbInstance(isolatedStack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
            });
            return isolatedStack;
        },
        resourceTypeCounts: {
            'AWS::EC2::SecurityGroup': 1,
            'AWS::IAM::Role': 1,
            'AWS::AutoScaling::AutoScalingGroup': 1,
            'AWS::EC2::NetworkInterface': 1,
            'AWS::SecretsManager::Secret': 1,
            'AWS::EC2::Volume': 1,
            'AWS::SSM::Parameter': 1,
        },
    });
});
//# sourceMappingURL=data:application/json;base64,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