"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends core_1.Construct {
    constructor(scope, id, props) {
        var _c;
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${this.node.uniqueId}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${this.node.uniqueId} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(this.node.uniqueId).digest('hex');
        this.uniqueTag = new core_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated. When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * - Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * - Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *    to only the principal(s) that require access.
 *
 * @stability stable
 */
class X509CertificatePem extends X509CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        if (((_c = props.validFor) !== null && _c !== void 0 ? _c : 1) < 1 && !core_1.Token.isUnresolved(props.validFor)) {
            core_1.Annotations.of(this).addError('Certificates must be valid for at least one day.');
        }
        (_d = props.signingCertificate) === null || _d === void 0 ? void 0 : _d.cert.grantRead(this.lambdaFunc);
        (_e = props.signingCertificate) === null || _e === void 0 ? void 0 : _e.key.grantRead(this.lambdaFunc);
        (_f = props.signingCertificate) === null || _f === void 0 ? void 0 : _f.passphrase.grantRead(this.lambdaFunc);
        (_h = (_g = props.signingCertificate) === null || _g === void 0 ? void 0 : _g.certChain) === null || _h === void 0 ? void 0 : _h.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: (_j = props.subject.o) !== null && _j !== void 0 ? _j : 'AWS',
                OU: (_k = props.subject.ou) !== null && _k !== void 0 ? _k : 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_l = props.encryptionKey) === null || _l === void 0 ? void 0 : _l.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
            CertificateValidFor: (_m = props.validFor) === null || _m === void 0 ? void 0 : _m.toString(),
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretArn: core_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretArn: core_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate.
     *
     * @stability stable
     */
    grantCertRead(grantee) {
        var _c;
        const result = this.cert.grantRead(grantee);
        (_c = this.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase.
     *
     * @stability stable
     */
    grantFullRead(grantee) {
        var _c;
        const result = this.cert.grantRead(grantee);
        (_c = this.certChain) === null || _c === void 0 ? void 0 : _c.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
_a = JSII_RTTI_SYMBOL_1;
X509CertificatePem[_a] = { fqn: "aws-rfdk.X509CertificatePem", version: "0.26.0" };
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly generated and stored in a Secret.
 *
 * Resources Deployed
 * ------------------------
 * - DynamoDB Table - Used for tracking resources created by the CustomResource.
 * - Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * - Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - Access to the AWS SecretsManager Secrets that are created by this construct should be tightly restricted
 *    to only the principal(s) that require access.
 *
 * @stability stable
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _c;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_c = props.encryptionKey) === null || _c === void 0 ? void 0 : _c.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
_b = JSII_RTTI_SYMBOL_1;
X509CertificatePkcs12[_b] = { fqn: "aws-rfdk.X509CertificatePkcs12", version: "0.26.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoieDUwOS1jZXJ0aWZpY2F0ZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIng1MDktY2VydGlmaWNhdGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7QUFBQTs7O0dBR0c7QUFFSCxpQ0FBaUM7QUFDakMsK0JBQTRCO0FBRTVCLHdEQUkrQjtBQUMvQiw4Q0FJMEI7QUFFMUIsb0RBSzZCO0FBQzdCLGdEQUFrRDtBQUNsRCxvRUFBOEQ7QUFDOUQsd0NBVXVCO0FBRXZCLGlGQUE0RDtBQTJHNUQsTUFBZSxtQkFBb0IsU0FBUSxnQkFBUztJQWFsRCxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQStCOztRQUN2RSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLElBQUksQ0FBQyxRQUFRLEdBQUcsSUFBSSxvQkFBSyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDdkMsWUFBWSxFQUFFLEVBQUUsSUFBSSxFQUFFLFlBQVksRUFBRSxJQUFJLEVBQUUsNEJBQWEsQ0FBQyxNQUFNLEVBQUU7WUFDaEUsT0FBTyxFQUFFLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLElBQUksRUFBRSw0QkFBYSxDQUFDLE1BQU0sRUFBRTtZQUMvRCxhQUFhLEVBQUUsb0JBQWEsQ0FBQyxPQUFPO1lBQ3BDLG9CQUFvQixFQUFFLElBQUk7WUFDMUIsV0FBVyxFQUFFLDBCQUFXLENBQUMsZUFBZTtTQUN6QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksMkJBQU0sQ0FBQyxJQUFJLEVBQUUsWUFBWSxFQUFFO1lBQy9DLFdBQVcsRUFBRSx5REFBeUQsSUFBSSxDQUFDLElBQUksQ0FBQyxRQUFRLEVBQUU7WUFDMUYsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQ2xDLG9CQUFvQixFQUFFO2dCQUNwQixpQkFBaUIsRUFBRSxRQUFRO2dCQUMzQixrQkFBa0IsRUFBRSxJQUFJO2dCQUN4QixZQUFZLEVBQUUsS0FBSztnQkFDbkIsY0FBYyxFQUFFLEVBQUU7Z0JBQ2xCLHVCQUF1QixFQUFFLElBQUk7YUFDOUI7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU0sR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sQ0FBQztRQUNyQyxNQUFNLGdCQUFnQixHQUFHLGFBQWEsQ0FBQztRQUN2QyxNQUFNLGdCQUFnQixHQUFRLDZCQUFJLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztRQUNyRCxNQUFNLGVBQWUsR0FBRyxnQkFBZ0IsQ0FBQyxNQUFNLENBQUMsQ0FBQztRQUNqRCxNQUFNLFlBQVksR0FBRyx5QkFBWSxDQUFDLG1CQUFtQixDQUFDLElBQUksRUFBRSxjQUFjLEVBQUUsZUFBZSxDQUFDLENBQUM7UUFFN0Y7OztXQUdHO1FBQ0gsSUFBSSxDQUFDLFVBQVUsR0FBRyxJQUFJLHFCQUFjLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUN0RCxXQUFXLEVBQUUsNkJBQTZCLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSw0QkFBNEI7WUFDeEYsSUFBSSxFQUFFLEtBQUssQ0FBQyxVQUFVO1lBQ3RCLFdBQVcsRUFBRTtnQkFDWCxRQUFRLEVBQUUsSUFBSSxDQUFDLFFBQVEsQ0FBQyxTQUFTO2dCQUNqQyxLQUFLLEVBQUUsT0FBTzthQUNmO1lBQ0QsT0FBTyxFQUFFLG9CQUFPLENBQUMsV0FBVztZQUM1QixNQUFNLEVBQUUsQ0FBRSxZQUFZLENBQUU7WUFDeEIsT0FBTyxFQUFFLEtBQUssQ0FBQyxhQUFhO1lBQzVCLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixZQUFZLEVBQUUsd0JBQWEsQ0FBQyxRQUFRO1NBQ3JDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyxDQUFDO1FBQ2xELElBQUksQ0FBQyxRQUFRLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUUsd0JBQXdCLENBQUMsQ0FBQztRQUMvRCxNQUFBLEtBQUssQ0FBQyxhQUFhLDBDQUFFLFlBQVksQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFO1FBQ25ELElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUUzQyxNQUFNLFdBQVcsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLEtBQUssQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN0RixJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksVUFBRyxDQUN0QixtQkFBbUIsV0FBVyxDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLEVBQUUsRUFDMUQsV0FBVyxDQUNaLENBQUM7UUFDRixNQUFNLFlBQVksR0FBMkIsRUFBRSxDQUFDO1FBQ2hELFlBQVksQ0FBQyw4QkFBOEIsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHLEVBQUUsQ0FBQyxHQUFHLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDO1FBRXhGLElBQUksQ0FBQyxVQUFVLENBQUMsZUFBZSxDQUFDLElBQUkseUJBQWUsQ0FBQztZQUNsRCxPQUFPLEVBQUU7Z0JBQ1AsNkJBQTZCO2dCQUM3Qiw2QkFBNkI7Z0JBQzdCLDRCQUE0QjtnQkFDNUIsK0JBQStCO2FBQ2hDO1lBQ0QsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO1lBQ2hCLFVBQVUsRUFBRTtnQkFDVixZQUFZLEVBQUUsWUFBWTthQUMzQjtTQUNGLENBQUMsQ0FBQyxDQUFDO0lBQ04sQ0FBQztDQUNGOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUE4QkQsTUFBYSxrQkFBbUIsU0FBUSxtQkFBbUI7Ozs7SUFLekQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUE4Qjs7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixVQUFVLEVBQUUsaUJBQUksQ0FBQyxTQUFTLENBQUMsV0FBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUM1RSxhQUFhLEVBQUUsMkJBQTJCO1lBQzFDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtTQUNuQyxDQUFDLENBQUM7UUFFSCxJQUFJLE9BQUMsS0FBSyxDQUFDLFFBQVEsbUNBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsWUFBSyxDQUFDLFlBQVksQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDLEVBQUU7WUFDcEUsa0JBQVcsQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxDQUFDLGtEQUFrRCxDQUFDLENBQUM7U0FDbkY7UUFFRCxNQUFBLEtBQUssQ0FBQyxrQkFBa0IsMENBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsVUFBVSxFQUFFO1FBQzFELE1BQUEsS0FBSyxDQUFDLGtCQUFrQiwwQ0FBRSxHQUFHLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxVQUFVLEVBQUU7UUFDekQsTUFBQSxLQUFLLENBQUMsa0JBQWtCLDBDQUFFLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRTtRQUNoRSxZQUFBLEtBQUssQ0FBQyxrQkFBa0IsMENBQUUsU0FBUywwQ0FBRSxTQUFTLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRTtRQUVoRSxNQUFNLGtCQUFrQixHQUFHLEtBQUssQ0FBQyxrQkFBa0I7WUFDakQsQ0FBQyxDQUFDO2dCQUNBLElBQUksRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLFNBQVM7Z0JBQzdDLEdBQUcsRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsR0FBRyxDQUFDLFNBQVM7Z0JBQzNDLFVBQVUsRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsVUFBVSxDQUFDLFNBQVM7Z0JBQ3pELFNBQVMsRUFBRSxLQUFLLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRTthQUNsRztZQUNELENBQUMsQ0FBQyxTQUFTLENBQUM7UUFDZCxNQUFNLFVBQVUsR0FBNkI7WUFDM0MsaUJBQWlCLEVBQUU7Z0JBQ2pCLEVBQUUsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLEVBQUU7Z0JBQ3BCLENBQUMsUUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsbUNBQUksS0FBSztnQkFDM0IsRUFBRSxRQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsRUFBRSxtQ0FBSSxVQUFVO2FBQ25DO1lBQ0QsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUztZQUNyQyxNQUFNLEVBQUU7Z0JBQ04sVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSTtnQkFDMUIsV0FBVyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSTtnQkFDM0IsYUFBYSxRQUFFLEtBQUssQ0FBQyxhQUFhLDBDQUFFLE1BQU07Z0JBQzFDLElBQUksRUFBRTtvQkFDSjt3QkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHO3dCQUN2QixLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLO3FCQUM1QjtpQkFDRjthQUNGO1lBQ0Qsa0JBQWtCLEVBQUUsa0JBQWtCO1lBQ3RDLG1CQUFtQixRQUFFLEtBQUssQ0FBQyxRQUFRLDBDQUFFLFFBQVEsRUFBRTtTQUNoRCxDQUFDO1FBQ0YsTUFBTSxRQUFRLEdBQUcsSUFBSSxxQkFBYyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDbkQsWUFBWSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVztZQUN6QyxVQUFVO1lBQ1YsWUFBWSxFQUFFLDRCQUE0QjtTQUMzQyxDQUFDLENBQUM7UUFDSCxJQUFJLElBQUksQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFO1lBQ3hCLDBGQUEwRjtZQUMxRixRQUFRLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQ25EO1FBRUQsSUFBSSxDQUFDLElBQUksR0FBRywyQkFBTSxDQUFDLG9CQUFvQixDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDcEQsU0FBUyxFQUFFLFlBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQztZQUNsRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7U0FDbkMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEdBQUcsR0FBRywyQkFBTSxDQUFDLG9CQUFvQixDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7WUFDbEQsU0FBUyxFQUFFLFlBQUssQ0FBQyxRQUFRLENBQUMsUUFBUSxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQztZQUNqRCxhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7U0FDbkMsQ0FBQyxDQUFDO1FBQ0gsa0hBQWtIO1FBQ2xILDBEQUEwRDtRQUMxRCxJQUFJLGtCQUFrQixFQUFFO1lBQ3RCLE1BQU0saUJBQWlCLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxXQUFXLENBQUMsQ0FBQztZQUN2RCxJQUFJLENBQUMsU0FBUyxHQUFHLGlCQUFpQjtnQkFDaEMsQ0FBQyxDQUFDLDJCQUFNLENBQUMsb0JBQW9CLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtvQkFDL0MsU0FBUyxFQUFFLFlBQUssQ0FBQyxRQUFRLENBQUMsaUJBQWlCLENBQUM7b0JBQzVDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtpQkFDbkMsQ0FBQztnQkFDRixDQUFDLENBQUMsU0FBUyxDQUFDO1NBQ2Y7SUFDSCxDQUFDOzs7Ozs7SUFLTSxhQUFhLENBQUMsT0FBbUI7O1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzVDLE1BQUEsSUFBSSxDQUFDLFNBQVMsMENBQUUsU0FBUyxDQUFDLE9BQU8sRUFBRTtRQUNuQyxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDOzs7Ozs7SUFLTSxhQUFhLENBQUMsT0FBbUI7O1FBQ3RDLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxDQUFDO1FBQzVDLE1BQUEsSUFBSSxDQUFDLFNBQVMsMENBQUUsU0FBUyxDQUFDLE9BQU8sRUFBRTtRQUNuQyxJQUFJLENBQUMsR0FBRyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUM1QixJQUFJLENBQUMsVUFBVSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNuQyxPQUFPLE1BQU0sQ0FBQztJQUNoQixDQUFDOztBQW5HSCxnREFvR0M7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7QUF5REQsTUFBYSxxQkFBc0IsU0FBUSxtQkFBbUI7Ozs7SUFJNUQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQzs7UUFDekUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixVQUFVLEVBQUUsaUJBQUksQ0FBQyxTQUFTLENBQUMsV0FBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxRQUFRLENBQUMsQ0FBQztZQUM1RSxhQUFhLEVBQUUsMEJBQTBCO1lBQ3pDLGFBQWEsRUFBRSxLQUFLLENBQUMsYUFBYTtTQUNuQyxDQUFDLENBQUM7UUFFSCxLQUFLLENBQUMsaUJBQWlCLENBQUMsYUFBYSxDQUFDLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQztRQUV2RCxNQUFNLFVBQVUsR0FBaUM7WUFDL0MsVUFBVSxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsU0FBUztZQUNyQyxNQUFNLEVBQUU7Z0JBQ04sVUFBVSxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSTtnQkFDMUIsV0FBVyxFQUFFLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSTtnQkFDM0IsYUFBYSxRQUFFLEtBQUssQ0FBQyxhQUFhLDBDQUFFLE1BQU07Z0JBQzFDLElBQUksRUFBRTtvQkFDSjt3QkFDRSxHQUFHLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxHQUFHO3dCQUN2QixLQUFLLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLO3FCQUM1QjtpQkFDRjthQUNGO1lBQ0QsV0FBVyxFQUFFO2dCQUNYLElBQUksRUFBRSxLQUFLLENBQUMsaUJBQWlCLENBQUMsSUFBSSxDQUFDLFNBQVM7Z0JBQzVDLFNBQVMsRUFBRSxLQUFLLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsU0FBUyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsRUFBRTtnQkFDL0YsR0FBRyxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLENBQUMsU0FBUztnQkFDMUMsVUFBVSxFQUFFLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxVQUFVLENBQUMsU0FBUzthQUN6RDtTQUNGLENBQUM7UUFFRixNQUFNLFFBQVEsR0FBRyxJQUFJLHFCQUFjLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUNuRCxZQUFZLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXO1lBQ3pDLFVBQVU7WUFDVixZQUFZLEVBQUUsMEJBQTBCO1NBQ3pDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxJQUFJLEdBQUcsMkJBQU0sQ0FBQyxvQkFBb0IsQ0FBQyxJQUFJLEVBQUUsTUFBTSxFQUFFO1lBQ3BELFNBQVMsRUFBRSxZQUFLLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDbEQsYUFBYSxFQUFFLEtBQUssQ0FBQyxhQUFhO1NBQ25DLENBQUMsQ0FBQztJQUNMLENBQUM7O0FBNUNILHNEQTZDQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogQ29weXJpZ2h0IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogQXBhY2hlLTIuMFxuICovXG5cbmltcG9ydCAqIGFzIGNyeXB0byBmcm9tICdjcnlwdG8nO1xuaW1wb3J0IHsgam9pbiB9IGZyb20gJ3BhdGgnO1xuXG5pbXBvcnQge1xuICBBdHRyaWJ1dGVUeXBlLFxuICBCaWxsaW5nTW9kZSxcbiAgVGFibGUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1keW5hbW9kYic7XG5pbXBvcnQge1xuICBHcmFudCxcbiAgSUdyYW50YWJsZSxcbiAgUG9saWN5U3RhdGVtZW50LFxufSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IElLZXkgfSBmcm9tICdAYXdzLWNkay9hd3Mta21zJztcbmltcG9ydCB7XG4gIENvZGUsXG4gIEZ1bmN0aW9uIGFzIExhbWJkYUZ1bmN0aW9uLFxuICBMYXllclZlcnNpb24sXG4gIFJ1bnRpbWUsXG59IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgUmV0ZW50aW9uRGF5cyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCB7IElTZWNyZXQsIFNlY3JldCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQge1xuICBBbm5vdGF0aW9ucyxcbiAgQ29uc3RydWN0LFxuICBDdXN0b21SZXNvdXJjZSxcbiAgRHVyYXRpb24sXG4gIElDb25zdHJ1Y3QsXG4gIFJlbW92YWxQb2xpY3ksXG4gIFN0YWNrLFxuICBUYWcsXG4gIFRva2VuLFxufSBmcm9tICdAYXdzLWNkay9jb3JlJztcblxuaW1wb3J0IHsgQVJOUyB9IGZyb20gJy4uLy4uL2xhbWJkYXMvbGFtYmRhTGF5ZXJWZXJzaW9uQXJucyc7XG5pbXBvcnQgeyBJWDUwOUNlcnRpZmljYXRlRW5jb2RlUGtjczEyLCBJWDUwOUNlcnRpZmljYXRlR2VuZXJhdGUgfSBmcm9tICcuLi8uLi9sYW1iZGFzL25vZGVqcy94NTA5LWNlcnRpZmljYXRlJztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBEaXN0aW5ndWlzaGVkTmFtZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjbjogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgbz86IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgb3U/OiBzdHJpbmc7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgWDUwOUNlcnRpZmljYXRlUGVtUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdWJqZWN0OiBEaXN0aW5ndWlzaGVkTmFtZTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzaWduaW5nQ2VydGlmaWNhdGU/OiBYNTA5Q2VydGlmaWNhdGVQZW07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdmFsaWRGb3I/OiBudW1iZXI7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVg1MDlDZXJ0aWZpY2F0ZVBlbSBleHRlbmRzIElDb25zdHJ1Y3Qge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgY2VydDogSVNlY3JldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGtleTogSVNlY3JldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcGFzc3BocmFzZTogSVNlY3JldDtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNlcnRDaGFpbj86IElTZWNyZXQ7XG59XG5cbmludGVyZmFjZSBYNTA5Q2VydGlmaWNhdGVCYXNlUHJvcHMge1xuICByZWFkb25seSBsYW1iZGFDb2RlOiBDb2RlO1xuICByZWFkb25seSBsYW1iZGFIYW5kbGVyOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGVuY3J5cHRpb25LZXk/OiBJS2V5O1xufVxuXG5hYnN0cmFjdCBjbGFzcyBYNTA5Q2VydGlmaWNhdGVCYXNlIGV4dGVuZHMgQ29uc3RydWN0IHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IHBhc3NwaHJhc2U6IElTZWNyZXQ7XG5cbiAgLy8gVGhlIER5bmFtb0RCIFRhYmxlIHRoYXQgaXMgdXNlZCBhcyBhIGJhY2tpbmcgc3RvcmUgZm9yIHRoZSBDdXN0b21SZXNvdXJjZSB1dGlsaXplZCBpbiB0aGlzIGNvbnN0cnVjdC5cbiAgcHJvdGVjdGVkIGRhdGFiYXNlOiBUYWJsZTtcbiAgLy8gVGhlIExhbWJkYSB0aGF0IGJhY2tzIHRoZSBDdXN0b21SZXNvdXJjZS5cbiAgcHJvdGVjdGVkIGxhbWJkYUZ1bmM6IExhbWJkYUZ1bmN0aW9uO1xuXG4gIHByb3RlY3RlZCB1bmlxdWVUYWc6IFRhZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogWDUwOUNlcnRpZmljYXRlQmFzZVByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuZGF0YWJhc2UgPSBuZXcgVGFibGUodGhpcywgJ1RhYmxlJywge1xuICAgICAgcGFydGl0aW9uS2V5OiB7IG5hbWU6ICdQaHlzaWNhbElkJywgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkcgfSxcbiAgICAgIHNvcnRLZXk6IHsgbmFtZTogJ0N1c3RvbVJlc291cmNlJywgdHlwZTogQXR0cmlidXRlVHlwZS5TVFJJTkcgfSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICAgIHNlcnZlclNpZGVFbmNyeXB0aW9uOiB0cnVlLFxuICAgICAgYmlsbGluZ01vZGU6IEJpbGxpbmdNb2RlLlBBWV9QRVJfUkVRVUVTVCxcbiAgICB9KTtcblxuICAgIHRoaXMucGFzc3BocmFzZSA9IG5ldyBTZWNyZXQodGhpcywgJ1Bhc3NwaHJhc2UnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogYFBhc3NwaHJhc2UgZm9yIHRoZSBwcml2YXRlIGtleSBvZiB0aGUgWDUwOUNlcnRpZmljYXRlICR7dGhpcy5ub2RlLnVuaXF1ZUlkfWAsXG4gICAgICBlbmNyeXB0aW9uS2V5OiBwcm9wcy5lbmNyeXB0aW9uS2V5LFxuICAgICAgZ2VuZXJhdGVTZWNyZXRTdHJpbmc6IHtcbiAgICAgICAgZXhjbHVkZUNoYXJhY3RlcnM6ICdcIigpJFxcJycsIC8vIEV4Y2x1ZGUgY2hhcmFjdGVycyB0aGF0IG1pZ2h0IGludGVyYWN0IHdpdGggY29tbWFuZCBzaGVsbHMuXG4gICAgICAgIGV4Y2x1ZGVQdW5jdHVhdGlvbjogdHJ1ZSxcbiAgICAgICAgaW5jbHVkZVNwYWNlOiBmYWxzZSxcbiAgICAgICAgcGFzc3dvcmRMZW5ndGg6IDI0LFxuICAgICAgICByZXF1aXJlRWFjaEluY2x1ZGVkVHlwZTogdHJ1ZSxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCByZWdpb24gPSBTdGFjay5vZih0aGlzKS5yZWdpb247XG4gICAgY29uc3Qgb3BlblNzbExheWVyTmFtZSA9ICdvcGVuc3NsLWFsMic7XG4gICAgY29uc3Qgb3BlblNzbExheWVyQXJuczogYW55ID0gQVJOU1tvcGVuU3NsTGF5ZXJOYW1lXTtcbiAgICBjb25zdCBvcGVuU3NsTGF5ZXJBcm4gPSBvcGVuU3NsTGF5ZXJBcm5zW3JlZ2lvbl07XG4gICAgY29uc3Qgb3BlblNzbExheWVyID0gTGF5ZXJWZXJzaW9uLmZyb21MYXllclZlcnNpb25Bcm4odGhpcywgJ09wZW5Tc2xMYXllcicsIG9wZW5Tc2xMYXllckFybik7XG5cbiAgICAvKlxuICAgICAqIFdlIGNhbm5vdCBtYWtlIHRoaXMgYSBzaW5nbGV0b24gZnVuY3Rpb247IGRvaW5nIHNvIHdvdWxkIGNyZWF0ZSBjaXJjdWxhciByZWZlcmVuY2VzIGluIHRoZSBsYW1iZGEgcm9sZSAodG8gc2lnblxuICAgICAqIGEgY2VydCB3ZSBuZWVkIGEgY2VydCB0aGF0IHRoaXMgbGFtYmRhIGdlbmVyYXRlZCkuXG4gICAgICovXG4gICAgdGhpcy5sYW1iZGFGdW5jID0gbmV3IExhbWJkYUZ1bmN0aW9uKHRoaXMsICdHZW5lcmF0b3InLCB7XG4gICAgICBkZXNjcmlwdGlvbjogYFVzZWQgYnkgYSBYNTA5Q2VydGlmaWNhdGUgJHt0aGlzLm5vZGUudW5pcXVlSWR9IHRvIGdlbmVyYXRlIGNlcnRpZmljYXRlcy5gLFxuICAgICAgY29kZTogcHJvcHMubGFtYmRhQ29kZSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIERBVEFCQVNFOiB0aGlzLmRhdGFiYXNlLnRhYmxlTmFtZSxcbiAgICAgICAgREVCVUc6ICdmYWxzZScsXG4gICAgICB9LFxuICAgICAgcnVudGltZTogUnVudGltZS5OT0RFSlNfMTJfWCxcbiAgICAgIGxheWVyczogWyBvcGVuU3NsTGF5ZXIgXSxcbiAgICAgIGhhbmRsZXI6IHByb3BzLmxhbWJkYUhhbmRsZXIsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKDMwKSxcbiAgICAgIGxvZ1JldGVudGlvbjogUmV0ZW50aW9uRGF5cy5PTkVfV0VFSyxcbiAgICB9KTtcbiAgICB0aGlzLmRhdGFiYXNlLmdyYW50UmVhZFdyaXRlRGF0YSh0aGlzLmxhbWJkYUZ1bmMpO1xuICAgIHRoaXMuZGF0YWJhc2UuZ3JhbnQodGhpcy5sYW1iZGFGdW5jLCAnZHluYW1vZGI6RGVzY3JpYmVUYWJsZScpO1xuICAgIHByb3BzLmVuY3J5cHRpb25LZXk/LmdyYW50RW5jcnlwdCh0aGlzLmxhbWJkYUZ1bmMpO1xuICAgIHRoaXMucGFzc3BocmFzZS5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jKTtcblxuICAgIGNvbnN0IHVuaXF1ZVZhbHVlID0gY3J5cHRvLmNyZWF0ZUhhc2goJ21kNScpLnVwZGF0ZSh0aGlzLm5vZGUudW5pcXVlSWQpLmRpZ2VzdCgnaGV4Jyk7XG4gICAgdGhpcy51bmlxdWVUYWcgPSBuZXcgVGFnKFxuICAgICAgYFg1MDlTZWNyZXRHcmFudC0ke3VuaXF1ZVZhbHVlLnNsaWNlKDAsIDgpLnRvVXBwZXJDYXNlKCl9YCxcbiAgICAgIHVuaXF1ZVZhbHVlLFxuICAgICk7XG4gICAgY29uc3QgdGFnQ29uZGl0aW9uOiB7IFtrZXk6IHN0cmluZ106IGFueSB9ID0ge307XG4gICAgdGFnQ29uZGl0aW9uW2BzZWNyZXRzbWFuYWdlcjpSZXNvdXJjZVRhZy8ke3RoaXMudW5pcXVlVGFnLmtleX1gXSA9IHRoaXMudW5pcXVlVGFnLnZhbHVlO1xuXG4gICAgdGhpcy5sYW1iZGFGdW5jLmFkZFRvUm9sZVBvbGljeShuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3NlY3JldHNtYW5hZ2VyOkNyZWF0ZVNlY3JldCcsXG4gICAgICAgICdzZWNyZXRzbWFuYWdlcjpEZWxldGVTZWNyZXQnLFxuICAgICAgICAnc2VjcmV0c21hbmFnZXI6VGFnUmVzb3VyY2UnLFxuICAgICAgICAnc2VjcmV0c21hbmFnZXI6UHV0U2VjcmV0VmFsdWUnLFxuICAgICAgXSxcbiAgICAgIHJlc291cmNlczogWycqJ10sXG4gICAgICBjb25kaXRpb25zOiB7XG4gICAgICAgIFN0cmluZ0VxdWFsczogdGFnQ29uZGl0aW9uLFxuICAgICAgfSxcbiAgICB9KSk7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFg1MDlDZXJ0aWZpY2F0ZVBlbSBleHRlbmRzIFg1MDlDZXJ0aWZpY2F0ZUJhc2UgaW1wbGVtZW50cyBJWDUwOUNlcnRpZmljYXRlUGVtIHtcbiAgcHVibGljIHJlYWRvbmx5IGNlcnQ6IElTZWNyZXQ7XG4gIHB1YmxpYyByZWFkb25seSBrZXk6IElTZWNyZXQ7XG4gIHB1YmxpYyByZWFkb25seSBjZXJ0Q2hhaW4/OiBJU2VjcmV0O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBYNTA5Q2VydGlmaWNhdGVQZW1Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgbGFtYmRhQ29kZTogQ29kZS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdsYW1iZGFzJywgJ25vZGVqcycpKSxcbiAgICAgIGxhbWJkYUhhbmRsZXI6ICd4NTA5LWNlcnRpZmljYXRlLmdlbmVyYXRlJyxcbiAgICAgIGVuY3J5cHRpb25LZXk6IHByb3BzLmVuY3J5cHRpb25LZXksXG4gICAgfSk7XG5cbiAgICBpZiAoKHByb3BzLnZhbGlkRm9yID8/IDEpIDwgMSAmJiAhVG9rZW4uaXNVbnJlc29sdmVkKHByb3BzLnZhbGlkRm9yKSkge1xuICAgICAgQW5ub3RhdGlvbnMub2YodGhpcykuYWRkRXJyb3IoJ0NlcnRpZmljYXRlcyBtdXN0IGJlIHZhbGlkIGZvciBhdCBsZWFzdCBvbmUgZGF5LicpO1xuICAgIH1cblxuICAgIHByb3BzLnNpZ25pbmdDZXJ0aWZpY2F0ZT8uY2VydC5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jKTtcbiAgICBwcm9wcy5zaWduaW5nQ2VydGlmaWNhdGU/LmtleS5ncmFudFJlYWQodGhpcy5sYW1iZGFGdW5jKTtcbiAgICBwcm9wcy5zaWduaW5nQ2VydGlmaWNhdGU/LnBhc3NwaHJhc2UuZ3JhbnRSZWFkKHRoaXMubGFtYmRhRnVuYyk7XG4gICAgcHJvcHMuc2lnbmluZ0NlcnRpZmljYXRlPy5jZXJ0Q2hhaW4/LmdyYW50UmVhZCh0aGlzLmxhbWJkYUZ1bmMpO1xuXG4gICAgY29uc3Qgc2lnbmluZ0NlcnRpZmljYXRlID0gcHJvcHMuc2lnbmluZ0NlcnRpZmljYXRlXG4gICAgICA/IHtcbiAgICAgICAgQ2VydDogcHJvcHMuc2lnbmluZ0NlcnRpZmljYXRlLmNlcnQuc2VjcmV0QXJuLFxuICAgICAgICBLZXk6IHByb3BzLnNpZ25pbmdDZXJ0aWZpY2F0ZS5rZXkuc2VjcmV0QXJuLFxuICAgICAgICBQYXNzcGhyYXNlOiBwcm9wcy5zaWduaW5nQ2VydGlmaWNhdGUucGFzc3BocmFzZS5zZWNyZXRBcm4sXG4gICAgICAgIENlcnRDaGFpbjogcHJvcHMuc2lnbmluZ0NlcnRpZmljYXRlLmNlcnRDaGFpbiA/IHByb3BzLnNpZ25pbmdDZXJ0aWZpY2F0ZS5jZXJ0Q2hhaW4uc2VjcmV0QXJuIDogJycsXG4gICAgICB9XG4gICAgICA6IHVuZGVmaW5lZDtcbiAgICBjb25zdCBwcm9wZXJ0aWVzOiBJWDUwOUNlcnRpZmljYXRlR2VuZXJhdGUgPSB7XG4gICAgICBEaXN0aW5ndWlzaGVkTmFtZToge1xuICAgICAgICBDTjogcHJvcHMuc3ViamVjdC5jbixcbiAgICAgICAgTzogcHJvcHMuc3ViamVjdC5vID8/ICdBV1MnLFxuICAgICAgICBPVTogcHJvcHMuc3ViamVjdC5vdSA/PyAnVGhpbmtib3gnLFxuICAgICAgfSxcbiAgICAgIFBhc3NwaHJhc2U6IHRoaXMucGFzc3BocmFzZS5zZWNyZXRBcm4sXG4gICAgICBTZWNyZXQ6IHtcbiAgICAgICAgTmFtZVByZWZpeDogdGhpcy5ub2RlLnBhdGgsXG4gICAgICAgIERlc2NyaXB0aW9uOiB0aGlzLm5vZGUucGF0aCxcbiAgICAgICAgRW5jcnlwdGlvbktleTogcHJvcHMuZW5jcnlwdGlvbktleT8ua2V5QXJuLFxuICAgICAgICBUYWdzOiBbXG4gICAgICAgICAge1xuICAgICAgICAgICAgS2V5OiB0aGlzLnVuaXF1ZVRhZy5rZXksXG4gICAgICAgICAgICBWYWx1ZTogdGhpcy51bmlxdWVUYWcudmFsdWUsXG4gICAgICAgICAgfSxcbiAgICAgICAgXSxcbiAgICAgIH0sXG4gICAgICBTaWduaW5nQ2VydGlmaWNhdGU6IHNpZ25pbmdDZXJ0aWZpY2F0ZSxcbiAgICAgIENlcnRpZmljYXRlVmFsaWRGb3I6IHByb3BzLnZhbGlkRm9yPy50b1N0cmluZygpLFxuICAgIH07XG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0RlZmF1bHQnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IHRoaXMubGFtYmRhRnVuYy5mdW5jdGlvbkFybixcbiAgICAgIHByb3BlcnRpZXMsXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OlJGREtfWDUwOUdlbmVyYXRvcicsXG4gICAgfSk7XG4gICAgaWYgKHRoaXMubGFtYmRhRnVuYy5yb2xlKSB7XG4gICAgICAvLyBUaGVyZSdzIGEgcmFjZSBvbiB1cGRhdGUgd2hlcmUgdGhpcyByZXNvdXJjZSBjb3VsZCBleGVjdXRlIGJlZm9yZSB0aGUgcm9sZSBoYXMgdXBkYXRlZC5cbiAgICAgIHJlc291cmNlLm5vZGUuYWRkRGVwZW5kZW5jeSh0aGlzLmxhbWJkYUZ1bmMucm9sZSk7XG4gICAgfVxuXG4gICAgdGhpcy5jZXJ0ID0gU2VjcmV0LmZyb21TZWNyZXRBdHRyaWJ1dGVzKHRoaXMsICdDZXJ0Jywge1xuICAgICAgc2VjcmV0QXJuOiBUb2tlbi5hc1N0cmluZyhyZXNvdXJjZS5nZXRBdHQoJ0NlcnQnKSksXG4gICAgICBlbmNyeXB0aW9uS2V5OiBwcm9wcy5lbmNyeXB0aW9uS2V5LFxuICAgIH0pO1xuICAgIHRoaXMua2V5ID0gU2VjcmV0LmZyb21TZWNyZXRBdHRyaWJ1dGVzKHRoaXMsICdLZXknLCB7XG4gICAgICBzZWNyZXRBcm46IFRva2VuLmFzU3RyaW5nKHJlc291cmNlLmdldEF0dCgnS2V5JykpLFxuICAgICAgZW5jcnlwdGlvbktleTogcHJvcHMuZW5jcnlwdGlvbktleSxcbiAgICB9KTtcbiAgICAvLyBXZSdsbCBvbmx5IGhhdmUgYSBjaGFpbiBpZiB3ZSB1c2VkIGEgY2EgdG8gc2lnbiB0aGlzIGNlcnQuIFdlIGNhbm5vdCBjaGVjayBmb3IgY2VydENoYWluUmVzb3VyY2UgYmVpbmcgYW4gZW1wdHlcbiAgICAvLyBzdHJpbmcgYmVjYXVzZSBpdCBpcyBhbiB1bnJlc29sdmVkIHRva2VuIGF0IHRoaXMgcG9pbnQuXG4gICAgaWYgKHNpZ25pbmdDZXJ0aWZpY2F0ZSkge1xuICAgICAgY29uc3QgY2VydENoYWluUmVzb3VyY2UgPSByZXNvdXJjZS5nZXRBdHQoJ0NlcnRDaGFpbicpO1xuICAgICAgdGhpcy5jZXJ0Q2hhaW4gPSBjZXJ0Q2hhaW5SZXNvdXJjZVxuICAgICAgICA/IFNlY3JldC5mcm9tU2VjcmV0QXR0cmlidXRlcyh0aGlzLCAnQ2VydENoYWluJywge1xuICAgICAgICAgIHNlY3JldEFybjogVG9rZW4uYXNTdHJpbmcoY2VydENoYWluUmVzb3VyY2UpLFxuICAgICAgICAgIGVuY3J5cHRpb25LZXk6IHByb3BzLmVuY3J5cHRpb25LZXksXG4gICAgICAgIH0pXG4gICAgICAgIDogdW5kZWZpbmVkO1xuICAgIH1cbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBwdWJsaWMgZ3JhbnRDZXJ0UmVhZChncmFudGVlOiBJR3JhbnRhYmxlKTogR3JhbnQge1xuICAgIGNvbnN0IHJlc3VsdCA9IHRoaXMuY2VydC5ncmFudFJlYWQoZ3JhbnRlZSk7XG4gICAgdGhpcy5jZXJ0Q2hhaW4/LmdyYW50UmVhZChncmFudGVlKTtcbiAgICByZXR1cm4gcmVzdWx0O1xuICB9XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBncmFudEZ1bGxSZWFkKGdyYW50ZWU6IElHcmFudGFibGUpOiBHcmFudCB7XG4gICAgY29uc3QgcmVzdWx0ID0gdGhpcy5jZXJ0LmdyYW50UmVhZChncmFudGVlKTtcbiAgICB0aGlzLmNlcnRDaGFpbj8uZ3JhbnRSZWFkKGdyYW50ZWUpO1xuICAgIHRoaXMua2V5LmdyYW50UmVhZChncmFudGVlKTtcbiAgICB0aGlzLnBhc3NwaHJhc2UuZ3JhbnRSZWFkKGdyYW50ZWUpO1xuICAgIHJldHVybiByZXN1bHQ7XG4gIH1cbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgWDUwOUNlcnRpZmljYXRlUGtjczEyUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzb3VyY2VDZXJ0aWZpY2F0ZTogWDUwOUNlcnRpZmljYXRlUGVtO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgZW5jcnlwdGlvbktleT86IElLZXk7XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBJWDUwOUNlcnRpZmljYXRlUGtjczEyIGV4dGVuZHMgSUNvbnN0cnVjdCB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGNlcnQ6IElTZWNyZXQ7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgcGFzc3BocmFzZTogSVNlY3JldDtcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgWDUwOUNlcnRpZmljYXRlUGtjczEyIGV4dGVuZHMgWDUwOUNlcnRpZmljYXRlQmFzZSBpbXBsZW1lbnRzIElYNTA5Q2VydGlmaWNhdGVQa2NzMTIge1xuXG4gIHB1YmxpYyByZWFkb25seSBjZXJ0OiBJU2VjcmV0O1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBYNTA5Q2VydGlmaWNhdGVQa2NzMTJQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCwge1xuICAgICAgbGFtYmRhQ29kZTogQ29kZS5mcm9tQXNzZXQoam9pbihfX2Rpcm5hbWUsICcuLicsICcuLicsICdsYW1iZGFzJywgJ25vZGVqcycpKSxcbiAgICAgIGxhbWJkYUhhbmRsZXI6ICd4NTA5LWNlcnRpZmljYXRlLmNvbnZlcnQnLFxuICAgICAgZW5jcnlwdGlvbktleTogcHJvcHMuZW5jcnlwdGlvbktleSxcbiAgICB9KTtcblxuICAgIHByb3BzLnNvdXJjZUNlcnRpZmljYXRlLmdyYW50RnVsbFJlYWQodGhpcy5sYW1iZGFGdW5jKTtcblxuICAgIGNvbnN0IHByb3BlcnRpZXM6IElYNTA5Q2VydGlmaWNhdGVFbmNvZGVQa2NzMTIgPSB7XG4gICAgICBQYXNzcGhyYXNlOiB0aGlzLnBhc3NwaHJhc2Uuc2VjcmV0QXJuLFxuICAgICAgU2VjcmV0OiB7XG4gICAgICAgIE5hbWVQcmVmaXg6IHRoaXMubm9kZS5wYXRoLFxuICAgICAgICBEZXNjcmlwdGlvbjogdGhpcy5ub2RlLnBhdGgsXG4gICAgICAgIEVuY3J5cHRpb25LZXk6IHByb3BzLmVuY3J5cHRpb25LZXk/LmtleUFybixcbiAgICAgICAgVGFnczogW1xuICAgICAgICAgIHtcbiAgICAgICAgICAgIEtleTogdGhpcy51bmlxdWVUYWcua2V5LFxuICAgICAgICAgICAgVmFsdWU6IHRoaXMudW5pcXVlVGFnLnZhbHVlLFxuICAgICAgICAgIH0sXG4gICAgICAgIF0sXG4gICAgICB9LFxuICAgICAgQ2VydGlmaWNhdGU6IHtcbiAgICAgICAgQ2VydDogcHJvcHMuc291cmNlQ2VydGlmaWNhdGUuY2VydC5zZWNyZXRBcm4sXG4gICAgICAgIENlcnRDaGFpbjogcHJvcHMuc291cmNlQ2VydGlmaWNhdGUuY2VydENoYWluID8gcHJvcHMuc291cmNlQ2VydGlmaWNhdGUuY2VydENoYWluLnNlY3JldEFybiA6ICcnLFxuICAgICAgICBLZXk6IHByb3BzLnNvdXJjZUNlcnRpZmljYXRlLmtleS5zZWNyZXRBcm4sXG4gICAgICAgIFBhc3NwaHJhc2U6IHByb3BzLnNvdXJjZUNlcnRpZmljYXRlLnBhc3NwaHJhc2Uuc2VjcmV0QXJuLFxuICAgICAgfSxcbiAgICB9O1xuXG4gICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0RlZmF1bHQnLCB7XG4gICAgICBzZXJ2aWNlVG9rZW46IHRoaXMubGFtYmRhRnVuYy5mdW5jdGlvbkFybixcbiAgICAgIHByb3BlcnRpZXMsXG4gICAgICByZXNvdXJjZVR5cGU6ICdDdXN0b206OlJGREtfWDUwOV9QS0NTMTInLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jZXJ0ID0gU2VjcmV0LmZyb21TZWNyZXRBdHRyaWJ1dGVzKHRoaXMsICdDZXJ0Jywge1xuICAgICAgc2VjcmV0QXJuOiBUb2tlbi5hc1N0cmluZyhyZXNvdXJjZS5nZXRBdHQoJ0NlcnQnKSksXG4gICAgICBlbmNyeXB0aW9uS2V5OiBwcm9wcy5lbmNyeXB0aW9uS2V5LFxuICAgIH0pO1xuICB9XG59XG4iXX0=