"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ScriptAsset = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
var ScriptExtension;
(function (ScriptExtension) {
    ScriptExtension[ScriptExtension[".sh"] = 0] = ".sh";
    ScriptExtension[ScriptExtension[".ps1"] = 1] = ".ps1";
})(ScriptExtension || (ScriptExtension = {}));
var ScriptPathPrefix;
(function (ScriptPathPrefix) {
    ScriptPathPrefix[ScriptPathPrefix["bash"] = 0] = "bash";
    ScriptPathPrefix[ScriptPathPrefix["powershell"] = 1] = "powershell";
})(ScriptPathPrefix || (ScriptPathPrefix = {}));
/**
 * This method returns the path to a script based on RFDK conventional directory structure and the target
 * operating system of the script.
 *
 * The directory structure convention keeps linux scripts in `${scriptRoot}//bash/*.sh` and Windows scripts in
 * `${scriptRoot}/powershell/*.ps1`.
 *
 * @param osType
 * @param scriptName
 */
function getConventionalScriptPath(params) {
    const { rootDir: scriptDir, baseName: scriptName, osType } = params;
    // Make sure we have a known osType. The error message is pretty obtuse if we don't:
    //  The "path" argument must be of type string. Received undefined
    if (ScriptPathPrefix[osType] === undefined || ScriptExtension[osType] == undefined) {
        throw Error(`Unknown osType: ${osType}`);
    }
    return path.join(scriptDir, ScriptPathPrefix[osType], `${scriptName}${ScriptExtension[osType]}`);
}
/**
 * An S3 asset that contains a shell script intended to be executed through instance user data.
 *
 * This is used by other constructs to generalize the concept of a script
 * (bash or powershell) that executes on an instance.
 * It provides a wrapper around the CDK’s S3 Asset construct
 * ( https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-s3-assets.Asset.html )
 *
 * The script asset is placed into and fetched from the CDK bootstrap S3 bucket.
 *
 * Resources Deployed
 * ------------------------
 * - An Asset which is uploaded to the bootstrap S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class ScriptAsset extends aws_s3_assets_1.Asset {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Returns a {@link ScriptAsset} instance by computing the path to the script using RFDK's script directory structure convention.
     *
     * By convention, scripts are kept in a `scripts` directory in each `aws-rfdk/*` package. The scripts are organized
     * based on target shell (and implicitly target operating system). The directory structure looks like:
     *
     * ```
     * scripts/
     *    bash/
     *      script-one.sh
     *      script-two.sh
     *    powershell
     *      script-one.ps1
     *      script-one.ps1
     * ```
     *
     * @param scope The scope for the created {@link ScriptAsset}.
     * @param id The construct id for the created {@link ScriptAsset}.
     * @param scriptParams The parameters that are used to compute the conventional path to the script file.
     * @stability stable
     */
    static fromPathConvention(scope, id, scriptParams) {
        const scriptPath = getConventionalScriptPath(scriptParams);
        return new ScriptAsset(scope, id, { path: scriptPath });
    }
    /**
     * Adds commands to the {@link IScriptHost} to download and execute the ScriptAsset.
     *
     * @param props The parameters for executing the script.
     * @stability stable
     */
    executeOn(props) {
        const { host, args } = props;
        // Grant permission to fetch the script asset
        this.grantRead(host);
        // Add a command to the user data to download the script asset
        const instanceScriptPath = host.userData.addS3DownloadCommand({
            bucket: this.bucket,
            bucketKey: this.s3ObjectKey,
        });
        // Add a command to the user data to execute the downloaded script
        host.userData.addExecuteFileCommand({
            filePath: instanceScriptPath,
            arguments: (args || []).join(' '),
        });
    }
}
exports.ScriptAsset = ScriptAsset;
_a = JSII_RTTI_SYMBOL_1;
ScriptAsset[_a] = { fqn: "aws-rfdk.ScriptAsset", version: "0.26.0" };
//# sourceMappingURL=data:application/json;base64,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