"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark The default access point is writeable only by the root user.
 * @todo Add support for specifying an AccessPoint for the EFS filesystem to  enforce user and group information for all file system requests.
 */
class MountableEfs {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton();
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    mountAssetSingleton() {
        var _b;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh', '!ec2-certificates.crt'],
        });
    }
}
exports.MountableEfs = MountableEfs;
_a = JSII_RTTI_SYMBOL_1;
MountableEfs[_a] = { fqn: "aws-rfdk.MountableEfs", version: "0.26.0" };
//# sourceMappingURL=data:application/json;base64,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