/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import * as efs from '@aws-cdk/aws-efs';
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, IMountingInstance, LinuxMountPointProps } from './mountable-filesystem';
/**
 * Properties that are required to create a {@link MountableEfs}.
 *
 * @stability stable
 */
export interface MountableEfsProps {
    /**
     * The {@link https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-efs.FileSystem.html|EFS} filesystem that will be mounted by the object.
     *
     * @stability stable
     */
    readonly filesystem: efs.IFileSystem;
    /**
     * Extra NFSv4 mount options that will be added to /etc/fstab for the file system. See: {@link https://www.man7.org/linux/man-pages//man5/nfs.5.html}.
     *
     * The given values will be joined together into a single string by commas.
     * ex: ['soft', 'rsize=4096'] will become 'soft,rsize=4096'
     *
     * @default No extra options.
     * @stability stable
     */
    readonly extraMountOptions?: string[];
}
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto an instance.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 * @remark The default access point is writeable only by the root user.
 * @todo Add support for specifying an AccessPoint for the EFS filesystem to  enforce user and group information for all file system requests.
 */
export declare class MountableEfs implements IMountableLinuxFilesystem {
    protected readonly scope: Construct;
    protected readonly props: MountableEfsProps;
    /**
     * @stability stable
     */
    constructor(scope: Construct, props: MountableEfsProps);
    /**
     * Mount the filesystem to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the UserData of the instance to mount the filesystem on startup.
     * If required, the instance's security group is granted ingress to the filesystem's security
     * group on the required ports.
     *
     * @stability stable
     * @inheritdoc true
     */
    mountToLinuxInstance(target: IMountingInstance, mount: LinuxMountPointProps): void;
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     *
     * @stability stable
     */
    protected mountAssetSingleton(): Asset;
}
