"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MongoDbInstaller = exports.MongoDbSsplLicenseAcceptance = exports.MongoDbVersion = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
/**
 * Versions of MongoDB Community Edition that the {@link MongoDbInstaller} is
 * able to install.
 */
var MongoDbVersion;
(function (MongoDbVersion) {
    MongoDbVersion["COMMUNITY_3_6"] = "3.6";
})(MongoDbVersion = exports.MongoDbVersion || (exports.MongoDbVersion = {}));
/**
 * Choices for signifying the user's stance on the terms of the SSPL.
 *
 * See: https://www.mongodb.com/licensing/server-side-public-license
 *
 * @stability stable
 */
var MongoDbSsplLicenseAcceptance;
(function (MongoDbSsplLicenseAcceptance) {
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_REJECTS_SSPL"] = 0] = "USER_REJECTS_SSPL";
    MongoDbSsplLicenseAcceptance[MongoDbSsplLicenseAcceptance["USER_ACCEPTS_SSPL"] = 1] = "USER_ACCEPTS_SSPL";
})(MongoDbSsplLicenseAcceptance = exports.MongoDbSsplLicenseAcceptance || (exports.MongoDbSsplLicenseAcceptance = {}));
/**
 * This class provides a mechanism to install a version of MongoDB Community Edition during the initial launch of an instance.
 *
 * MongoDB is installed from the official sources using the system
 * package manger (yum). It installs the mongodb-org metapackage which will install the following packages:
 * 1) mongodb-org-mongos;
 * 2) mongodb-org-server;
 * 3) mongodb-org-shell; and
 * 4) mongodb-org-tools.
 *
 * Successful installation of MongoDB with this class requires:
 * 1) Explicit acceptance of the terms of the SSPL license, under which MongoDB is distributed; and
 * 2) The instance on which the installation is being performed is in a subnet that can access
 * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org
 *
 * Resources Deployed
 * ------------------------
 * - A CDK Asset package containing the installation scripts is deployed to your CDK staging bucket.
 *
 * Security Considerations
 * ------------------------
 * - Since this class installs MongoDB from official sources dynamically during instance start-up, it is succeptable
 *    to an attacker compromising the official MongoDB Inc. distribution channel for MongoDB. Such a compromise may
 *    result in the installation of unauthorized MongoDB binaries. Executing this attack would require an attacker
 *    compromise both the official installation packages and the MongoDB Inc. gpg key with which they are signed.
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class MongoDbInstaller {
    /**
     * @stability stable
     */
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
        // Legal requirement: Users of this class must agree to the terms of the SSPL, without exception.
        // Developer note: It is a legal requirement that the default be USER_REJECTS_SSPL, so this check
        // must throw an error for every value except USER_ACCEPTS_SSPL.
        if (props.userSsplAcceptance !== MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL) {
            throw new Error(MongoDbInstaller.SSPL_ACCEPT_MESSAGE);
        }
    }
    /**
     * Install MongoDB to the given instance at instance startup.
     *
     * This is accomplished by
     * adding scripting to the instance's UserData to install MongoDB.
     *
     * Notes:
     * 1) The instance on which the installation is being performed must be in a subnet that can access
     * the official MongoDB sites: https://repo.mongodb.org/ and https://www.mongodb.org; and
     * 2) At this time, this method only supports installation onto instances that are running an operating system
     * that is compatible with x86-64 RedHat 7 -- this includes Amazon Linux 2, RedHat 7, and CentOS 7.
     *
     * @param target The target instance onto which to install MongoDB.
     * @stability stable
     */
    installOnLinuxInstance(target) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        const installerScriptAsset = this.installerAssetSingleton();
        installerScriptAsset.grantRead(target.grantPrincipal);
        const installerScript = target.userData.addS3DownloadCommand({
            bucket: installerScriptAsset.bucket,
            bucketKey: installerScriptAsset.s3ObjectKey,
        });
        target.userData.addCommands(`bash ${installerScript}`);
    }
    /**
     * Fetch the Asset singleton for the installation script, or generate it if needed.
     *
     * @stability stable
     */
    installerAssetSingleton() {
        var _b;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '5b141ac9-fde5-45d8-9961-b7108fb3b18a';
        const uniqueId = 'MongoDbInstallerAsset' + uuid.replace(/[-]/g, '');
        return (_b = stack.node.tryFindChild(uniqueId)) !== null && _b !== void 0 ? _b : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'mongodb', this.props.version, 'installMongoDb.sh'),
        });
    }
}
exports.MongoDbInstaller = MongoDbInstaller;
_a = JSII_RTTI_SYMBOL_1;
MongoDbInstaller[_a] = { fqn: "aws-rfdk.MongoDbInstaller", version: "0.26.0" };
/**
 * The SSPL licensing message that is presented to the user if they create an instance of
 * this class without explicitly accepting the SSPL.
 *
 * Note to developers: The text of this string is a legal requirement, and must not be altered
 * witout approval.
 */
MongoDbInstaller.SSPL_ACCEPT_MESSAGE = `
The MongoDbInstaller will install MongoDB Community Edition onto one or more EC2 instances.

MongoDB is provided by MongoDB Inc. under the SSPL license. By installing MongoDB, you are agreeing to the terms of this license.
Follow the link below to read the terms of the SSPL license.
https://www.mongodb.com/licensing/server-side-public-license

By using the MongoDbInstaller to install MongoDB you agree to the terms of the SSPL license.

Please set the userSsplAcceptance property to USER_ACCEPTS_SSPL to signify your acceptance of the terms of the SSPL license.
`;
//# sourceMappingURL=data:application/json;base64,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