"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AWSLimitExhaustedError = exports.LoadBalancerFactory = void 0;
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const health_monitor_1 = require("./health-monitor");
/**
 * This class is responsible for managing the statistics for all the
 * load balancers created in this construct. It is also responsible to search
 * for the finding the first Load balancer/Listener which can accomodate the
 * worker-fleet based on its size.
 *
 * A typical load balancer hierarchy looks like following:
 *  |__ Load Balancer 1
 *  |         |____________Listener 1
 *  |         |                 |_______Target Group 1 ------- Target/Fleet
 *  |         |                 |_______Target Group 2 ------- Target/Fleet
 *  |         |
 *  |         |____________Listener 2
 *  |                           |_______Target Group 1 ------- Target/Fleet
 *  |                           |_______Target Group 2 ------- Target/Fleet
 *  |
 *  |__ Load Balancer 2
 *            |____________Listener 1
 *            |                 |_______Target Group 1 ------- Target/Fleet
 *            |                 |_______Target Group 2 ------- Target/Fleet
 *            |
 *            |____________Listener 2
 *                              |_______Target Group 1 ------- Target/Fleet
 *                              |_______Target Group 2 ------- Target/Fleet
 *
 *  Components:
 *  1. LoadBalancerFactory: This is the root node of the tree. It contains the
 *     map of load balancer to its managers. It is responsible for creating a
 *     new load balancer if required. It delegates the registerFleet calls to
 *     downstream and returns parent load balancer, listener and target group
 *     of the registered fleet if the registration was successful
 *
 *  2. LoadBalancerManager: This class manages a single load balancer. It
 *     contains a map of all the listeners->manager. It also contains the component
 *     counts like listener, target group and target count. It delegates the
 *     registration call to downstream listeners and updates the stats when
 *     the registration is successful. It returns the parent listener and
 *     target group on successful registration.
 *
 *  3. ListenerManager: This class managers a single Listener. It contains a map
 *     of all of its target groups to its associated fleet. It also contains the
 *     component counts. It returns the target group on registration.
 */
class LoadBalancerFactory {
    constructor(healthMonitorScope, vpc) {
        this.loadBalancerMap = new Map();
        this.healthMonitorScope = healthMonitorScope;
        this.vpc = vpc;
    }
    static getAccountLimit(limitName, defaultValue, elbAccountLimits) {
        if (!elbAccountLimits) {
            return defaultValue;
        }
        const foundLimit = elbAccountLimits.find(limit => limit.name === limitName);
        if (!foundLimit) {
            return defaultValue;
        }
        return foundLimit.max;
    }
    /**
     * This method scans all the load balancers and its listeners and registers the fleet
     * to the load balancer and/or listener which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the load balancer, listener
     * and target group to which the fleet was registered.
     *
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(fleet, healthCheckConfig, healthMonitorProps) {
        let loadBalancerParent = null;
        let listenerParent = null;
        let targetGroupParent = null;
        // iterate through each load balancer and try registering to each one.
        for (const [loadBalancer, loadBalancerMeta] of this.loadBalancerMap.entries()) {
            try {
                const { listener, targetGroup } = loadBalancerMeta.registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps);
                loadBalancerParent = loadBalancer;
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        // Check if fleet was not registered.
        if (!loadBalancerParent) {
            // If this section is reached, no load balancer was found which could
            // accommodate fleet, create a new one and register
            loadBalancerParent = this.createLoadBalancer(this.healthMonitorScope, this.loadBalancerMap.size, healthMonitorProps);
            const loadBalancerManager = new LoadBalancerManager();
            // Add it to the map
            this.loadBalancerMap.set(loadBalancerParent, loadBalancerManager);
            // try registering the fleet to the new load balancer
            try {
                const { listener, targetGroup } = loadBalancerManager.registerWorkerFleet(loadBalancerParent, fleet, healthCheckConfig, healthMonitorProps);
                listenerParent = listener;
                targetGroupParent = targetGroup;
            }
            catch (e) {
                throw e;
            }
        }
        /* istanbul ignore next */
        if (!loadBalancerParent || !listenerParent || !targetGroupParent) {
            /* istanbul ignore next */
            throw new Error('Fleet registered successfully but a parent was found null');
        }
        return {
            loadBalancer: loadBalancerParent,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new load balancer within the given scope.
     *
     * @param scope
     * @param loadBalancerindex
     */
    createLoadBalancer(scope, loadBalancerindex, healthMonitorProps) {
        var _a;
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(scope, `ALB_${loadBalancerindex}`, {
            vpc: this.vpc,
            internetFacing: false,
            vpcSubnets: healthMonitorProps.vpcSubnets,
            deletionProtection: (_a = healthMonitorProps.deletionProtection) !== null && _a !== void 0 ? _a : true,
        });
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        return loadBalancer;
    }
}
exports.LoadBalancerFactory = LoadBalancerFactory;
LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER = 50;
LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER = 1000;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER = 5;
LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER = 100;
/**
 * This class manages the properties of a single load balancer and its statistics.
 * It is also responsible to scan through all the listeners registered under it
 * and register the given fleet.
 */
class LoadBalancerManager {
    constructor() {
        this.listenerMap = new Map();
        this.loadBalancerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accomodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the listener
     * and target group to which the fleet was registered.
     *
     * @param loadBalancer
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, fleet, healthCheckConfig, healthMonitorProps) {
        // this initializes with 0 and keeps the track of all components
        // newly added down the hierarchy.
        const statsDelta = new LoadBalancerComponentStats();
        // Do all the load balancer level service limit checks first
        // check for target limit in load balancer
        const targetPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('targets-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGETS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetCount + fleet.targetCapacity) > targetPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "targets-per-application-load-balancer" reached. Limit: ' +
                targetPerLoadBalancerLimit);
        }
        // check for target group limit in load balancer
        const targetGroupsPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.loadBalancerComponentCount.targetGroupCount + 1) > targetGroupsPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-application-load-balancer" reached. Limit: ' +
                targetGroupsPerLoadBalancerLimit);
        }
        let listenerParent = null;
        let targetGroupParent = null;
        // try registering to each listener.
        for (const [listener, listenerMeta] of this.listenerMap.entries()) {
            try {
                const { componentsAdded, targetGroup } = listenerMeta.registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps);
                statsDelta.add(componentsAdded);
                listenerParent = listener;
                targetGroupParent = targetGroup;
                break;
            }
            catch (e) {
                // suppress all AWSLimitExhaustedError, we will scale in case of this error
                /* istanbul ignore next */
                if (!(e instanceof AWSLimitExhaustedError)) {
                    /* istanbul ignore next */
                    throw e;
                }
            }
        }
        /* istanbul ignore next */
        if (!listenerParent) {
            // If this section is reached, no listener was found which could accommodate fleet
            // create new listener and register
            const listenersPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('listeners-per-application-load-balancer', LoadBalancerFactory.DEFAULT_LISTENERS_PER_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
            if ((this.loadBalancerComponentCount.listenerCount + 1) > listenersPerLoadBalancerLimit) {
                throw new AWSLimitExhaustedError('AWS service limit "listeners-per-application-load-balancer" reached. Limit: ' +
                    listenersPerLoadBalancerLimit);
            }
            listenerParent = this.createListener(fleet.targetScope, loadBalancer);
            const listenerManager = new ListenerManager();
            this.listenerMap.set(listenerParent, listenerManager);
            statsDelta.add(new LoadBalancerComponentStats(1, 0, 0));
            try {
                const { componentsAdded, targetGroup } = listenerManager.registerWorkerFleet(loadBalancer, listenerParent, fleet, healthCheckConfig, healthMonitorProps);
                targetGroupParent = targetGroup;
                statsDelta.add(componentsAdded);
            }
            catch (e) {
                throw e;
            }
        }
        // update the current load balancer's stats
        this.loadBalancerComponentCount.add(statsDelta);
        return {
            componentsAdded: statsDelta,
            listener: listenerParent,
            targetGroup: targetGroupParent,
        };
    }
    /**
     * Following method creates a new listener in the fleet's scope and
     * registers it to the given load balancer.
     *
     * @param scope
     * @param loadBalancer
     */
    createListener(scope, loadBalancer) {
        return new aws_elasticloadbalancingv2_1.ApplicationListener(scope, 'Listener', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT + this.listenerMap.size,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            loadBalancer,
            open: false,
        });
    }
}
/**
 * This class manages the properties of a single listener and all the components
 * under its hierarchy.
 * It is also responsible to create a new target group and register the given fleet.
 */
class ListenerManager {
    constructor() {
        this.targetMap = new Map();
        this.listenerComponentCount = new LoadBalancerComponentStats();
    }
    /**
     * This method scans all the listeners of this load balancer and registers the fleet
     * to one which can accommodate it.
     * This method also updates the statistics for the given fleet size.
     * If the registration is successful, it then returns the target group
     * to which the fleet was registered.
     *
     * @param loadBalancer
     * @param listener
     * @param fleet
     * @param healthCheckConfig
     * @param elbAccountLimits
     */
    registerWorkerFleet(loadBalancer, listener, fleet, healthCheckConfig, healthMonitorProps) {
        const componentsAdded = new LoadBalancerComponentStats();
        // Do all listener level service limit checks
        // check for target limit in listener
        const targetGroupPerLoadBalancerLimit = LoadBalancerFactory.getAccountLimit('target-groups-per-action-on-application-load-balancer', LoadBalancerFactory.DEFAULT_TARGET_GROUPS_PER_ACTION_ON_APPLICATION_LOAD_BALANCER, healthMonitorProps.elbAccountLimits);
        if ((this.listenerComponentCount.targetGroupCount + 1) > targetGroupPerLoadBalancerLimit) {
            throw new AWSLimitExhaustedError('AWS service limit "target-groups-per-action-on-application-load-balancer" reached. Limit: ' +
                targetGroupPerLoadBalancerLimit);
        }
        // latest version of CDK does not support 'forwardConfig' in listener rule yet. This means
        // we cannot add multiple target groups to a single listener. Adding this check till this
        // feature is supported.
        if (this.listenerComponentCount.targetGroupCount > 0) {
            throw new AWSLimitExhaustedError('Unable to add more than 1 Target Group to Listener.');
        }
        // Create a new target group
        const targetGroup = this.createTargetGroup(fleet.targetScope, loadBalancer, listener, fleet, healthCheckConfig);
        this.targetMap.set(targetGroup, fleet);
        // update the listener stats
        componentsAdded.targetGroupCount++;
        componentsAdded.targetCount += fleet.targetCapacity;
        // update the current listener's stats
        this.listenerComponentCount.add(componentsAdded);
        return {
            componentsAdded,
            targetGroup,
        };
    }
    /**
     * Following method creates a new new target group in the fleet's scope and
     * registers it to the given listener.
     *
     * @param scope
     * @param loadBalancer
     * @param listener
     * @param monitorableFleet
     * @param healthCheckConfig
     */
    createTargetGroup(scope, loadBalancer, listener, monitorableFleet, healthCheckConfig) {
        const targetGroup = new aws_elasticloadbalancingv2_1.ApplicationTargetGroup(scope, 'TargetGroup', {
            port: health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT,
            protocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
            targets: [monitorableFleet.targetToMonitor],
            healthCheck: {
                port: healthCheckConfig.port ? healthCheckConfig.port.toString() : health_monitor_1.HealthMonitor.LOAD_BALANCER_LISTENING_PORT.toString(),
                interval: healthCheckConfig.interval || health_monitor_1.HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL,
                healthyThresholdCount: healthCheckConfig.instanceHealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD,
                unhealthyThresholdCount: healthCheckConfig.instanceUnhealthyThresholdCount || health_monitor_1.HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD,
                protocol: aws_elasticloadbalancingv2_1.Protocol.HTTP,
            },
            vpc: loadBalancer.vpc,
        });
        listener.addTargetGroups('TargetGroup', {
            targetGroups: [targetGroup],
        });
        return targetGroup;
    }
}
/**
 * This class contains the statistics of all the nested load balancer
 * components like listener count, target group count and target count.
 * This statistics object will be associated with each load balancer
 * and listener for tracking the count of components.
 */
class LoadBalancerComponentStats {
    constructor(listenerCount = 0, targetGroupCount = 0, targetCount = 0) {
        this.listenerCount = listenerCount;
        this.targetGroupCount = targetGroupCount;
        this.targetCount = targetCount;
    }
    add(operand) {
        this.listenerCount += operand.listenerCount;
        this.targetGroupCount += operand.targetGroupCount;
        this.targetCount += operand.targetCount;
    }
}
class AWSLimitExhaustedError extends Error {
    constructor(message) {
        super(message);
    }
}
exports.AWSLimitExhaustedError = AWSLimitExhaustedError;
//# sourceMappingURL=data:application/json;base64,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