"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.HealthMonitor = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_cloudwatch_actions_1 = require("@aws-cdk/aws-cloudwatch-actions");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
const load_balancer_manager_1 = require("./load-balancer-manager");
const runtime_info_1 = require("./runtime-info");
/**
 *  A new or imported Health Monitor.
 */
class HealthMonitorBase extends core_1.Construct {
}
/**
 * This construct is responsible for the deep health checks of compute instances.
 *
 * It also replaces unhealthy instances and suspends unhealthy fleets.
 * Although, using this constructs adds up additional costs for monitoring,
 * it is highly recommended using this construct to help avoid / minimize runaway costs for compute instances.
 *
 * An instance is considered to be unhealthy when:
 *    1) Deadline client is not installed on it;
 *    2) Deadline client is installed but not running on it;
 *    3) RCS is not configured correctly for Deadline client;
 *    4) it is unable to connect to RCS due to any infrastructure issues;
 *    5) the health monitor is unable to reach it because of some infrastructure issues.
 *
 * A fleet is considered to be unhealthy when:
 *    1) at least 1 instance is unhealthy for the configured grace period;
 *    2) a percentage of unhealthy instances in the fleet is above a threshold at any given point of time.
 *
 * This internally creates an array of application load balancers and attaches
 * the worker-fleet (which internally is implemented as an Auto Scaling Group) to its listeners.
 * There is no load-balancing traffic on the load balancers,
 * it is only used for health checks.
 * Intention is to use the default properties of laod balancer health
 * checks which does HTTP pings at frequent intervals to all the
 * instances in the fleet and determines its health. If any of the
 * instance is found unhealthy, it is replaced. The target group
 * also publishes the unhealthy target count metric which is used
 * to identify the unhealthy fleet.
 *
 * Other than the default instance level protection, it also creates a lambda
 * which is responsible to set the fleet size to 0 in the event of a fleet
 * being sufficiently unhealthy to warrant termination.
 * This lambda is triggered by CloudWatch alarms via SNS (Simple Notification Service).
 *
 * Resources Deployed
 * ------------------------
 * - Application Load Balancer(s) doing frequent pings to the workers.
 * - An Amazon Simple Notification Service (SNS) topic for all unhealthy fleet notifications.
 * - An AWS Key Management Service (KMS) Key to encrypt SNS messages - If no encryption key is provided.
 * - An Amazon CloudWatch Alarm that triggers if a worker fleet is unhealthy for a long period.
 * - Another CloudWatch Alarm that triggers if the healthy host percentage of a worker fleet is lower than allowed.
 * - A single AWS Lambda function that sets fleet size to 0 when triggered in response to messages on the SNS Topic.
 * - Execution logs of the AWS Lambda function are published to a log group in Amazon CloudWatch.
 *
 * Security Considerations
 * ------------------------
 * - The AWS Lambda that is deployed through this construct will be created from a deployment package
 *    that is uploaded to your CDK bootstrap bucket during deployment. You must limit write access to
 *    your CDK bootstrap bucket to prevent an attacker from modifying the actions performed by this Lambda.
 *    We strongly recommend that you either enable Amazon S3 server access logging on your CDK bootstrap bucket,
 *    or enable AWS CloudTrail on your account to assist in post-incident analysis of compromised production
 *    environments.
 * - The AWS Lambda that is created by this construct to terminate unhealthy worker fleets has permission to
 *    UpdateAutoScalingGroup ( https://docs.aws.amazon.com/autoscaling/ec2/APIReference/API_UpdateAutoScalingGroup.html )
 *    on all of the fleets that this construct is monitoring. You should not grant any additional actors/principals the
 *    ability to modify or execute this Lambda.
 * - Execution of the AWS Lambda for terminating unhealthy workers is triggered by messages to the Amazon Simple
 *    Notification Service (SNS) Topic that is created by this construct. Any principal that is able to publish notification
 *    to this SNS Topic can cause the Lambda to execute and reduce one of your worker fleets to zero instances. You should
 *    not grant any additional principals permissions to publish to this SNS Topic.
 *
 * @stability stable
 */
class HealthMonitor extends HealthMonitorBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.env = {
            account: this.stack.account,
            region: this.stack.region,
        };
        this.props = props;
        this.lbFactory = new load_balancer_manager_1.LoadBalancerFactory(this, props.vpc);
        const topicEncryptKey = props.encryptionKey || new aws_kms_1.Key(this, 'SNSEncryptionKey', {
            description: `This key is used to encrypt SNS messages for ${this.node.uniqueId}.`,
            enableKeyRotation: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            trustAccountIdentities: true,
        });
        // allow cloudwatch service to send encrypted messages
        topicEncryptKey.grant(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'), 'kms:Decrypt', 'kms:GenerateDataKey');
        this.unhealthyFleetActionTopic = new aws_sns_1.Topic(this, 'UnhealthyFleetTopic', {
            masterKey: topicEncryptKey,
        });
        this.unhealthyFleetActionTopic.grantPublish(new aws_iam_1.ServicePrincipal('cloudwatch.amazonaws.com'));
        this.alarmTopicAction = new aws_cloudwatch_actions_1.SnsAction(this.unhealthyFleetActionTopic);
        this.unhealthyFleetActionLambda = new aws_lambda_1.SingletonFunction(this, 'UnhealthyFleetAction', {
            code: aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', '..', 'lambdas', 'nodejs', 'unhealthyFleetAction')),
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            lambdaPurpose: 'unhealthyFleetTermination',
            timeout: core_1.Duration.seconds(300),
            uuid: '28bccf6a-aa76-478c-9239-e2f5bcc0254c',
        });
        this.unhealthyFleetActionTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(this.unhealthyFleetActionLambda));
        // Tag deployed resources with RFDK meta-data
        runtime_info_1.tagConstruct(this);
    }
    /**
     * Attaches the load-balancing target to the ELB for instance-level monitoring.
     *
     * The ELB does frequent pings to the workers and determines
     * if a worker node is unhealthy. If so, it replaces the instance.
     *
     * It also creates an Alarm for healthy host percent and suspends the
     * fleet if the given alarm is breaching. It sets the maxCapacity
     * property of the auto-scaling group to 0. This should be
     * reset manually after fixing the issue.
     *
     * @stability stable
     */
    registerFleet(monitorableFleet, healthCheckConfig) {
        const { loadBalancer, targetGroup } = this.lbFactory.registerWorkerFleet(monitorableFleet, healthCheckConfig, this.props);
        this.createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup);
    }
    createFleetAlarms(monitorableFleet, healthCheckConfig, loadBalancer, targetGroup) {
        monitorableFleet.connections.allowFrom(loadBalancer, aws_ec2_1.Port.tcp(healthCheckConfig.port || HealthMonitor.LOAD_BALANCER_LISTENING_PORT));
        const percentMetric = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD,
        });
        // When unhealthy fleet is more than healthyFleetThresholdPercent or 35% at any given period of 5 minutes
        const immediateTerminationAlarm = percentMetric.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetTermination', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: 100 - (healthCheckConfig.healthyFleetThresholdPercent || HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD,
            actionsEnabled: true,
        });
        immediateTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        // When at least one node is unhealthy over a period of 2 hours
        const percentMetricGracePeriod = new aws_cloudwatch_1.MathExpression({
            label: 'UnhealthyHostPercent',
            expression: 'IF(fleetCapacity, 100*(unhealthyHostCount/fleetCapacity), 0)',
            usingMetrics: {
                unhealthyHostCount: targetGroup.metricUnhealthyHostCount({
                    statistic: 'max',
                }),
                fleetCapacity: monitorableFleet.targetCapacityMetric,
            },
            period: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE,
        });
        const gracePeriodTerminationAlarm = percentMetricGracePeriod.createAlarm(monitorableFleet.targetScope, 'UnhealthyFleetGracePeriod', {
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
            threshold: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE,
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            datapointsToAlarm: HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE,
            actionsEnabled: true,
        });
        gracePeriodTerminationAlarm.addAlarmAction(this.alarmTopicAction);
        monitorableFleet.targetUpdatePolicy.attachToRole(this.unhealthyFleetActionLambda.role);
    }
}
exports.HealthMonitor = HealthMonitor;
_a = JSII_RTTI_SYMBOL_1;
HealthMonitor[_a] = { fqn: "aws-rfdk.HealthMonitor", version: "0.26.0" };
/**
 * Default health check listening port.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_PORT = 63415;
/**
 * Resource Tracker in Deadline currently publish health status every 5 min, hence keeping this same.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(5);
/**
 * Resource Tracker in Deadline currently determines host unhealthy in 15 min, hence keeping this count.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_UNHEALTHY_HOST_THRESHOLD = 3;
/**
 * This is the minimum possible value of ALB health-check config, we want to mark worker healthy ASAP.
 *
 * @stability stable
 */
HealthMonitor.DEFAULT_HEALTHY_HOST_THRESHOLD = 2;
/**
 * Since we are not doing any load balancing, this port is just an arbitrary port.
 *
 * @stability stable
 */
HealthMonitor.LOAD_BALANCER_LISTENING_PORT = 8081;
/**
 * This number is taken from Resource Tracker implementation. If a fleet's healthy percent
 * is less than this threshold at any given point of time, it is suspended.
 */
HealthMonitor.DEFAULT_HEALTHY_FLEET_THRESHOLD_PERCENT_HARD = 65;
/**
 * This number is taken from Resource Tracker implementation. If a fleet has at least 1
 * unhealthy host for a period of 2 hours, it is suspended.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_THRESHOLD_PERCENT_GRACE = 0;
/**
 * This number is taken from Resource Tracker implementation. We monitor unhealthy fleet for immediate
 * termination for a period fo 5 minutes.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_HARD = core_1.Duration.minutes(5);
/**
 * In Resource Tracker, we evaluate the fleet's health for determining the grace period over a period
 * of 5 minutes. For the first unhealthy signal, a instance can take upto 10min (max), hence we are
 * setting this period to be 15.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_GRACE = core_1.Duration.minutes(15);
/**
 * This number is taken from Resource Tracker implementation. Fleet is terminated immediately if it
 * has unhealthy host percent above the hard limit.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_HARD = 1;
/**
 * This number is taken from Resource Tracker implementation. The grace period duration is 2 hours,
 * since the grace period is 15 minutes, we need continuous 8 data points crossing the threshold.
 */
HealthMonitor.DEFAULT_UNHEALTHY_FLEET_ALARM_PERIOD_THRESHOLD_GRACE = 8;
//# sourceMappingURL=data:application/json;base64,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