"use strict";
var _a, _b, _c;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConnectableApplicationEndpoint = exports.ApplicationEndpoint = exports.Endpoint = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
/**
 * Connection endpoint.
 *
 * Consists of a combination of hostname, port, and transport protocol.
 *
 * @stability stable
 */
class Endpoint {
    /**
     * Constructs an Endpoint instance.
     *
     * @param props The properties for the endpoint.
     * @stability stable
     */
    constructor(props) {
        var _d;
        const { address, port } = props;
        const protocol = (_d = props.protocol) !== null && _d !== void 0 ? _d : aws_ec2_1.Protocol.TCP;
        if (!core_1.Token.isUnresolved(port) && !Endpoint.isValidPort(port)) {
            throw new Error(`Port must be an integer between [${Endpoint.MIN_PORT}, ${Endpoint.MAX_PORT}] but got: ${port}`);
        }
        this.hostname = address;
        this.portNumber = port;
        this.protocol = protocol;
        this.port = new aws_ec2_1.Port({
            protocol: this.protocol,
            fromPort: port,
            toPort: port,
            stringRepresentation: this.renderPort(port),
        });
        this.socketAddress = `${address}:${this.portAsString()}`;
    }
    /**
     * Determines if a port is valid
     *
     * @param port: The port number
     * @returns boolean whether the port is valid
     */
    static isValidPort(port) {
        return Number.isInteger(port) && port >= Endpoint.MIN_PORT && port <= Endpoint.MAX_PORT;
    }
    /**
     * Returns the port number as a string representation that can be used for embedding within other strings.
     *
     * This is intended to deal with CDK's token system. Numeric CDK tokens are not expanded when their string
     * representation is embedded in a string. This function returns the port either as an unresolved string token or
     * as a resolved string representation of the port value.
     *
     * @returns An (un)resolved string representation of the endpoint's port number
     * @stability stable
     */
    portAsString() {
        if (core_1.Token.isUnresolved(this.portNumber)) {
            return core_1.Token.asString(this.portNumber);
        }
        else {
            return this.portNumber.toString();
        }
    }
    renderPort(port) {
        return core_1.Token.isUnresolved(port) ? '{IndirectPort}' : port.toString();
    }
}
exports.Endpoint = Endpoint;
_a = JSII_RTTI_SYMBOL_1;
Endpoint[_a] = { fqn: "aws-rfdk.Endpoint", version: "0.26.0" };
/**
 * The minimum port value
 */
Endpoint.MIN_PORT = 1;
/**
 * The maximum port value
 */
Endpoint.MAX_PORT = 65535;
/**
 * An endpoint serving http or https for an application.
 *
 * @stability stable
 */
class ApplicationEndpoint extends Endpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint.
     * @stability stable
     */
    constructor(props) {
        var _d;
        super({
            address: props.address,
            port: props.port,
            protocol: aws_ec2_1.Protocol.TCP,
        });
        this.applicationProtocol = (_d = props.protocol) !== null && _d !== void 0 ? _d : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
    }
}
exports.ApplicationEndpoint = ApplicationEndpoint;
_b = JSII_RTTI_SYMBOL_1;
ApplicationEndpoint[_b] = { fqn: "aws-rfdk.ApplicationEndpoint", version: "0.26.0" };
/**
 * An endpoint serving http or https for an application.
 *
 * @stability stable
 */
class ConnectableApplicationEndpoint extends ApplicationEndpoint {
    /**
     * Constructs a {@link ApplicationEndpoint} instance.
     *
     * @param props The properties for the application endpoint.
     * @stability stable
     */
    constructor(props) {
        super(props);
        this.connections = props.connections;
    }
}
exports.ConnectableApplicationEndpoint = ConnectableApplicationEndpoint;
_c = JSII_RTTI_SYMBOL_1;
ConnectableApplicationEndpoint[_c] = { fqn: "aws-rfdk.ConnectableApplicationEndpoint", version: "0.26.0" };
//# sourceMappingURL=data:application/json;base64,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