"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CloudWatchAgent = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
const path = require("path");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_ssm_1 = require("@aws-cdk/aws-ssm");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("./script-assets");
/**
 * This construct is a thin wrapper that provides the ability to install and configure the CloudWatchAgent ( https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html ) on one or more EC2 instances during instance startup.
 *
 * It accomplishes this by downloading and executing the configuration script on the instance.
 * The script will download the CloudWatch Agent installer,
 * optionally verify the installer, and finally install the CloudWatch Agent.
 * The installer is downloaded via the Amazon S3 API, thus, this construct can be used
 * on instances that have no access to the internet as long as the VPC contains
 * an VPC Gateway Endpoint for S3 ( https://docs.aws.amazon.com/vpc/latest/userguide/vpc-endpoints-s3.html ).
 *
 * {@link CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR} - Context variable to skip validation
 * of the downloaded CloudWatch Agent installer if set to 'TRUE'.
 * WARNING: Only use this if your deployments are failing due to a validation failure,
 * but you have verified that the failure is benign.
 *
 * Resources Deployed
 * ------------------------
 * - String SSM Parameter in Systems Manager Parameter Store to store the cloudwatch agent configuration;
 * - A script Asset which is uploaded to S3 bucket.
 *
 * Security Considerations
 * ------------------------
 * - Using this construct on an instance will result in that instance dynamically downloading and running scripts
 *    from your CDK bootstrap bucket when that instance is launched. You must limit write access to your CDK bootstrap
 *    bucket to prevent an attacker from modifying the actions performed by these scripts. We strongly recommend that
 *    you either enable Amazon S3 server access logging on your CDK bootstrap bucket, or enable AWS CloudTrail on your
 *    account to assist in post-incident analysis of compromised production environments.
 *
 * @stability stable
 */
class CloudWatchAgent extends core_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        // Create the asset for the configuration script
        this.configurationScript = script_assets_1.ScriptAsset.fromPathConvention(scope, 'CloudWatchConfigurationScriptAsset', {
            osType: props.host.osType,
            baseName: 'configureCloudWatchAgent',
            rootDir: path.join(__dirname, '../scripts/'),
        });
        // Create a new SSM Parameter holding the json configuration
        this.ssmParameterForConfig = new aws_ssm_1.StringParameter(scope, 'StringParameter', {
            description: 'config file for Repository logs config',
            stringValue: props.cloudWatchConfig,
        });
        this.grantRead(props.host);
        this.configure(props.host, this.node.tryGetContext(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR) === 'TRUE');
    }
    /**
     * Grants read permissions to the principal on the assets bucket and parameter store.
     */
    grantRead(grantee) {
        this.configurationScript.grantRead(grantee);
        this.ssmParameterForConfig.grantRead(grantee);
    }
    /**
     * Configures the CloudWatch Agent on the target host.
     *
     * This is done by adding UserData commands to the target host.
     *
     * @param host The host to configure the CloudWatch agent on
     * @param skipValidation Skips the validation of the CloudWatch agent installer if set to true.
     */
    configure(host, skipValidation) {
        // Grant access to the required CloudWatch Agent installer files
        const cloudWatchAgentBucket = aws_s3_1.Bucket.fromBucketArn(this, 'CloudWatchAgentBucket', 'arn:aws:s3:::amazoncloudwatch-agent');
        cloudWatchAgentBucket.grantRead(host);
        const scriptArgs = [];
        if (skipValidation) {
            // Flags must be set before positional arguments for some scripts
            scriptArgs.push(CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG);
        }
        scriptArgs.push(this.ssmParameterForConfig.parameterName);
        this.configurationScript.executeOn({
            host,
            args: scriptArgs,
        });
    }
}
exports.CloudWatchAgent = CloudWatchAgent;
_a = JSII_RTTI_SYMBOL_1;
CloudWatchAgent[_a] = { fqn: "aws-rfdk.CloudWatchAgent", version: "0.26.0" };
/**
 * The context variable to indicate that CloudWatch agent installer validation should be skipped.
 *
 * @stability stable
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_CTX_VAR = 'RFDK_SKIP_CWAGENT_VALIDATION';
/**
 * The flag for configureCloudWatchAgent script to skip CloudWatch agent installer validation.
 */
CloudWatchAgent.SKIP_CWAGENT_VALIDATION_FLAG = '-s';
//# sourceMappingURL=data:application/json;base64,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