// Copyright (c) Mito
import React, { useEffect, useState } from 'react';
import XIcon from '../../icons/XIcon';
import AxisSection, { GraphAxisType } from './AxisSection';
import LoadingSpinner from './LoadingSpinner';
import { TaskpaneType } from '../taskpanes';
import useDelayedAction from '../../../hooks/useDelayedAction';
import Select from '../../elements/Select';
import Col from '../../spacing/Col';
import Row from '../../spacing/Row';
import TextButton from '../../elements/TextButton';
import { useCopyToClipboard } from '../../../hooks/useCopyToClipboard';
import { intersection } from '../../../utils/arrays';
import DropdownItem from '../../elements/DropdownItem';
// import css
import '../../../../css/taskpanes/Graph/GraphSidebar.css';
import '../../../../css/taskpanes/Graph/LoadingSpinner.css';
import DefaultEmptyTaskpane from '../DefaultTaskpane/DefaultEmptyTaskpane';
import { isNumberDtype } from '../../../utils/dtypes';
import Toggle from '../../elements/Toggle';
export var GraphType;
(function (GraphType) {
    GraphType["SCATTER"] = "scatter";
    GraphType["BAR"] = "bar";
    GraphType["HISTOGRAM"] = "histogram";
    GraphType["BOX"] = "box";
    GraphType["SUMMARY_STAT"] = "summary_stat";
})(GraphType || (GraphType = {}));
// Millisecond delay between loading graphs, so that
// we don't load to many graphs when the user is clicking around
const LOAD_GRAPH_TIMEOUT = 1000;
// Graphing a dataframe with more than this number of rows will
// give the user the option to apply the safety filter
// Note: This must be kept in sync with the graphing heuristic in the mitosheet/graph folder
const GRAPH_SAFETY_FILTER_CUTOFF = 1000;
// Tooltips used to explain the Safety filter toggle
const SAFETY_FILTER_DISABLED_MESSAGE = `Because you’re graphing less than ${GRAPH_SAFETY_FILTER_CUTOFF} rows of data, you can safely graph your data without applying a filter first.`;
const SAFETY_FILTER_ENABLED_MESSAGE = `Turning on Filter to Safe Size only graphs the first ${GRAPH_SAFETY_FILTER_CUTOFF} rows of your dataframe, ensuring that your browser tab won’t crash. Turning off Filter to Safe Size graphs the entire dataframe and may slow or crash your browser tab.`;
// Helper function for creating default graph params 
const getDefaultGraphParams = (sheetDataArray, sheetIndex) => {
    const safetyFilter = getDefaultSafetyFilter(sheetDataArray, sheetIndex);
    return {
        graphType: GraphType.BAR,
        xAxisColumnIDs: [],
        yAxisColumnIDs: [],
        safetyFilter: safetyFilter
    };
};
// Helper function for getting the default safety filter status
const getDefaultSafetyFilter = (sheetDataArray, sheetIndex) => {
    return sheetDataArray[sheetIndex] === undefined || sheetDataArray[sheetIndex].numRows > GRAPH_SAFETY_FILTER_CUTOFF;
};
/*
    A helper function for getting the params for the graph fpr this sheet when
    opening the graphing taskpane, or when switching to a sheet.

    Notably, will filter oout any columns that are no longer in the dataset,
    which stops the user from having invalid columns selected in their graph
    params.
*/
const getGraphParams = (lastGraphParams, sheetIndex, sheetDataArray) => {
    const lastParams = lastGraphParams[sheetIndex];
    if (lastParams !== undefined) {
        // Filter out column headers that no longer exist
        const validColumnIDs = sheetDataArray[sheetIndex] !== undefined ? sheetDataArray[sheetIndex].data.map(c => c.columnID) : [];
        const xAxisColumnIDs = intersection(validColumnIDs, lastParams.xAxisColumnIDs);
        const yAxisColumnIDs = intersection(validColumnIDs, lastParams.yAxisColumnIDs);
        return {
            graphType: lastParams.graphType,
            xAxisColumnIDs: xAxisColumnIDs,
            yAxisColumnIDs: yAxisColumnIDs,
            safetyFilter: lastParams.safetyFilter
        };
    }
    return getDefaultGraphParams(sheetDataArray, sheetIndex);
};
/*
    This is the main component that displays all graphing
    functionality, allowing the user to build and view graphs.
*/
const GraphSidebar = (props) => {
    const [selectedSheetIndex, _setSelectedSheetIndex] = useState(props.graphSidebarSheet);
    // A wrapper around changing the selected sheet index that makes sure
    // the correct sheet is displayed, and also loads the most recent graph
    // from this sheet if it exists
    const setSelectedSheetIndex = (newSelectedSheetIndex) => {
        const newParams = getGraphParams(props.lastGraphParams, newSelectedSheetIndex, props.sheetDataArray);
        // Note we update the sheet before the graph parameters
        _setSelectedSheetIndex(newSelectedSheetIndex);
        _setGraphParams(newParams);
        props.setUIState(prevUIState => {
            return Object.assign(Object.assign({}, prevUIState), { selectedSheetIndex: newSelectedSheetIndex });
        });
    };
    // When opening the graphing modal, if there are params for the last graph that was made
    // for this sheet, then take them. Otherwise, just take the default params
    const [graphParams, _setGraphParams] = useState(getGraphParams(props.lastGraphParams, selectedSheetIndex, props.sheetDataArray));
    // When we update the graph params, we also update the lastGraphParams in the 
    // main Mito component, so that we can open the graph to the same state next time
    const setGraphParams = (newGraphParams) => {
        _setGraphParams(newGraphParams);
        props.setLastGraphParams(selectedSheetIndex, newGraphParams);
    };
    const [graphObj, setGraphObj] = useState(undefined);
    const [loading, setLoading] = useState(false);
    const [_copyGraphCode, graphCodeCopied] = useCopyToClipboard((graphObj === null || graphObj === void 0 ? void 0 : graphObj.generation_code) || '');
    const [changeLoadingGraph] = useDelayedAction(LOAD_GRAPH_TIMEOUT);
    // If the graph has non-default params, then it has been configured
    const [graphHasNeverBeenConfigured, setGraphHasNeverBeenConfigured] = useState(graphParams === getDefaultGraphParams(props.sheetDataArray, selectedSheetIndex));
    // We log when the graph has been opened
    useEffect(() => {
        void props.mitoAPI.sendLogMessage('opened_graph');
    }, []);
    /*
        Gets fired whenever the user makes a change to their graph.
    
        It calls the loadNewGraph function which is on a delay, as to
        not overload the backend with new graph creation requests.
    */
    useEffect(() => {
        // If the graph has never been configured, then don't display the loading indicator
        // or try to create the graph
        if (graphHasNeverBeenConfigured) {
            setGraphHasNeverBeenConfigured(false);
            return;
        }
        // Start the loading icon as soon as the user makes a change to the graph
        setLoading(true);
        void loadNewGraph();
        // We also log when the columns are selected, when they change
        void props.mitoAPI.sendLogMessage('graph_selected_column_changed', {
            'generated_graph': graphParams.xAxisColumnIDs.length !== 0 || graphParams.yAxisColumnIDs.length !== 0,
            'graph_type': graphParams.graphType,
            'x_axis_column_ids': graphParams.xAxisColumnIDs,
            'y_axis_column_ids': graphParams.yAxisColumnIDs
        });
    }, [graphParams]);
    // When we get a new graph script, we execute it here. This is a workaround
    // that is required because we need to make sure this code runs, which it does
    // not when it is a script tag inside innerHtml (which react does not execute
    // for safety reasons).
    useEffect(() => {
        try {
            if (graphObj === undefined) {
                return;
            }
            const executeScript = new Function(graphObj.script);
            executeScript();
        }
        catch (e) {
            console.error("Failed to execute graph function", e);
        }
    }, [graphObj]);
    /*
        This is the actual function responsible for loading the new
        graph from the backend, making sure this graph is the correct
        size.
    */
    const getGraphAsync = async () => {
        var _a;
        const boundingRect = (_a = document.getElementById('graph-div')) === null || _a === void 0 ? void 0 : _a.getBoundingClientRect();
        if (boundingRect !== undefined) {
            const loadedGraphHTMLAndScript = await props.mitoAPI.getGraph(graphParams.graphType, selectedSheetIndex, graphParams.safetyFilter, graphParams.xAxisColumnIDs, graphParams.yAxisColumnIDs, `${(boundingRect === null || boundingRect === void 0 ? void 0 : boundingRect.height) - 10}px`, `${(boundingRect === null || boundingRect === void 0 ? void 0 : boundingRect.width) - 20}px` // Subtract pixels from the height & width to account for padding
            );
            setGraphObj(loadedGraphHTMLAndScript);
        }
        // Turn off the loading icon once the user get their graph back
        setLoading(false);
    };
    /*
        Whenever the graph is changed we set a timeout to start loading a new
        graph. This runs after LOAD_GRAPH_TIMEOUT.
    
        This makes sure we don't send unnecessary messages to the backend if the user
        is switching axes/graph types quickly.
    */
    const loadNewGraph = async () => {
        changeLoadingGraph(getGraphAsync);
    };
    // Toggles the safety filter component of the graph params
    const toggleSafetyFilter = () => {
        const newSafetyFilter = !graphParams.safetyFilter;
        setGraphParams(Object.assign(Object.assign({}, graphParams), { safetyFilter: newSafetyFilter }));
    };
    const removeNonNumberColumnIDs = (columnIDs) => {
        const filteredColumnIDs = columnIDs.filter(columnID => {
            return isNumberDtype(props.columnDtypesMap[columnID]);
        });
        return filteredColumnIDs;
    };
    const _setGraphType = (graphType) => {
        let xAxisColumnIDsCopy = [...graphParams.xAxisColumnIDs];
        let yAxisColumnIDsCopy = [...graphParams.yAxisColumnIDs];
        /*
            If the user switches to a Box plot or Histogram, then we make sure that
            1. all of the selected columns are numbers.
            2. there are not columns in both the x and y axis.
        */
        if (graphType === GraphType.BOX || graphType === GraphType.HISTOGRAM) {
            xAxisColumnIDsCopy = removeNonNumberColumnIDs(xAxisColumnIDsCopy);
            yAxisColumnIDsCopy = removeNonNumberColumnIDs(yAxisColumnIDsCopy);
            // Make sure that only one axis has selected column headers. 
            if (graphParams.xAxisColumnIDs.length > 0 && graphParams.yAxisColumnIDs.length > 0) {
                yAxisColumnIDsCopy = [];
            }
        }
        // Log that we reset the selected columns
        if (xAxisColumnIDsCopy.length !== graphParams.xAxisColumnIDs.length || yAxisColumnIDsCopy.length !== graphParams.yAxisColumnIDs.length) {
            void props.mitoAPI.sendLogMessage('reset_graph_columns_on_graph_type_change');
        }
        // Log that the user switched graph types
        void props.mitoAPI.sendLogMessage('switched_graph_type', {
            'graph_type': graphType,
            'x_axis_column_ids': xAxisColumnIDsCopy,
            'y_axis_column_ids': yAxisColumnIDsCopy,
        });
        // Update the graph type
        setGraphParams(Object.assign(Object.assign({}, graphParams), { graphType: graphType, xAxisColumnIDs: xAxisColumnIDsCopy, yAxisColumnIDs: yAxisColumnIDsCopy }));
    };
    /*
        Function responsible for updating the selected column headers for each axis.
        Set the columnHeader at the index of the graphAxis selected columns array.
    
        To remove a column, leave the columnHeader empty.
    */
    const updateAxisData = (graphAxis, index, columnID) => {
        // Get the current axis data
        let axisColumnIDs = [];
        if (graphAxis === GraphAxisType.X_AXIS) {
            axisColumnIDs = graphParams.xAxisColumnIDs;
        }
        else {
            axisColumnIDs = graphParams.yAxisColumnIDs;
        }
        // Make a copy of the column headers before editing them
        const axisColumnIDsCopy = [...axisColumnIDs];
        if (columnID === undefined) {
            axisColumnIDsCopy.splice(index, 1);
        }
        else {
            axisColumnIDsCopy[index] = columnID;
        }
        // Update the axis data
        if (graphAxis === GraphAxisType.X_AXIS) {
            setGraphParams(Object.assign(Object.assign({}, graphParams), { xAxisColumnIDs: axisColumnIDsCopy }));
        }
        else {
            setGraphParams(Object.assign(Object.assign({}, graphParams), { yAxisColumnIDs: axisColumnIDsCopy }));
        }
    };
    const copyGraphCode = () => {
        _copyGraphCode();
        // Log that the user copied the graph code
        void props.mitoAPI.sendLogMessage('copy_graph_code', {
            'generated_graph': graphParams.xAxisColumnIDs.length !== 0 || graphParams.yAxisColumnIDs.length !== 0,
            'graph_type': graphParams.graphType,
            'x_axis_column_ids': graphParams.xAxisColumnIDs,
            'y_axis_column_ids': graphParams.yAxisColumnIDs
        });
    };
    if (props.sheetDataArray.length === 0) {
        // Since the UI for the graphing takes up the whole screen, we don't even let the user keep it open
        props.setUIState(prevUIState => {
            return Object.assign(Object.assign({}, prevUIState), { currOpenTaskpane: { type: TaskpaneType.NONE } });
        });
        return React.createElement(DefaultEmptyTaskpane, { setUIState: props.setUIState });
    }
    else {
        return (React.createElement("div", { className: 'graph-sidebar-div' },
            React.createElement("div", { className: 'graph-sidebar-graph-div', id: 'graph-div' },
                graphObj === undefined && graphParams.xAxisColumnIDs.length === 0 && graphParams.yAxisColumnIDs.length === 0 &&
                    React.createElement("p", { className: 'graph-sidebar-welcome-text' }, "To generate a graph, select a axis."),
                graphObj !== undefined &&
                    React.createElement("div", { dangerouslySetInnerHTML: { __html: graphObj === null || graphObj === void 0 ? void 0 : graphObj.html } })),
            React.createElement("div", { className: 'graph-sidebar-toolbar-div' },
                React.createElement(Row, { justify: 'space-between', align: 'center' },
                    React.createElement(Col, null,
                        React.createElement("p", { className: 'text-header-2' }, "Generate Graph")),
                    React.createElement(Col, null,
                        React.createElement(XIcon, { onClick: () => {
                                props.setUIState((prevUIState) => {
                                    return Object.assign(Object.assign({}, prevUIState), { currOpenTaskpane: { type: TaskpaneType.NONE } });
                                });
                                void props.mitoAPI.sendLogMessage('closed_graph');
                            } }))),
                React.createElement("div", { className: 'graph-sidebar-toolbar-content' },
                    React.createElement(Row, { justify: 'space-between', align: 'center' },
                        React.createElement(Col, null,
                            React.createElement("p", { className: 'text-header-3' }, "Data Source")),
                        React.createElement(Col, null,
                            React.createElement(Select, { value: props.dfNames[selectedSheetIndex], onChange: (newDfName) => {
                                    const newIndex = props.dfNames.indexOf(newDfName);
                                    setSelectedSheetIndex(newIndex);
                                }, width: 'small' }, props.dfNames.map(dfName => {
                                return (React.createElement(DropdownItem, { key: dfName, title: dfName }));
                            })))),
                    React.createElement(Row, { justify: 'space-between', align: 'center' },
                        React.createElement(Col, null,
                            React.createElement("p", { className: 'text-header-3' }, "Chart Type")),
                        React.createElement(Col, null,
                            React.createElement(Select, { value: graphParams.graphType, onChange: (graphType) => {
                                    _setGraphType(graphType);
                                }, width: 'small', dropdownWidth: 'medium' },
                                React.createElement(DropdownItem, { title: GraphType.BAR }),
                                React.createElement(DropdownItem, { title: GraphType.BOX, subtext: 'Only supports number columns' }),
                                React.createElement(DropdownItem, { title: GraphType.HISTOGRAM, subtext: 'Only supports number columns' }),
                                React.createElement(DropdownItem, { title: GraphType.SCATTER })))),
                    React.createElement(AxisSection
                    /*
                        We use a key here to force the Axis Section to update when the user changes the xAxisColumnHeaders.
                        A key is required because react does not know that the object xAxisColumnHeaders changed in all cases.
                        Particularly, when the user changes the xAxisColumnHeaders from [A, B, A] to [B, A] by
                        deleting the first A, React does not recognize that the change has occurred and so the Axis Section does
                        not update even though the graph updates.

                        We append the indicator xAxis to the front of the list to ensure that both AxisSections have unique keys.
                        When the Axis Sections don't have unique keys, its possible for the sections to become duplicated as per
                        the React warnings.
                    */
                    , { 
                        /*
                            We use a key here to force the Axis Section to update when the user changes the xAxisColumnHeaders.
                            A key is required because react does not know that the object xAxisColumnHeaders changed in all cases.
                            Particularly, when the user changes the xAxisColumnHeaders from [A, B, A] to [B, A] by
                            deleting the first A, React does not recognize that the change has occurred and so the Axis Section does
                            not update even though the graph updates.

                            We append the indicator xAxis to the front of the list to ensure that both AxisSections have unique keys.
                            When the Axis Sections don't have unique keys, its possible for the sections to become duplicated as per
                            the React warnings.
                        */
                        key: ['xAxis'].concat(graphParams.xAxisColumnIDs).join(''), columnIDsMap: props.columnIDsMapArray[selectedSheetIndex], columnDtypesMap: props.columnDtypesMap, graphType: graphParams.graphType, graphAxis: GraphAxisType.X_AXIS, selectedColumnIDs: graphParams.xAxisColumnIDs, otherAxisSelectedColumnIDs: graphParams.yAxisColumnIDs, updateAxisData: updateAxisData, mitoAPI: props.mitoAPI }),
                    React.createElement(AxisSection
                    // See note about keys for Axis Sections above.
                    , { 
                        // See note about keys for Axis Sections above.
                        key: ['yAxis'].concat(graphParams.yAxisColumnIDs).join(''), columnIDsMap: props.columnIDsMapArray[selectedSheetIndex], columnDtypesMap: props.columnDtypesMap, graphType: graphParams.graphType, graphAxis: GraphAxisType.Y_AXIS, selectedColumnIDs: graphParams.yAxisColumnIDs, otherAxisSelectedColumnIDs: graphParams.xAxisColumnIDs, updateAxisData: updateAxisData, mitoAPI: props.mitoAPI }),
                    React.createElement(Row, { justify: 'space-between', align: 'center', title: getDefaultSafetyFilter(props.sheetDataArray, selectedSheetIndex) ? SAFETY_FILTER_ENABLED_MESSAGE : SAFETY_FILTER_DISABLED_MESSAGE },
                        React.createElement(Col, null,
                            React.createElement("p", { className: 'text-header-3' }, "Filter to safe size")),
                        React.createElement(Col, null,
                            React.createElement(Toggle, { value: graphParams.safetyFilter, onChange: toggleSafetyFilter, disabled: !getDefaultSafetyFilter(props.sheetDataArray, selectedSheetIndex) })))),
                React.createElement("div", { className: 'graph-sidebar-toolbar-code-export-button' },
                    React.createElement(TextButton, { variant: 'dark', onClick: copyGraphCode, disabled: loading || graphObj === undefined }, !graphCodeCopied
                        ? "Copy Graph Code"
                        : "Copied!"))),
            loading &&
                React.createElement("div", { className: 'popup-div' },
                    React.createElement(LoadingSpinner, null),
                    React.createElement("p", { className: 'popup-text-div' }, "loading"))));
    }
};
export default GraphSidebar;
//# sourceMappingURL=GraphSidebar.js.map