"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GithubWorkflow = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const case_1 = require("case");
const _resolve_1 = require("../_resolve");
const component_1 = require("../component");
const util_1 = require("../util");
const yaml_1 = require("../yaml");
/**
 * Workflow for GitHub.
 *
 * A workflow is a configurable automated process made up of one or more jobs.
 *
 * @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions
 */
class GithubWorkflow extends component_1.Component {
    constructor(github, name, options = {}) {
        super(github.project);
        this.events = {};
        this.jobs = {};
        this.name = name;
        this.concurrency = options.concurrency;
        this.projenCredentials = github.projenCredentials;
        this.actions = github.actions;
        const workflowsEnabled = github.workflowsEnabled || options.force;
        if (workflowsEnabled) {
            this.file = new yaml_1.YamlFile(this.project, `.github/workflows/${name.toLocaleLowerCase()}.yml`, {
                obj: () => this.renderWorkflow(),
                // GitHub needs to read the file from the repository in order to work.
                committed: true,
            });
        }
    }
    /**
     * Add events to triggers the workflow.
     *
     * @param events The event(s) to trigger the workflow.
     */
    on(events) {
        this.events = {
            ...this.events,
            ...events,
        };
    }
    /**
     * Adds a single job to the workflow.
     * @param id The job name (unique within the workflow)
     * @param job The job specification
     */
    addJob(id, job) {
        this.addJobs({ [id]: job });
    }
    /**
     * Add jobs to the workflow.
     *
     * @param jobs Jobs to add.
     */
    addJobs(jobs) {
        verifyJobConstraints(jobs);
        this.jobs = {
            ...this.jobs,
            ...jobs,
        };
    }
    /**
     * Get a single job from the workflow.
     * @param id The job name (unique within the workflow)
     */
    getJob(id) {
        return this.jobs[id];
    }
    /**
     * Updates a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    updateJob(id, job) {
        this.updateJobs({ [id]: job });
    }
    /**
     * Updates jobs for this worklow
     * Does a complete replace, it does not try to merge the jobs
     *
     * @param jobs Jobs to update.
     */
    updateJobs(jobs) {
        verifyJobConstraints(jobs);
        const newJobIds = Object.keys(jobs);
        const updatedJobs = Object.entries(this.jobs).map(([jobId, job]) => {
            if (newJobIds.includes(jobId)) {
                return [jobId, jobs[jobId]];
            }
            return [jobId, job];
        });
        this.jobs = {
            ...Object.fromEntries(updatedJobs),
        };
    }
    /**
     * Removes a single job to the workflow.
     * @param id The job name (unique within the workflow)
     */
    removeJob(id) {
        const updatedJobs = Object.entries(this.jobs).filter(([jobId]) => jobId !== id);
        this.jobs = {
            ...Object.fromEntries(updatedJobs),
        };
    }
    renderWorkflow() {
        return {
            name: this.name,
            "run-name": this.runName,
            on: snakeCaseKeys(this.events),
            concurrency: this.concurrency,
            jobs: renderJobs(this.jobs, this.actions),
        };
    }
}
exports.GithubWorkflow = GithubWorkflow;
_a = JSII_RTTI_SYMBOL_1;
GithubWorkflow[_a] = { fqn: "projen.github.GithubWorkflow", version: "0.67.87" };
function snakeCaseKeys(obj) {
    if (typeof obj !== "object" || obj == null) {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(snakeCaseKeys);
    }
    const result = {};
    for (let [k, v] of Object.entries(obj)) {
        if (typeof v === "object" && v != null) {
            v = snakeCaseKeys(v);
        }
        result[case_1.snake(k)] = v;
    }
    return result;
}
function renderJobs(jobs, actions) {
    const result = {};
    for (const [name, job] of Object.entries(jobs)) {
        result[name] = renderJob(job);
    }
    return result;
    /** @see https://docs.github.com/en/actions/reference/workflow-syntax-for-github-actions */
    function renderJob(job) {
        const steps = new Array();
        // https://docs.github.com/en/actions/using-workflows/reusing-workflows#supported-keywords-for-jobs-that-call-a-reusable-workflow
        if ("uses" in job) {
            return {
                name: job.name,
                needs: arrayOrScalar(job.needs),
                if: job.if,
                permissions: util_1.kebabCaseKeys(job.permissions),
                concurrency: job.concurrency,
                uses: job.uses,
                with: job.with,
                secrets: job.secrets,
                strategy: renderJobStrategy(job.strategy),
            };
        }
        if (job.tools) {
            steps.push(...setupTools(job.tools));
        }
        const userDefinedSteps = util_1.kebabCaseKeys(_resolve_1.resolve(job.steps), false);
        steps.push(...userDefinedSteps);
        return {
            name: job.name,
            needs: arrayOrScalar(job.needs),
            "runs-on": arrayOrScalar(job.runsOn),
            permissions: util_1.kebabCaseKeys(job.permissions),
            environment: job.environment,
            concurrency: job.concurrency,
            outputs: renderJobOutputs(job.outputs),
            env: job.env,
            defaults: util_1.kebabCaseKeys(job.defaults),
            if: job.if,
            steps: steps.map(renderStep),
            "timeout-minutes": job.timeoutMinutes,
            strategy: renderJobStrategy(job.strategy),
            "continue-on-error": job.continueOnError,
            container: job.container,
            services: job.services,
        };
    }
    function renderJobOutputs(output) {
        if (output == null) {
            return undefined;
        }
        const rendered = {};
        for (const [name, { stepId, outputName }] of Object.entries(output)) {
            rendered[name] = `\${{ steps.${stepId}.outputs.${outputName} }}`;
        }
        return rendered;
    }
    function renderJobStrategy(strategy) {
        if (strategy == null) {
            return undefined;
        }
        const rendered = {
            "max-parallel": strategy.maxParallel,
            "fail-fast": strategy.failFast,
        };
        if (strategy.matrix) {
            const matrix = {
                include: strategy.matrix.include,
                exclude: strategy.matrix.exclude,
            };
            for (const [key, values] of Object.entries(strategy.matrix.domain ?? {})) {
                if (key in matrix) {
                    // A domain key was set to `include`, or `exclude`:
                    throw new Error(`Illegal job strategy matrix key: ${key}`);
                }
                matrix[key] = values;
            }
            rendered.matrix = matrix;
        }
        return rendered;
    }
    function renderStep(step) {
        return {
            name: step.name,
            id: step.id,
            if: step.if,
            uses: step.uses && actions.get(step.uses),
            env: step.env,
            run: step.run,
            with: step.with,
            "continue-on-error": step.continueOnError,
            "timeout-minutes": step.timeoutMinutes,
            "working-directory": step.workingDirectory,
        };
    }
}
function arrayOrScalar(arr) {
    if (arr == null || arr.length === 0) {
        return arr;
    }
    if (arr.length === 1) {
        return arr[0];
    }
    return arr;
}
function setupTools(tools) {
    const steps = [];
    if (tools.java) {
        steps.push({
            uses: "actions/setup-java@v3",
            with: { distribution: "temurin", "java-version": tools.java.version },
        });
    }
    if (tools.node) {
        steps.push({
            uses: "actions/setup-node@v3",
            with: { "node-version": tools.node.version },
        });
    }
    if (tools.python) {
        steps.push({
            uses: "actions/setup-python@v4",
            with: { "python-version": tools.python.version },
        });
    }
    if (tools.go) {
        steps.push({
            uses: "actions/setup-go@v3",
            with: { "go-version": tools.go.version },
        });
    }
    if (tools.dotnet) {
        steps.push({
            uses: "actions/setup-dotnet@v3",
            with: { "dotnet-version": tools.dotnet.version },
        });
    }
    return steps;
}
function verifyJobConstraints(jobs) {
    // verify that job has a "permissions" statement to ensure workflow can
    // operate in repos with default tokens set to readonly
    for (const [id, job] of Object.entries(jobs)) {
        if (!job.permissions) {
            throw new Error(`${id}: all workflow jobs must have a "permissions" clause to ensure workflow can operate in restricted repositories`);
        }
    }
    // verify that job has a "runsOn" statement to ensure a worker can be selected appropriately
    for (const [id, job] of Object.entries(jobs)) {
        if (!("uses" in job)) {
            if ("runsOn" in job && job.runsOn.length === 0) {
                throw new Error(`${id}: at least one runner selector labels must be provided in "runsOn" to ensure a runner instance can be selected`);
            }
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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