# -*- coding: utf-8 -*-
import pandas as pd
import numpy as np


from .ecg_hrv import ecg_hrv


def ecg_intervalrelated(data, sampling_rate=1000):
    """
    Performs ECG analysis on longer periods of data (typically > 10 seconds),
    such as resting-state data.

    Parameters
    ----------
    data : DataFrame, dict
        A DataFrame containing the different processed signal(s) as
        different columns, typically generated by `ecg_process()` or
        `bio_process()`. Can also take a dict containing sets of
        separately processed DataFrames.
    sampling_rate : int
        The sampling frequency of the signal (in Hz, i.e., samples/second).

    Returns
    -------
    DataFrame
        A dataframe containing the analyzed ECG features. The analyzed
        features consist of the following:
        - *"ECG_Rate_Mean"*: the mean heart rate.
        - *"ECG_HRV"*: the different heart rate variability metrices.
        See `ecg_hrv()` docstrings for details.

    See Also
    --------
    bio_process, ecg_eventrelated

    Examples
    ----------
    >>> import neurokit2 as nk
    >>> import pandas as pd
    >>>
    >>> # Download data
    >>> data = pd.read_csv("https://raw.githubusercontent.com/neuropsychology/NeuroKit/master/data/bio_resting_5min_100hz.csv")
    >>>
    >>> # Process the data
    >>> df, info = nk.ecg_process(data["ECG"], sampling_rate=100)
    >>>
    >>> # Single dataframe is passed
    >>> nk.ecg_intervalrelated(df)
    >>>
    >>> epochs = nk.epochs_create(df, events=[0, 15000], sampling_rate=100, epochs_end=20)
    >>> nk.ecg_intervalrelated(epochs)
    """
    intervals = {}

    # Format input
    if isinstance(data, pd.DataFrame):
        rate_cols = [col for col in data.columns if 'ECG_Rate' in col]
        if len(rate_cols) == 1:
            intervals["Rate_Mean"] = data[rate_cols[0]].values.mean()
            hrv = ecg_hrv(data, sampling_rate=sampling_rate)
            intervals["HRV_RMSSD"] = float(hrv["HRV_RMSSD"])
            intervals["HRV_MeanNN"] = float(hrv["HRV_MeanNN"])
            intervals["HRV_SDNN"] = float(hrv["HRV_SDNN"])
            intervals["HRV_SDSD"] = float(hrv["HRV_SDSD"])
            intervals["HRV_CVNN"] = float(hrv["HRV_CVNN"])
            intervals["HRV_CVSD"] = float(hrv["HRV_CVSD"])
            intervals["HRV_MedianNN"] = float(hrv["HRV_MedianNN"])
            intervals["HRV_MadNN"] = float(hrv["HRV_MadNN"])
            intervals["HRV_MCVNN"] = float(hrv["HRV_MCVNN"])
            intervals["HRV_pNN50"] = float(hrv["HRV_pNN50"])
            intervals["HRV_pNN20"] = float(hrv["HRV_pNN20"])
            intervals["HRV_TINN"] = float(hrv["HRV_TINN"])
            intervals["HRV_HTI"] = float(hrv["HRV_HTI"])
            intervals["HRV_ULF"] = float(hrv["HRV_ULF"])
            intervals["HRV_VLF"] = float(hrv["HRV_VLF"])
            intervals["HRV_LF"] = float(hrv["HRV_LF"])
            intervals["HRV_HF"] = float(hrv["HRV_HF"])
            intervals["HRV_VHF"] = float(hrv["HRV_VHF"])
            intervals["HRV_LFHF"] = float(hrv["HRV_LFHF"])
            intervals["HRV_LFn"] = float(hrv["HRV_LFn"])
            intervals["HRV_HFn"] = float(hrv["HRV_HFn"])
            intervals["HRV_LnHF"] = float(hrv["HRV_LnHF"])
            intervals["HRV_SD1"] = float(hrv["HRV_SD1"])
            intervals["HRV_SD2"] = float(hrv["HRV_SD2"])
            intervals["HRV_SD2SD1"] = float(hrv["HRV_SD2SD1"])
            intervals["HRV_CSI"] = float(hrv["HRV_CSI"])
            intervals["HRV_CVI"] = float(hrv["HRV_CVI"])
            intervals["HRV_CSI_Modified"] = float(hrv["HRV_CSI_Modified"])
            intervals["HRV_SampEn"] = float(hrv["HRV_SampEn"])
        else:
            raise ValueError("NeuroKit error: ecg_intervalrelated(): Wrong input,"
                             "we couldn't extract heart rate. Please make sure"
                             "your DataFrame contains an `ECG_Rate` column.")
        ecg_intervals = pd.DataFrame.from_dict(intervals,
                                               orient="index").T.add_prefix("ECG_")

    elif isinstance(data, dict):
        for index in data:
            intervals[index] = {}  # Initialize empty container

            # Rate
            intervals[index] = _ecg_intervalrelated_formatinput(data[index],
                                                                intervals[index])
        ecg_intervals = pd.DataFrame.from_dict(intervals, orient="index")

    return ecg_intervals

# =============================================================================
# Internals
# =============================================================================


def _ecg_intervalrelated_formatinput(interval, output={}):
    """Format input for dictionary
    """
    # Sanitize input
    colnames = interval.columns.values
    if len([i for i in colnames if "ECG_Rate" in i]) == 0:
        raise ValueError("NeuroKit error: ecg_intervalrelated(): Wrong input,"
                         "we couldn't extract heart rate. Please make sure"
                         "your DataFrame contains an `ECG_Rate` column.")
        return output

    signal = interval["ECG_Rate"].values
    output["ECG_Rate_Mean"] = np.mean(signal)

    return output
