#!/usr/bin/python3 -u


import os
import json
from time import sleep
import dateutil.parser
import sys

from loguru import logger
import zmq

from sudoisbot.sink import simplestate
from sudoisbot.network.sub import Subscriber, SubscriberTimedOutError


def as_bytes(astring):
    if isinstance(astring, bytes):
        return astring
    else:
        return astring.encode()


class ZFluxClient(object):
    def __init__(self, addr=None, topic=None):
        self.addr = addr
        self.topic = as_bytes(topic)

        self.context = zmq.Context()
        self.socket = self.context.socket(zmq.PUB)

        if addr:
            self.connect()

    def connect(self, addr=None):
        if not addr:
            addr = self.addr
        self.socket.connect(addr)
        logger.info(f"connected to: {addr}, emitting on topic: {self.topic}")

    def disconnect(self):
        self.socket.close()
        self.context.destroy()
        logger.debug("zflux client disconnected")

    def __enter__(self):
        if self.addr:
            self.connect(self.addr)
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        #self.disconnect()
        pass

    def send(self, msg):
        self.socket.send_multipart([self.topic, json.dumps(msg).encode()])

class Sink(object):
    def __init__(self, topics, write_path, zflux=None):
        self.zflux = zflux
        self.topics = topics
        self.setup_loggers(write_path)

        self.state_dir = write_path

    def setup_loggers(self, writepath):
        # change to 11 or 19 to show with debug logging
        logger.level("TXT", no=9, color="<yellow>")
        logger.level("SINK", no=11, color="<green>")

        for topic in self.topics:
            def matcher(topic):
                def inner(arg):
                    extra_topic = arg['extra'].get('topic', b"")
                    return extra_topic == as_bytes(topic)
                return inner


            logger.add(os.path.join(writepath, f"{topic}.txt"),
                       level="TXT", format="{message}",
                       filter=matcher(topic))

    def make_subscriber(self, addr):
        return Subscriber(addr, self.topics)

    def listen(self, addr):
        try:
            # with self.make_subscriber(addr) as sub:
            #     for topic, msg in sub.recv():
            #         self.handle_msg(topic, msg)
            #
            # commented out because testing to no gracefully disconnected to get
            # publishers to buffer when sink is dead
            sub = self.make_subscriber(addr)
            sub.connect()
            for topic, msg, cached in sub.recv():
                if cached:
                    logger.info(f"got a cached {topic} message from {msg['time']}")
                self.handle_msg(topic, msg)

        except zmq.error.Again:
            logger.info(f"timeout after {sub.rcvtimeo_secs}s..")
            raise SubscriberTimedOutError

    def handle_msg(self, topic, msg):
        self.log(topic, msg)

        #self.update_db(topic, msg)         # todo: keep records in sql
        self.append_file(topic, msg)
        self.update_state(topic, msg)
        self.send_zflux(msg)
        if msg['tags']['name'] == "study":
            logger.warning(msg)

    def update_state(self, topic, newstate):
        measurement = newstate['measurement']
        filename = os.path.join(self.state_dir, f"{measurement}-state.json")
        simplestate.update_state(newstate, filename)

    def send_zflux(self, msg):
        if self.zflux:
            self.zflux.send(msg)

    def append_file(self, topic, msg):
        logger.bind(topic=topic).log("TXT", json.dumps(msg))

    def log(self, topic, msg):
        measurement = msg['measurement']


        name = msg['tags']['name']
        if 'value' in msg['fields']:
            value = f": {msg['fields']['value']}"
        else:
            value = ""
        logger.log("SINK", f"{topic}: {measurement} from '{name}'{value}")





def main(args, config):

    with ZFluxClient(topic=config['zflux']['topic']) as zflux:
        zflux.connect(config['zflux']['addr'])

        write_path = args.write_path or config['sink']['write_path']

        sink = Sink(config['sink']['topics'], write_path, zflux)
        while True:
            try:
                addr = config['sink']['addr']
                sink.listen(addr)
            except SubscriberTimedOutError:
                sleep(1.0)
                logger.info("reconnecting")
            except KeyboardInterrupt:
                logger.info("ok ill leave then")
                return

def suicide_snail(timestamp, max_delay):
    # suicide snail (move to common sub code?)
    delay = datetime.now() - datetime.fromisoformat(timestamp)
    if  min(delay.seconds, 0) > max_delay:
        logger.error(f"suicide snail: {delay.seconds} secs")
        raise SystemExit("suicide snail")



#from sudoisbot.sink import models
#from playhouse.db_url import connect
#from peewee import IntegrityError
#     db = connect(dburl)
#     models.db.initialize(db)
#         try:
#             with models.db:
#                 models.Temps.create(timestamp=j['timestamp'],
#                                     name=j['name'],
#                                     temp=j['temp'],
#                                     extra=extra)
#         except IntegrityError as e:
#             logger.error(e)
