#!/usr/bin/env python


import unittest


from .age import EstimatedAge
from .context import Context
from .individual import AgeSexStature, BurialInfo, Individual, LongBoneMeasurement, OsteologicalSex
from .joints import Joints
from .left_right import LeftRight
from .mouth import Mouth
from .occupational_markers import OccupationalMarkers
from .sex import Sex
from .trauma import Trauma


class OsteologicalSexTest(unittest.TestCase):
    def test_to_pd_series(self):
        pelvic = Sex.MALE
        cranium = None
        combined = Sex.MALE_ASSUMED
        os = OsteologicalSex(pelvic, cranium, combined)

        df = os.to_pd_data_frame('id1')

        self.assertEqual(df.to_json(orient='records'), '[{"pelvic_cat":"MALE","pelvic_val":100,"pelvic_bin_cat":"MALE","pelvic_bin_val":100,"combined_cat":"MALE_ASSUMED","combined_val":80,"combined_bin_cat":"MALE","combined_bin_val":100}]')


class AgeSexStatureTest(unittest.TestCase):
    def test_to_pd_series(self):
        os = OsteologicalSex(Sex.MALE, None, Sex.MALE_ASSUMED)
        age = EstimatedAge.empty()
        femur = LongBoneMeasurement.empty_lr()
        humerus = LongBoneMeasurement.empty_lr()
        tibia = LongBoneMeasurement.empty_lr()
        stature = ''
        body_mass = 'None'

        ass = AgeSexStature(os, age, femur, humerus, tibia, stature, body_mass)

        df = ass.to_pd_data_frame('id1', prefix='ass_')

        self.assertEqual(df.to_json(orient='records'), '[{"id":"id1","ass_stature":"","ass_body_mass":"None","ass_age_category":"UNKNOWN","ass_age_category_val":0,"ass_age_ranged_val":null,"ass_femur_left_max":null,"ass_femur_left_bi":null,"ass_femur_left_head":null,"ass_femur_left_distal":null,"ass_femur_right_max":null,"ass_femur_right_bi":null,"ass_femur_right_head":null,"ass_femur_right_distal":null,"ass_femur_avg_max":null,"ass_femur_avg_bi":null,"ass_femur_avg_head":null,"ass_femur_avg_distal":null,"ass_humerus_left_max":null,"ass_humerus_left_bi":null,"ass_humerus_left_head":null,"ass_humerus_left_distal":null,"ass_humerus_right_max":null,"ass_humerus_right_bi":null,"ass_humerus_right_head":null,"ass_humerus_right_distal":null,"ass_humerus_avg_max":null,"ass_humerus_avg_bi":null,"ass_humerus_avg_head":null,"ass_humerus_avg_distal":null,"ass_tibia_left_max":null,"ass_tibia_left_bi":null,"ass_tibia_left_head":null,"ass_tibia_left_distal":null,"ass_tibia_right_max":null,"ass_tibia_right_bi":null,"ass_tibia_right_head":null,"ass_tibia_right_distal":null,"ass_tibia_avg_max":null,"ass_tibia_avg_bi":null,"ass_tibia_avg_head":null,"ass_tibia_avg_distal":null,"ass_osteological_sex_pelvic_cat":"MALE","ass_osteological_sex_pelvic_val":100,"ass_osteological_sex_pelvic_bin_cat":"MALE","ass_osteological_sex_pelvic_bin_val":100,"ass_osteological_sex_combined_cat":"MALE_ASSUMED","ass_osteological_sex_combined_val":80,"ass_osteological_sex_combined_bin_cat":"MALE","ass_osteological_sex_combined_bin_val":100}]')


class LongBoneMeasurementTest(unittest.TestCase):
    def test_lr_avg(self):
        left_femur = LongBoneMeasurement(1.0, None, 1.0, None)
        right_femur = LongBoneMeasurement(2.0, 2.0, None, None)

        femur = LeftRight(left_femur, right_femur)
        avg_femur = femur.avg()
        self.assertEqual(avg_femur.max, 1.5)
        self.assertEqual(avg_femur.bi, 2.0)
        self.assertEqual(avg_femur.head, 1.0)
        self.assertEqual(avg_femur.distal, None)

        femur = LeftRight(LongBoneMeasurement.empty(), right_femur)
        avg_femur = femur.avg()
        self.assertEqual(avg_femur, right_femur)

        femur = LeftRight(left_femur, LongBoneMeasurement.empty())
        avg_femur = femur.avg()
        self.assertEqual(avg_femur, left_femur)

        femur = LeftRight(LongBoneMeasurement.empty(), LongBoneMeasurement.empty())
        avg_femur = femur.avg()
        self.assertEqual(avg_femur, LongBoneMeasurement.empty())


class IndividualTest(unittest.TestCase):
    def test_basic(self):
        burial_info = BurialInfo('site_name', 'site_id')
        age_sex_sature = AgeSexStature.empty()
        mouth = Mouth.empty()
        occupational_markers = OccupationalMarkers.empty()
        joints = Joints.empty()
        trauma = Trauma.empty()
        context = Context.empty()

        individual = Individual('id_1', burial_info, age_sex_sature, mouth, occupational_markers, joints, trauma, context)

        self.assertEqual(individual.id, 'id_1')

    def test_to_pd_data_frame(self):
        burial_info = BurialInfo('site_name', 'site_id')
        age_sex_sature = AgeSexStature.empty()
        mouth = Mouth.empty()
        occupational_markers = OccupationalMarkers.empty()
        joints = Joints.empty()
        trauma = Trauma.empty()
        context = Context({'spear': True, 'pot': False})

        individual = Individual('id_1', burial_info, age_sex_sature, mouth, occupational_markers, joints, trauma, context)
        df = individual.to_pd_data_frame()

        self.assertEqual(df.to_json(orient='records'), '[{"id":"id_1","site_name":"site_name","site_id":"site_id","mouth_number_of_teeth":0,"mouth_tooth_0_tooth":"NA","mouth_tooth_0_tooth_val":null,"mouth_tooth_0_calculus":"NA","mouth_tooth_0_calculus_val":null,"mouth_tooth_0_eh":"NA","mouth_tooth_0_eh_val":null,"mouth_tooth_0_cavities":"NA","mouth_tooth_0_cavities_val":null,"mouth_tooth_0_abcess":"NA","mouth_tooth_0_abcess_val":null,"mouth_tooth_1_tooth":"NA","mouth_tooth_1_tooth_val":null,"mouth_tooth_1_calculus":"NA","mouth_tooth_1_calculus_val":null,"mouth_tooth_1_eh":"NA","mouth_tooth_1_eh_val":null,"mouth_tooth_1_cavities":"NA","mouth_tooth_1_cavities_val":null,"mouth_tooth_1_abcess":"NA","mouth_tooth_1_abcess_val":null,"mouth_tooth_2_tooth":"NA","mouth_tooth_2_tooth_val":null,"mouth_tooth_2_calculus":"NA","mouth_tooth_2_calculus_val":null,"mouth_tooth_2_eh":"NA","mouth_tooth_2_eh_val":null,"mouth_tooth_2_cavities":"NA","mouth_tooth_2_cavities_val":null,"mouth_tooth_2_abcess":"NA","mouth_tooth_2_abcess_val":null,"mouth_tooth_3_tooth":"NA","mouth_tooth_3_tooth_val":null,"mouth_tooth_3_calculus":"NA","mouth_tooth_3_calculus_val":null,"mouth_tooth_3_eh":"NA","mouth_tooth_3_eh_val":null,"mouth_tooth_3_cavities":"NA","mouth_tooth_3_cavities_val":null,"mouth_tooth_3_abcess":"NA","mouth_tooth_3_abcess_val":null,"mouth_tooth_4_tooth":"NA","mouth_tooth_4_tooth_val":null,"mouth_tooth_4_calculus":"NA","mouth_tooth_4_calculus_val":null,"mouth_tooth_4_eh":"NA","mouth_tooth_4_eh_val":null,"mouth_tooth_4_cavities":"NA","mouth_tooth_4_cavities_val":null,"mouth_tooth_4_abcess":"NA","mouth_tooth_4_abcess_val":null,"mouth_tooth_5_tooth":"NA","mouth_tooth_5_tooth_val":null,"mouth_tooth_5_calculus":"NA","mouth_tooth_5_calculus_val":null,"mouth_tooth_5_eh":"NA","mouth_tooth_5_eh_val":null,"mouth_tooth_5_cavities":"NA","mouth_tooth_5_cavities_val":null,"mouth_tooth_5_abcess":"NA","mouth_tooth_5_abcess_val":null,"mouth_tooth_6_tooth":"NA","mouth_tooth_6_tooth_val":null,"mouth_tooth_6_calculus":"NA","mouth_tooth_6_calculus_val":null,"mouth_tooth_6_eh":"NA","mouth_tooth_6_eh_val":null,"mouth_tooth_6_cavities":"NA","mouth_tooth_6_cavities_val":null,"mouth_tooth_6_abcess":"NA","mouth_tooth_6_abcess_val":null,"mouth_tooth_7_tooth":"NA","mouth_tooth_7_tooth_val":null,"mouth_tooth_7_calculus":"NA","mouth_tooth_7_calculus_val":null,"mouth_tooth_7_eh":"NA","mouth_tooth_7_eh_val":null,"mouth_tooth_7_cavities":"NA","mouth_tooth_7_cavities_val":null,"mouth_tooth_7_abcess":"NA","mouth_tooth_7_abcess_val":null,"mouth_tooth_8_tooth":"NA","mouth_tooth_8_tooth_val":null,"mouth_tooth_8_calculus":"NA","mouth_tooth_8_calculus_val":null,"mouth_tooth_8_eh":"NA","mouth_tooth_8_eh_val":null,"mouth_tooth_8_cavities":"NA","mouth_tooth_8_cavities_val":null,"mouth_tooth_8_abcess":"NA","mouth_tooth_8_abcess_val":null,"mouth_tooth_9_tooth":"NA","mouth_tooth_9_tooth_val":null,"mouth_tooth_9_calculus":"NA","mouth_tooth_9_calculus_val":null,"mouth_tooth_9_eh":"NA","mouth_tooth_9_eh_val":null,"mouth_tooth_9_cavities":"NA","mouth_tooth_9_cavities_val":null,"mouth_tooth_9_abcess":"NA","mouth_tooth_9_abcess_val":null,"mouth_tooth_10_tooth":"NA","mouth_tooth_10_tooth_val":null,"mouth_tooth_10_calculus":"NA","mouth_tooth_10_calculus_val":null,"mouth_tooth_10_eh":"NA","mouth_tooth_10_eh_val":null,"mouth_tooth_10_cavities":"NA","mouth_tooth_10_cavities_val":null,"mouth_tooth_10_abcess":"NA","mouth_tooth_10_abcess_val":null,"mouth_tooth_11_tooth":"NA","mouth_tooth_11_tooth_val":null,"mouth_tooth_11_calculus":"NA","mouth_tooth_11_calculus_val":null,"mouth_tooth_11_eh":"NA","mouth_tooth_11_eh_val":null,"mouth_tooth_11_cavities":"NA","mouth_tooth_11_cavities_val":null,"mouth_tooth_11_abcess":"NA","mouth_tooth_11_abcess_val":null,"mouth_tooth_12_tooth":"NA","mouth_tooth_12_tooth_val":null,"mouth_tooth_12_calculus":"NA","mouth_tooth_12_calculus_val":null,"mouth_tooth_12_eh":"NA","mouth_tooth_12_eh_val":null,"mouth_tooth_12_cavities":"NA","mouth_tooth_12_cavities_val":null,"mouth_tooth_12_abcess":"NA","mouth_tooth_12_abcess_val":null,"mouth_tooth_13_tooth":"NA","mouth_tooth_13_tooth_val":null,"mouth_tooth_13_calculus":"NA","mouth_tooth_13_calculus_val":null,"mouth_tooth_13_eh":"NA","mouth_tooth_13_eh_val":null,"mouth_tooth_13_cavities":"NA","mouth_tooth_13_cavities_val":null,"mouth_tooth_13_abcess":"NA","mouth_tooth_13_abcess_val":null,"mouth_tooth_14_tooth":"NA","mouth_tooth_14_tooth_val":null,"mouth_tooth_14_calculus":"NA","mouth_tooth_14_calculus_val":null,"mouth_tooth_14_eh":"NA","mouth_tooth_14_eh_val":null,"mouth_tooth_14_cavities":"NA","mouth_tooth_14_cavities_val":null,"mouth_tooth_14_abcess":"NA","mouth_tooth_14_abcess_val":null,"mouth_tooth_15_tooth":"NA","mouth_tooth_15_tooth_val":null,"mouth_tooth_15_calculus":"NA","mouth_tooth_15_calculus_val":null,"mouth_tooth_15_eh":"NA","mouth_tooth_15_eh_val":null,"mouth_tooth_15_cavities":"NA","mouth_tooth_15_cavities_val":null,"mouth_tooth_15_abcess":"NA","mouth_tooth_15_abcess_val":null,"mouth_tooth_16_tooth":"NA","mouth_tooth_16_tooth_val":null,"mouth_tooth_16_calculus":"NA","mouth_tooth_16_calculus_val":null,"mouth_tooth_16_eh":"NA","mouth_tooth_16_eh_val":null,"mouth_tooth_16_cavities":"NA","mouth_tooth_16_cavities_val":null,"mouth_tooth_16_abcess":"NA","mouth_tooth_16_abcess_val":null,"mouth_tooth_17_tooth":"NA","mouth_tooth_17_tooth_val":null,"mouth_tooth_17_calculus":"NA","mouth_tooth_17_calculus_val":null,"mouth_tooth_17_eh":"NA","mouth_tooth_17_eh_val":null,"mouth_tooth_17_cavities":"NA","mouth_tooth_17_cavities_val":null,"mouth_tooth_17_abcess":"NA","mouth_tooth_17_abcess_val":null,"mouth_tooth_18_tooth":"NA","mouth_tooth_18_tooth_val":null,"mouth_tooth_18_calculus":"NA","mouth_tooth_18_calculus_val":null,"mouth_tooth_18_eh":"NA","mouth_tooth_18_eh_val":null,"mouth_tooth_18_cavities":"NA","mouth_tooth_18_cavities_val":null,"mouth_tooth_18_abcess":"NA","mouth_tooth_18_abcess_val":null,"mouth_tooth_19_tooth":"NA","mouth_tooth_19_tooth_val":null,"mouth_tooth_19_calculus":"NA","mouth_tooth_19_calculus_val":null,"mouth_tooth_19_eh":"NA","mouth_tooth_19_eh_val":null,"mouth_tooth_19_cavities":"NA","mouth_tooth_19_cavities_val":null,"mouth_tooth_19_abcess":"NA","mouth_tooth_19_abcess_val":null,"mouth_tooth_20_tooth":"NA","mouth_tooth_20_tooth_val":null,"mouth_tooth_20_calculus":"NA","mouth_tooth_20_calculus_val":null,"mouth_tooth_20_eh":"NA","mouth_tooth_20_eh_val":null,"mouth_tooth_20_cavities":"NA","mouth_tooth_20_cavities_val":null,"mouth_tooth_20_abcess":"NA","mouth_tooth_20_abcess_val":null,"mouth_tooth_21_tooth":"NA","mouth_tooth_21_tooth_val":null,"mouth_tooth_21_calculus":"NA","mouth_tooth_21_calculus_val":null,"mouth_tooth_21_eh":"NA","mouth_tooth_21_eh_val":null,"mouth_tooth_21_cavities":"NA","mouth_tooth_21_cavities_val":null,"mouth_tooth_21_abcess":"NA","mouth_tooth_21_abcess_val":null,"mouth_tooth_22_tooth":"NA","mouth_tooth_22_tooth_val":null,"mouth_tooth_22_calculus":"NA","mouth_tooth_22_calculus_val":null,"mouth_tooth_22_eh":"NA","mouth_tooth_22_eh_val":null,"mouth_tooth_22_cavities":"NA","mouth_tooth_22_cavities_val":null,"mouth_tooth_22_abcess":"NA","mouth_tooth_22_abcess_val":null,"mouth_tooth_23_tooth":"NA","mouth_tooth_23_tooth_val":null,"mouth_tooth_23_calculus":"NA","mouth_tooth_23_calculus_val":null,"mouth_tooth_23_eh":"NA","mouth_tooth_23_eh_val":null,"mouth_tooth_23_cavities":"NA","mouth_tooth_23_cavities_val":null,"mouth_tooth_23_abcess":"NA","mouth_tooth_23_abcess_val":null,"mouth_tooth_24_tooth":"NA","mouth_tooth_24_tooth_val":null,"mouth_tooth_24_calculus":"NA","mouth_tooth_24_calculus_val":null,"mouth_tooth_24_eh":"NA","mouth_tooth_24_eh_val":null,"mouth_tooth_24_cavities":"NA","mouth_tooth_24_cavities_val":null,"mouth_tooth_24_abcess":"NA","mouth_tooth_24_abcess_val":null,"mouth_tooth_25_tooth":"NA","mouth_tooth_25_tooth_val":null,"mouth_tooth_25_calculus":"NA","mouth_tooth_25_calculus_val":null,"mouth_tooth_25_eh":"NA","mouth_tooth_25_eh_val":null,"mouth_tooth_25_cavities":"NA","mouth_tooth_25_cavities_val":null,"mouth_tooth_25_abcess":"NA","mouth_tooth_25_abcess_val":null,"mouth_tooth_26_tooth":"NA","mouth_tooth_26_tooth_val":null,"mouth_tooth_26_calculus":"NA","mouth_tooth_26_calculus_val":null,"mouth_tooth_26_eh":"NA","mouth_tooth_26_eh_val":null,"mouth_tooth_26_cavities":"NA","mouth_tooth_26_cavities_val":null,"mouth_tooth_26_abcess":"NA","mouth_tooth_26_abcess_val":null,"mouth_tooth_27_tooth":"NA","mouth_tooth_27_tooth_val":null,"mouth_tooth_27_calculus":"NA","mouth_tooth_27_calculus_val":null,"mouth_tooth_27_eh":"NA","mouth_tooth_27_eh_val":null,"mouth_tooth_27_cavities":"NA","mouth_tooth_27_cavities_val":null,"mouth_tooth_27_abcess":"NA","mouth_tooth_27_abcess_val":null,"mouth_tooth_28_tooth":"NA","mouth_tooth_28_tooth_val":null,"mouth_tooth_28_calculus":"NA","mouth_tooth_28_calculus_val":null,"mouth_tooth_28_eh":"NA","mouth_tooth_28_eh_val":null,"mouth_tooth_28_cavities":"NA","mouth_tooth_28_cavities_val":null,"mouth_tooth_28_abcess":"NA","mouth_tooth_28_abcess_val":null,"mouth_tooth_29_tooth":"NA","mouth_tooth_29_tooth_val":null,"mouth_tooth_29_calculus":"NA","mouth_tooth_29_calculus_val":null,"mouth_tooth_29_eh":"NA","mouth_tooth_29_eh_val":null,"mouth_tooth_29_cavities":"NA","mouth_tooth_29_cavities_val":null,"mouth_tooth_29_abcess":"NA","mouth_tooth_29_abcess_val":null,"mouth_tooth_30_tooth":"NA","mouth_tooth_30_tooth_val":null,"mouth_tooth_30_calculus":"NA","mouth_tooth_30_calculus_val":null,"mouth_tooth_30_eh":"NA","mouth_tooth_30_eh_val":null,"mouth_tooth_30_cavities":"NA","mouth_tooth_30_cavities_val":null,"mouth_tooth_30_abcess":"NA","mouth_tooth_30_abcess_val":null,"mouth_tooth_31_tooth":"NA","mouth_tooth_31_tooth_val":null,"mouth_tooth_31_calculus":"NA","mouth_tooth_31_calculus_val":null,"mouth_tooth_31_eh":"NA","mouth_tooth_31_eh_val":null,"mouth_tooth_31_cavities":"NA","mouth_tooth_31_cavities_val":null,"mouth_tooth_31_abcess":"NA","mouth_tooth_31_abcess_val":null,"mouth_tooth_mean":null,"mouth_tooth_max":null,"mouth_tooth_min":null,"mouth_tooth_count":0,"mouth_calculus_mean":null,"mouth_calculus_max":null,"mouth_calculus_min":null,"mouth_calculus_count":0,"mouth_eh_mean":null,"mouth_eh_max":null,"mouth_eh_min":null,"mouth_eh_count":0,"mouth_cavities_mean":null,"mouth_cavities_max":null,"mouth_cavities_min":null,"mouth_cavities_count":0,"mouth_abcess_mean":null,"mouth_abcess_max":null,"mouth_abcess_min":null,"mouth_abcess_count":0,"om_c_trapezius_left":null,"om_c_trapezius_right":null,"om_c_trapezius_avg":null,"om_c_trapezius_avg_val":null,"om_c_o_deltiod_left":null,"om_c_o_deltiod_right":null,"om_c_o_deltiod_avg":null,"om_c_o_deltiod_avg_val":null,"om_c_o_pectoralis_major_left":null,"om_c_o_pectoralis_major_right":null,"om_c_o_pectoralis_major_avg":null,"om_c_o_pectoralis_major_avg_val":null,"om_c_costoclaviclar_lig_left":null,"om_c_costoclaviclar_lig_right":null,"om_c_costoclaviclar_lig_avg":null,"om_c_costoclaviclar_lig_avg_val":null,"om_c_subcalvius_left":null,"om_c_subcalvius_right":null,"om_c_subcalvius_avg":null,"om_c_subcalvius_avg_val":null,"om_c_conoid_lig_left":null,"om_c_conoid_lig_right":null,"om_c_conoid_lig_avg":null,"om_c_conoid_lig_avg_val":null,"om_c_trapezoid_lig_left":null,"om_c_trapezoid_lig_right":null,"om_c_trapezoid_lig_avg":null,"om_c_trapezoid_lig_avg_val":null,"om_s_pectoralis_minor_left":null,"om_s_pectoralis_minor_right":null,"om_s_pectoralis_minor_avg":null,"om_s_pectoralis_minor_avg_val":null,"om_s_serratus_anterior_left":null,"om_s_serratus_anterior_right":null,"om_s_serratus_anterior_avg":null,"om_s_serratus_anterior_avg_val":null,"om_s_triceps_long_head_left":null,"om_s_triceps_long_head_right":null,"om_s_triceps_long_head_avg":null,"om_s_triceps_long_head_avg_val":null,"om_s_trapezius_left":null,"om_s_trapezius_right":null,"om_s_trapezius_avg":null,"om_s_trapezius_avg_val":null,"om_h_subscapularis_left":null,"om_h_subscapularis_right":null,"om_h_subscapularis_avg":null,"om_h_subscapularis_avg_val":null,"om_h_teres_major_left":null,"om_h_teres_major_right":null,"om_h_teres_major_avg":null,"om_h_teres_major_avg_val":null,"om_h_latissimus_dorsi_left":null,"om_h_latissimus_dorsi_right":null,"om_h_latissimus_dorsi_avg":null,"om_h_latissimus_dorsi_avg_val":null,"om_h_pectoralis_major_left":null,"om_h_pectoralis_major_right":null,"om_h_pectoralis_major_avg":null,"om_h_pectoralis_major_avg_val":null,"om_h_deltoid_left":null,"om_h_deltoid_right":null,"om_h_deltoid_avg":null,"om_h_deltoid_avg_val":null,"om_h_coracobrachialis_left":null,"om_h_coracobrachialis_right":null,"om_h_coracobrachialis_avg":null,"om_h_coracobrachialis_avg_val":null,"om_h_supraspinatus_left":null,"om_h_supraspinatus_right":null,"om_h_supraspinatus_avg":null,"om_h_supraspinatus_avg_val":null,"om_h_infraspinatus_left":null,"om_h_infraspinatus_right":null,"om_h_infraspinatus_avg":null,"om_h_infraspinatus_avg_val":null,"om_h_teres_minor_left":null,"om_h_teres_minor_right":null,"om_h_teres_minor_avg":null,"om_h_teres_minor_avg_val":null,"om_h_o_extensor_left":null,"om_h_o_extensor_right":null,"om_h_o_extensor_avg":null,"om_h_o_extensor_avg_val":null,"om_h_o_flexor_left":null,"om_h_o_flexor_right":null,"om_h_o_flexor_avg":null,"om_h_o_flexor_avg_val":null,"om_u_brachialis_left":null,"om_u_brachialis_right":null,"om_u_brachialis_avg":null,"om_u_brachialis_avg_val":null,"om_u_o_pronator_quadrataus_left":null,"om_u_o_pronator_quadrataus_right":null,"om_u_o_pronator_quadrataus_avg":null,"om_u_o_pronator_quadrataus_avg_val":null,"om_u_triceps_brachii_left":null,"om_u_triceps_brachii_right":null,"om_u_triceps_brachii_avg":null,"om_u_triceps_brachii_avg_val":null,"om_u_anconeus_left":null,"om_u_anconeus_right":null,"om_u_anconeus_avg":null,"om_u_anconeus_avg_val":null,"om_u_o_supinator_left":null,"om_u_o_supinator_right":null,"om_u_o_supinator_avg":null,"om_u_o_supinator_avg_val":null,"om_r_biceps_brachii_left":null,"om_r_biceps_brachii_right":null,"om_r_biceps_brachii_avg":null,"om_r_biceps_brachii_avg_val":null,"om_r_supinator_left":null,"om_r_supinator_right":null,"om_r_supinator_avg":null,"om_r_supinator_avg_val":null,"om_r_pronator_teres_left":null,"om_r_pronator_teres_right":null,"om_r_pronator_teres_avg":null,"om_r_pronator_teres_avg_val":null,"om_r_pronator_quadratus_left":null,"om_r_pronator_quadratus_right":null,"om_r_pronator_quadratus_avg":null,"om_r_pronator_quadratus_avg_val":null,"om_r_brachoradialis_left":null,"om_r_brachoradialis_right":null,"om_r_brachoradialis_avg":null,"om_r_brachoradialis_avg_val":null,"om_f_gluteus_minimus_left":null,"om_f_gluteus_minimus_right":null,"om_f_gluteus_minimus_avg":null,"om_f_gluteus_minimus_avg_val":null,"om_f_gluteus_medius_left":null,"om_f_gluteus_medius_right":null,"om_f_gluteus_medius_avg":null,"om_f_gluteus_medius_avg_val":null,"om_f_piriformus_left":null,"om_f_piriformus_right":null,"om_f_piriformus_avg":null,"om_f_piriformus_avg_val":null,"om_f_obturator_internus_left":null,"om_f_obturator_internus_right":null,"om_f_obturator_internus_avg":null,"om_f_obturator_internus_avg_val":null,"om_f_obturator_externus_left":null,"om_f_obturator_externus_right":null,"om_f_obturator_externus_avg":null,"om_f_obturator_externus_avg_val":null,"om_f_quadratis_femoris_left":null,"om_f_quadratis_femoris_right":null,"om_f_quadratis_femoris_avg":null,"om_f_quadratis_femoris_avg_val":null,"om_f_ilioposas_left":null,"om_f_ilioposas_right":null,"om_f_ilioposas_avg":null,"om_f_ilioposas_avg_val":null,"om_f_gluteus_maximus_left":null,"om_f_gluteus_maximus_right":null,"om_f_gluteus_maximus_avg":null,"om_f_gluteus_maximus_avg_val":null,"om_f_pectineus_left":null,"om_f_pectineus_right":null,"om_f_pectineus_avg":null,"om_f_pectineus_avg_val":null,"om_f_o_vastus_medialis_left":null,"om_f_o_vastus_medialis_right":null,"om_f_o_vastus_medialis_avg":null,"om_f_o_vastus_medialis_avg_val":null,"om_f_o_vastus_lateralis_left":null,"om_f_o_vastus_lateralis_right":null,"om_f_o_vastus_lateralis_avg":null,"om_f_o_vastus_lateralis_avg_val":null,"om_o_adductor_magnus_left":null,"om_o_adductor_magnus_right":null,"om_o_adductor_magnus_avg":null,"om_o_adductor_magnus_avg_val":null,"om_f_o_gastrocnemius_left":null,"om_f_o_gastrocnemius_right":null,"om_f_o_gastrocnemius_avg":null,"om_f_o_gastrocnemius_avg_val":null,"om_f_o_plantaris_left":null,"om_f_o_plantaris_right":null,"om_f_o_plantaris_avg":null,"om_f_o_plantaris_avg_val":null,"om_f_o_popliteus_left":null,"om_f_o_popliteus_right":null,"om_f_o_popliteus_avg":null,"om_f_o_popliteus_avg_val":null,"om_t_tensor_fascia_latae_left":null,"om_t_tensor_fascia_latae_right":null,"om_t_tensor_fascia_latae_avg":null,"om_t_tensor_fascia_latae_avg_val":null,"om_t_quadriceps_left":null,"om_t_quadriceps_right":null,"om_t_quadriceps_avg":null,"om_t_quadriceps_avg_val":null,"om_t_sartorius_left":null,"om_t_sartorius_right":null,"om_t_sartorius_avg":null,"om_t_sartorius_avg_val":null,"om_t_gracilis_left":null,"om_t_gracilis_right":null,"om_t_gracilis_avg":null,"om_t_gracilis_avg_val":null,"om_t_semitendinosus_left":null,"om_t_semitendinosus_right":null,"om_t_semitendinosus_avg":null,"om_t_semitendinosus_avg_val":null,"om_t_o_tibialus_anterior_left":null,"om_t_o_tibialus_anterior_right":null,"om_t_o_tibialus_anterior_avg":null,"om_t_o_tibialus_anterior_avg_val":null,"om_t_biceps_femoris_left":null,"om_t_biceps_femoris_right":null,"om_t_biceps_femoris_avg":null,"om_t_biceps_femoris_avg_val":null,"om_t_semimembranosus_left":null,"om_t_semimembranosus_right":null,"om_t_semimembranosus_avg":null,"om_t_semimembranosus_avg_val":null,"om_t_popliteus_left":null,"om_t_popliteus_right":null,"om_t_popliteus_avg":null,"om_t_popliteus_avg_val":null,"om_t_o_soleus_left":null,"om_t_o_soleus_right":null,"om_t_o_soleus_avg":null,"om_t_o_soleus_avg_val":null,"om_t_o_tibialis_posterior_left":null,"om_t_o_tibialis_posterior_right":null,"om_t_o_tibialis_posterior_avg":null,"om_t_o_tibialis_posterior_avg_val":null,"om_t_o_flexor_digitorium_left":null,"om_t_o_flexor_digitorium_right":null,"om_t_o_flexor_digitorium_avg":null,"om_t_o_flexor_digitorium_avg_val":null,"om_f_biceps_femoris_left":null,"om_f_biceps_femoris_right":null,"om_f_biceps_femoris_avg":null,"om_f_biceps_femoris_avg_val":null,"om_f_o_extensor_muscles_left":null,"om_f_o_extensor_muscles_right":null,"om_f_o_extensor_muscles_avg":null,"om_f_o_extensor_muscles_avg_val":null,"om_f_o_flexor_muscles_left":null,"om_f_o_flexor_muscles_right":null,"om_f_o_flexor_muscles_avg":null,"om_f_o_flexor_muscles_avg_val":null,"om_f_o_peroneus_longus_left":null,"om_f_o_peroneus_longus_right":null,"om_f_o_peroneus_longus_avg":null,"om_f_o_peroneus_longus_avg_val":null,"om_f_o_peronus_brevis_left":null,"om_f_o_peronus_brevis_right":null,"om_f_o_peronus_brevis_avg":null,"om_f_o_peronus_brevis_avg_val":null,"om_f_o_soleus_left":null,"om_f_o_soleus_right":null,"om_f_o_soleus_avg":null,"om_f_o_soleus_avg_val":null,"om_p_quadriceps_left":null,"om_p_quadriceps_right":null,"om_p_quadriceps_avg":null,"om_p_quadriceps_avg_val":null,"om_c_achilles_left":null,"om_c_achilles_right":null,"om_c_achilles_avg":null,"om_c_achilles_avg_val":null,"om_mean":null,"om_max":null,"om_min":null,"om_count":0,"joints_shoulder_left":null,"joints_shoulder_right":null,"joints_shoulder_avg":null,"joints_elbow_left":null,"joints_elbow_right":null,"joints_elbow_avg":null,"joints_wrist_left":null,"joints_wrist_right":null,"joints_wrist_avg":null,"joints_hip_left":null,"joints_hip_right":null,"joints_hip_avg":null,"joints_knee_left":null,"joints_knee_right":null,"joints_knee_avg":null,"joints_ankle_left":null,"joints_ankle_right":null,"joints_ankle_avg":null,"joints_sacro_illiac":null,"joints_sacro_illiac_avg":null,"joints_c1_3":null,"joints_c1_3_avg":null,"joints_c4_7":null,"joints_c4_7_avg":null,"joints_t1_4":null,"joints_t1_4_avg":null,"joints_t5_8":null,"joints_t5_8_avg":null,"joints_t9_12":null,"joints_t9_12_avg":null,"joints_l1_5":null,"joints_l1_5_avg":null,"joints_mean":null,"joints_max":null,"joints_min":null,"joints_count":0,"ass_stature":"","ass_body_mass":"","ass_age_category":"UNKNOWN","ass_age_category_val":0,"ass_age_ranged_val":null,"ass_femur_left_max":null,"ass_femur_left_bi":null,"ass_femur_left_head":null,"ass_femur_left_distal":null,"ass_femur_right_max":null,"ass_femur_right_bi":null,"ass_femur_right_head":null,"ass_femur_right_distal":null,"ass_femur_avg_max":null,"ass_femur_avg_bi":null,"ass_femur_avg_head":null,"ass_femur_avg_distal":null,"ass_humerus_left_max":null,"ass_humerus_left_bi":null,"ass_humerus_left_head":null,"ass_humerus_left_distal":null,"ass_humerus_right_max":null,"ass_humerus_right_bi":null,"ass_humerus_right_head":null,"ass_humerus_right_distal":null,"ass_humerus_avg_max":null,"ass_humerus_avg_bi":null,"ass_humerus_avg_head":null,"ass_humerus_avg_distal":null,"ass_tibia_left_max":null,"ass_tibia_left_bi":null,"ass_tibia_left_head":null,"ass_tibia_left_distal":null,"ass_tibia_right_max":null,"ass_tibia_right_bi":null,"ass_tibia_right_head":null,"ass_tibia_right_distal":null,"ass_tibia_avg_max":null,"ass_tibia_avg_bi":null,"ass_tibia_avg_head":null,"ass_tibia_avg_distal":null,"trauma_clavicle_left_cat":"NOT_PRESENT","trauma_clavicle_left_val":-1,"trauma_clavicle_right_cat":"NOT_PRESENT","trauma_clavicle_right_val":-1,"trauma_clavicle_avg_cat":"NOT_PRESENT","trauma_clavicle_avg_val":-1,"trauma_scapula_left_cat":"NOT_PRESENT","trauma_scapula_left_val":-1,"trauma_scapula_right_cat":"NOT_PRESENT","trauma_scapula_right_val":-1,"trauma_scapula_avg_cat":"NOT_PRESENT","trauma_scapula_avg_val":-1,"trauma_humerus_left_cat":"NOT_PRESENT","trauma_humerus_left_val":-1,"trauma_humerus_right_cat":"NOT_PRESENT","trauma_humerus_right_val":-1,"trauma_humerus_avg_cat":"NOT_PRESENT","trauma_humerus_avg_val":-1,"trauma_ulna_left_cat":"NOT_PRESENT","trauma_ulna_left_val":-1,"trauma_ulna_right_cat":"NOT_PRESENT","trauma_ulna_right_val":-1,"trauma_ulna_avg_cat":"NOT_PRESENT","trauma_ulna_avg_val":-1,"trauma_radius_left_cat":"NOT_PRESENT","trauma_radius_left_val":-1,"trauma_radius_right_cat":"NOT_PRESENT","trauma_radius_right_val":-1,"trauma_radius_avg_cat":"NOT_PRESENT","trauma_radius_avg_val":-1,"trauma_femur_left_cat":"NOT_PRESENT","trauma_femur_left_val":-1,"trauma_femur_right_cat":"NOT_PRESENT","trauma_femur_right_val":-1,"trauma_femur_avg_cat":"NOT_PRESENT","trauma_femur_avg_val":-1,"trauma_tibia_left_cat":"NOT_PRESENT","trauma_tibia_left_val":-1,"trauma_tibia_right_cat":"NOT_PRESENT","trauma_tibia_right_val":-1,"trauma_tibia_avg_cat":"NOT_PRESENT","trauma_tibia_avg_val":-1,"trauma_fibula_left_cat":"NOT_PRESENT","trauma_fibula_left_val":-1,"trauma_fibula_right_cat":"NOT_PRESENT","trauma_fibula_right_val":-1,"trauma_fibula_avg_cat":"NOT_PRESENT","trauma_fibula_avg_val":-1,"trauma_facial_bones_cat":"NOT_PRESENT","trauma_facial_bones_val":-1,"trauma_ribs_cat":"NOT_PRESENT","trauma_ribs_val":-1,"trauma_vertabrae_cat":"NOT_PRESENT","trauma_vertabrae_val":-1,"context_spear":true,"context_pot":false}]')


def main():
    unittest.main()


if __name__ == "__main__":
    main()
