# mesonettools.py
#
# Abhiram Pamula (apamula@okstate.edu)
# Ben Rubinstein (brubinst@hawk.iit.edu)
#
# last updated: 09/20/2022
#
# contains the MesonetTools class
import os, re
import pandas as pd
import numpy as np
import missingno as msno
from okmesopy import MesonetDownloader
from geopy import distance

class MesonetTools:
    '''
    The MesonetTools class contains methods to assist with processing time
        series generated by the MesonetDownloader class.
    '''

    def __init__(self, verbose=False):
        '''
        init method for the MesonetTools class

        arguments:
            verbose (bool): if true write detailed debugging to stdout
        '''
        self.verbose=verbose
        # these are used internally when replacing data
        self.nondatcols=['STID','DATETIME']
        self.errorcodes=[-994,-995,-996,-997,-998,-999]


    def replace_errors(self,df,code=1,column=None):
        '''
        Replace error codes in the dataset with NaN.

        Description of error codes:
            -999 - flagged bad by QA routines
            -998 - sensor not installed
            -997 - missing calibration coefficients
            -996 - station did not report
            -995 - data not reported on this time interval
            -994 - value is too wide to fit in column
        arguments:
            df (DataFrame or dict): the dataframe or dictionary of dataframes
                to be manipulated
            code (int): the specific error code to be replaced, the default 1
                replaces all error codes
            column (str): optional parameter that when specified changes only
                a single column

        returns:
            DataFrame or dict: the modified df object
        '''
        df = df.copy()
        # check if we've been given a dict or dataframe
        if self.__is_dict(df)==-1:
            if self.verbose:
                print('Warning: replace_errors() expects a DataFrame or dict'
                      ' not a {}. No actions performed.'.format(type(df)))
        # check that the error code argument is valid
        elif code != 1 and code not in self.errorcodes:
            if self.verbose:
                print('Warning: {} is not a valid error code. Nothing will'
                      ' be replaced. Use 1 or do not pass in a code argument'
                      ' to replace all error codes or enter one of the'
                      ' following: {}.'.format(code,self.errorcodes))
                print('help(MesonetTools.replace_errors) will give a'
                      ' description of the error codes.')
        # if df is a dictionary, recursively call this function for each of its keys
        elif self.__is_dict(df)==1:
            for key in df:
                df[key] = self.replace_errors(df[key],code,column)
        # if df is a dataframe
        elif self.__is_dict(df)==0:
            if code==1:
                # replace all error codes with NaN
                for i in self.errorcodes:
                    if column is None:
                        # replace for all columns
                        df = df.replace(str(i),np.nan)
                        df = df.replace(i,np.nan)
                    else:
                        # check if the column exists
                        if column in df.columns:
                            # replace for a single column
                            df[column] = df[column].replace(str(i),np.nan)
                            df[column] = df[column].replace(i,np.nan)
                        elif self.verbose:
                            print('Warning: there is no column named {}'
                                  ' in the dataframe. No actions will be'
                                  ' taken.'.format(column))
            # check if code is a valid error code
            else:
                if column is None:
                    # replace for all columns
                    df = df.replace(code,np.nan)
                    df = df.replace(code,np.nan)
                else:
                    # check if the column exists
                    if column in df.columns:
                        # replace for a single column
                        df[column] = df[column].replace(str(code),np.nan)
                        df[column] = df[column].replace(code,np.nan)
                    elif self.verbose:
                        print('Warning: there is no column named {}'
                              ' in the dataframe. No actions will be'
                              ' taken.'.format(column))
        return df


    def interpolate_missing(self,df,codes=[],column=None):
        '''
        Fills missing data with simple linear interpolation between known
            values
        
        This method will automatically ignore the following columns:
            'STID', 'DATETIME'

        arguments:
            df (DataFrame or dict): the dataframe or dictionary of dataframes
                to be manipulated
            codes (list): optional parameter that when specified interpolates
                only for the specified codes
            column (str): optional parameter that when specified changes only
                a single column

        returns:
            DataFrame or dict: the modified df object
        '''
        # check if we've been given a dict or dataframe
        if self.__is_dict(df)==-1:
            if self.verbose:
                print('Warning: interpolate_missing() expects a DataFrame or'
                      ' dict not a {}. No actions performed.'.format(type(df)))
        # check that at least one error code in the list is valid
        elif codes and all(i not in self.errorcodes for i in codes):
            if self.verbose:
                print('Warning: No valid error codes were entered: {}. No'
                      ' changes will be made. Use an empty list or do not pass'
                      ' in the codes argument to replace all error codes or'
                      ' enter at least one of the following valid error codes:'
                      ' {}'.format(codes,self.errorcodes))
                print('help(MesonetTools.replace_errors) will give a'
                      ' description of the error codes.')
        # if df is a dictionary, recursively call this function for each of its keys
        elif self.__is_dict(df)==1:
            for key in df:
                df[key] = self.interpolate_missing(df[key],codes,column)
        else:
            backup = pd.DataFrame()
            if codes:
                # store a backup so we can recovery error codes that shouldn't
                #   be replaced
                backup = df
            # replace all error codes
            df = self.replace_errors(df,column=column)
            if column is None:
                for ncolumn in df.columns:
                    if ncolumn not in self.nondatcols:
                        df[ncolumn] = df[ncolumn].interpolate()
            else:
                df[column] = df[column].interpolate()
            # if there were specific error codes provided, recover the others
            if codes:
                df = self.__copy_errors(df,backup,codes,column)
        return df


    def fill_neighbor_data(self,df,downloader,codes=[],column=None):
        '''
        Fills missing data with the value from the geographically closest
            station that has the missing observation
        
        This method will automatically ignore -995 error codes and the
            following columns: 'STID', 'DATETIME'

        arguments:
            df (DataFrame or dict): the dataframe or dictionary of dataframes
                to be manipulated
            downloader (MesonetDownloader): a MesonetDownloader object is
                required to calculate distances and download new data
            codes (list): optional parameter that when specified interpolates
                only for the specified codes
            column (str): optional parameter that when specified changes only
                a single column

        returns:
            DataFrame or dict: the modified df object
        '''
        # make sure that downloader is a MesonetDownloader object
        if not isinstance(downloader, MesonetDownloader):
            if self.verbose:
                print('Warning: downloader must be an okmesopy.MesonetDownloader'
                      ' object not a {}. No changes will be made.'.format(type(downloader)))
        # check if we've been given a dict or dataframe
        elif self.__is_dict(df)==-1:
            if self.verbose:
                print('Warning: fill_neighbor_data() expects a DataFrame or'
                      ' dict not a {}. No actions performed.'.format(type(df)))
        # check that at least one error code in the list is valid
        elif codes and all(i not in self.errorcodes for i in codes):
            if self.verbose:
                print('Warning: No valid error codes were entered: {}. No'
                      ' changes will be made. Use an empty list or do not pass'
                      ' in the codes argument to replace all error codes or'
                      ' enter at least one of the following valid error codes:'
                      ' {}'.format(codes,self.errorcodes))
                print('help(MesonetTools.replace_errors) will give a'
                      ' description of the error codes.')
        # if df is a dictionary, recursively call this function for each of its keys
        elif self.__is_dict(df)==1:
            for key in df:
                df[key] = self.fill_neighbor_data(df[key],downloader,codes,column)
        else:
            stid = df.loc[0,'STID']
            df = df.set_index(['DATETIME'])
            if not codes:
                codes = self.errorcodes.copy()
            # skip -995, no stations will have data on the not sampled intervals
            if -995 in codes: codes.remove(-995)
            for code in codes:
                df = self.replace_errors(df,code,column)
            # create a list of stations sorted by distance
            target_coord = downloader.get_station_coord(stid)
            coord_tuple = list(downloader.metadata.loc[:,['nlat','elon']].itertuples(index = False, name = None))
            for i in coord_tuple:
                if i == target_coord:
                    coord_tuple.remove(i)
            req_loc = downloader.metadata['stid'].loc[downloader.metadata['stid']!=stid]
            station_list=[]
            for i,j in zip(req_loc,coord_tuple):
                station_list.append([i,distance.distance(target_coord,j).miles])
            station_list = sorted(station_list, key = lambda x:(x[1], x[0]))
            stids = [i[0] for i in station_list]
            for station in stids:
                # break when all data has been filled
                if df.isnull().sum().sum()==0:
                    break
                df = self.__download_neighbor(df,downloader,station)
            df.reset_index(inplace=True)
        return df


    def summarize_missing(self,df,graph=False):
        '''
        Summarizes error codes in the provided dataframe

        arguments:
            df (DataFrame): the dataframe with missing data to be summarized
            graph (bool): an optional arugument that if true will generate
                visualization of the missing data using the missingno library
        '''
        # Generate a new dataframe that will hold the error info
        cols = df.columns.to_list()
        for col in self.nondatcols: cols.remove(col)
        errors = pd.DataFrame()
        # loop through each column
        for col in cols:
            # count the error codes
            counts = df.groupby(col).size()
            for error in self.errorcodes:
                # populate the error dataframe
                try:
                    count = counts[error]
                    errors.loc[col,error]=int(count)
                except:
                    # a key value error here means there were none of this
                    # error code in this column
                    errors.loc[col,error]=int(0)
            # add a total value for each column
            errors.loc[col,'TOTAL']=int(errors.loc[col,:].sum())
        # add a total for the error codes as well
        for col in errors.columns:
            errors.loc['TOTAL',col]=int(errors.loc[:,col].sum())
        # total values
        total = errors.sum().sum()
        total_corrected = total - errors.loc[:,-995].sum()
        # print a summary of the missing data
        print('Missing data summary for {} station:'.format(df.loc[0,'STID']))
        print('-----------------------')
        print('To see a description of each error code run help(MesonetTools.replace_errors)')
        print('Note: the -995 error code is used when data is not collected'
              ' on an interval. This code is generally normal and expected.')
        print('There are {} total missing data points and {} missing data'
              ' data points excluding -995 codes'.format(total,total_corrected))
        print('The following chart displays the number of each kind of error'
              ' code found in each column of the DataFrame.\n')
        print(errors)
        # graphically display errors if the chart arguement is true
        if graph:
            # prepare a dataframe for the missingno library, we need to
            # actually replace all the error codes with NaN so missingno can
            # recognize them
            rep_df = self.replace_errors(df)
            # also remove non data columns and set the index to DATETIME
            for col in self.nondatcols:
                if not col=='DATETIME': rep_df.drop(col,axis=1,inplace=True)
            msno.matrix(rep_df.set_index('DATETIME'),freq='5M')


    def save_timeseries(self,df,column,step=5):
        '''
        Saves the a data for a single variable for a single station as a
            PyHSPF readable timeseries. If a set of stations is provided,
            the arithmetic mean is used instead

        arguments:
            df (DataFrame or dict): the DataFrame to create a series from; if
                a dictionary is passed in, the arithmetic mean of all contained
                DataFrames is used
            column (str): the variable to create a time series for
            step (int): the time interval to use in minutes; must be at least
                5 and divisible by 5; the closest multiple of 5 will be used
                instead if not

        returns:
            tuple (datetime,int,series): the timeseries object in the form
                (start time,step size,data)
        '''
        column = column.upper()
        # validate the step size
        if step < 5:
            if self.verbose:
                print('{} was given as step size but it must be at least 5'
                      ' minutes. A step size of 5 minutes will be used'
                      ' instead'.format(step))
            step = 5
        elif step%5 != 0:
            if self.verbose:
                print('Step size must be a multiple of 5. The given step size'
                      ', {}, will be rounded up to {}.'.format(step,step + 5 - step%5))
            step = step + 5 - step%5
        # if a dictionary has been given, generate the average
        tempdf = pd.DataFrame()
        if self.__is_dict(df):
            # concatenate all the DataFrames together, using only the relevant
            #   column to save memory
            for key in df:
                start = df[key].loc[0,'DATETIME']
                # make sure the column exists
                if not column in df[key].columns:
                    print('Error: the column {} was not found in the dataframe'
                          ' for the {} station.'.format(column,key))
                    return None
                tempdf = pd.concat((tempdf,df[key].copy(column)))
            # remove all error codes so they don't throw off the averages
            tempdf = self.replace_errors(tempdf)
            # make df the mean
            df = tempdf.groupby(tempdf.index).mean()
        else:
            # get the start date and time
            start = df.loc[0,'DATETIME']
        data = []
        # easiest case we just use every data point
        if step == 5:
            for dat in df[column]:
                # if valid data append it, else append None
                if not np.isnan(dat) and dat not in self.errorcodes:
                    data.append(dat)
                else:
                    data.append(None)
        else:
            multiple = int(step/5)
            count = 0
            index = 1
            temp_val = 0
            for dat in df[column]:
                if index < multiple:
                    index += 1
                    # keep a running store of the sum of values
                    if not np.isnan(dat) and dat not in self.errorcodes:
                        temp_val += dat
                        # count is tracked separately from index so that error
                        #   codes don't throw off the averages
                        count += 1
                else:
                    # we've hit the interval we are looking for and now need to
                    #   add our data to the time series
                    if column == 'RAIN':
                        # rainfall is a special case because it needs to be
                        #   cumulative not averaged
                        data.append(temp_val)
                    elif count == 0:
                        data.append(None)
                    else:
                        # take the average and append it
                        data.append(temp_val/count)
                    # reset all our temporary values
                    count=0
                    index=1
                    temp_val=0
        ts = start, step, data
        return ts


    def save_csv(self,df,path,force=False,filename=''):
        '''
        Saves a downloaded dataset as a CSV file.

        arguments:
            df (DataFrame or dict): the DataFrame or dictionary of DataFrames
                to be saved
            path (str): the directory to save the file to
            force (bool): if set to true will overwrite existing files
            filename (str): optional name of the csv file; filenames must be
                alphanumeric characters, hyphens, underscores, and periods only
        '''
        path = os.path.normpath(path)
        try:
            if not os.path.isdir(path): os.mkdir(path)
        except PermissionError:
            errstr = ('The specified directory {} does not exist and you do not'
                      ' have permission to create it. Please try a different'
                      ' directory.')
            raise PermissionError(errstr.format(path))
        # check if the filename is valid
        if re.search(r'[^A-Za-z0-9\._\-\\]',filename):
            print('Warning: Filenames can only contain alphanumeric characters'
                  ', hyphens, underscores, and periods. Enter a new filename'
                  ' or leave the argument blank to use an autogenerated name.')
            return
        # if we have a dictionary, concatenate all the DataFrames together
        if self.__is_dict(df):
            concat_df = pd.DataFrame()
            for key in df:
                concat_df = pd.concat((concat_df,df[key]))
                # generate the file name
                if not filename:
                    start = df[key].loc[0,'DATETIME'].strftime('%m%d%y')
                    end = df[key].iloc[-1]['DATETIME'].strftime('%m%d%y')
                    filename = '{}_and-{}-more_{}-{}.csv'.format(key,len(df)-1,start,end)
            # if file exists check if the force argument is true
            if os.path.exists('{}/{}'.format(path,filename)) and not force:
                print('Warning: The file {} already exists in the directory {}.'
                      ' Please choose a new filename or directory or set the force'
                      ' argument to true to overwrite the existing file.'.format(filename,path))
                return
            concat_df.to_csv('{}/{}'.format(path,filename),index=False)
        else:
            if not filename:
                stid = df.loc[0,'STID']
                start = df.loc[0,'DATETIME'].strftime('%m%d%y')
                end = df.iloc[-1]['DATETIME'].strftime('%m%d%y')
                filename = '{}_{}-{}.csv'.format(stid,start,end)
            # if file exists check if the force argument is true
            if os.path.exists('{}/{}'.format(path,filename)) and not force:
                print('Warning: The file {} already exists in the directory {}.'
                      ' Please choose a new filename or directory or set the force'
                      ' argument to true to overwrite the existing file.'.format(filename,path))
                return
            df.to_csv('{}/{}'.format(path,filename),index=False)


    def __download_neighbor(self,df,downloader,station_id):
        '''
        Helper function that downloads and fills data from a neighboring station

        arguments:
            df (DataFrame): the dataframe with missing data to be filled 
            downloader (MesonetDownloader): a MesonetDownloader object is
                required to download new data
            station_id (str): the station ID for the neighboring station

        returns:
            DataFrame: the modified df object
        '''
        # get a list of dates with missing data
        missing_dates = []
        for dt in df[df.isna().any(axis=1)].index.unique():
            if dt.date() not in missing_dates: missing_dates.append(dt.date())
        # download data for each of the missing dates
        for miss_date in missing_dates:
            date = pd.to_datetime(miss_date).date()
            neighbor_df = downloader.download_station_data(station_id,date,date)
            if neighbor_df is not None:
                # we don't want to copy over any error codes so replace all of them
                neighbor_df = self.replace_errors(neighbor_df)
                # fill in data
                neighbor_df.set_index(['DATETIME'],inplace=True)
                df = df.fillna(neighbor_df)
        return df


    def __copy_errors(self,df,backup,codes,column=None):
        '''
        Helper function that copies error codes back into a dataframe

        arguments:
            df (DataFrame): the dataframe to be manipulated
            backup (DataFrame): a copy of df with error codes still in place 
            codes (list): a list of codes to copy back into df
            column (str): optional parameter that when specified changes only
                a single column

        returns:
            DataFrame: the modified df object
        '''
        # TODO: fix the SettingWithCopyError?
        pd.options.mode.chained_assignment = None
        for code in self.errorcodes:
            if code not in codes:
                if column is not None:
                    # these commands cause a SettingWithCopyError. I think it
                    #   is a false positive but I'm not fully sure I
                    #   understand the error properly
                    df[column].loc[backup[column]==code] = code
                else:
                    for ncolumn in df.columns:
                        df[ncolumn].loc[backup[ncolumn]==code] = code
        pd.options.mode.chained_assignment = 'warn'
        return df


    def __is_dict(self,df):
        '''
        MesonetDownloader creates single dataframes and dictionaries of
            dataframes. Returns 1 for dict, 0 for dataframe, and -1 as an
            error code for anything else

        arguments:
            df (DataFrame or dict): the object to type check

        returns:
            int: 1 for a dictionary, 0 for a DataFrame, -1 otherwise
        '''
        if isinstance(df,dict):
            return 1
        elif isinstance(df,pd.DataFrame):
            return 0
        else:
            return -1
