# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pycmtensor']

package_data = \
{'': ['*']}

install_requires = \
['Cython>=0.29.28,<0.30.0',
 'aesara>=2.4.0,<3.0.0',
 'dill>=0.3.4,<0.4.0',
 'ipykernel>=6.9.1,<7.0.0',
 'ipywidgets>=7.6.5,<8.0.0',
 'numpy>=1.19.0,<1.22.0',
 'pandas>=1.3.5,<2.0.0',
 'pydot>=1.4.2,<2.0.0',
 'scipy>=1.7.1,<1.8.0',
 'tqdm>=4.62.3,<5.0.0']

setup_kwargs = {
    'name': 'pycmtensor',
    'version': '1.0.2',
    'description': 'Python Tensor based package for Deep neural net assisted Discrete Choice Modelling.',
    'long_description': '# PyCMTensor\n\n![](https://img.shields.io/pypi/pyversions/pycmtensor)\n[![PyPI version](https://badge.fury.io/py/pycmtensor.svg)](https://badge.fury.io/py/pycmtensor)\n[![Tests](https://github.com/mwong009/pycmtensor/actions/workflows/tests.yml/badge.svg)](https://github.com/mwong009/pycmtensor/actions/workflows/tests.yml)\n[![Documentation Status](https://readthedocs.org/projects/pycmtensor/badge/?version=develop)](https://pycmtensor.readthedocs.io/en/develop/?badge=develop)\n[![codecov](https://codecov.io/gh/mwong009/pycmtensor/branch/master/graph/badge.svg?token=LFwgggDyjS)](https://codecov.io/gh/mwong009/pycmtensor)\n![Licence](https://img.shields.io/badge/Licence-MIT-blue)\n\nA tensor-based choice modelling Python package with deep learning capabilities. \nRead the documentation at <https://pycmtensor.readthedocs.io>\n\n## About PyCMTensor\n\nPyCMTensor is a discrete choice model development platform which is designed with the use of deep learning in mind, enabling users to write more complex models using neural networks.\nPyCMTensor is build on [Aesara library](https://github.com/aesara-devs/aesara), and uses many features commonly found in deep learning packages such as Tensorflow and Keras.\n`Aesara` was chosen as the back end mathematical library because of its hackable, open-source nature.\nAs users of [Biogeme](https://biogeme.epfl.ch), you will be familiar with the syntax of PyCMTensor.\n\nThis package allows one to incorporate neural networks into discrete choice models that boosts accuracy of model estimates which still being able to produce all the same statistical analysis found in traditional choice modelling software.\n\nPyCMTensor aims to provide developers and researchers with deep learning tools for econometric and travel behaviour modelling with **reproducible** and **interpretable** results.\n\n### PyCMTensor and Biogeme\n\nPyCMTensor improves on Biogeme in situations where much more complex models are necessary, for example, integrating neural networks into discrete choice models.\nPyCMTensor also include the ability to estimate models using 1st order stochastic gradient descent methods by default, such as Nesterov Accelerated Gradient, Adam, or RMSProp.\n## Features\n\n* Estimate complex choice models with neural networks using deep learning algorithms\n* Combines traditional econometric models (e.g. Multinomial Logit) with deep learning models (ResNets)\n* Shares similar programming syntax with `Biogeme`, allowing easy transition between `Biogeme` and PyCMTensor methods\n* Uses tensor based mathematical operations from the advanced features found in the `Aesara` library\n\n---\n\n## Pre-install\n\nTo install PyCMTensor, you need [Conda](https://docs.conda.io/en/latest/miniconda.html) (Full Anaconda works fine, but **miniconda** is recommmended for a minimal installation). \nEnsure that Conda is using at least `Python 3.9`.\n\nOnce Conda is installed, install the required dependencies from conda by running the following \ncommand in your terminal:\n\n**Windows**\n\n```console\nconda install pip git conda-forge::cxx-compiler conda-forge::libpython blas mkl-service numpy\n```\n**Linux/MacOS**\n\n```console\nconda install blas mkl-service conda-forge::cxx-compiler\n```\n\nNote: MacOS user should also [install](https://www.ics.uci.edu/~pattis/common/handouts/macclion/clang.html) `Clang` for a fast compiled code.\n\n### Install PyCMTensor\n\nThen, run this command in your terminal to download and install the latest branch of PyCMTensor from [PyPI](https://pypi.org/project/pycmtensor/):\n```console\npip install pycmtensor -U\n```\n\n*Optional*: If you want the development version from the Github repository:\n```console\npip install git+https://github.com/mwong009/pycmtensor.git@develop -U\n```\n\nThe development branch is the most recent update of PyCMTensor. If you want a stable branch (master), remove ``@develop`` at the end of the ``.git`` url.\n\n---\n\n## How to use\n\nPyCMTensor uses syntax very similar to `Biogeme`. Users of `Biogeme` should be familiar \nwith the syntax.\n\nStart an interactive session (`IPython` or Jupyter Notebook) and import:\n```Python\nimport pycmtensor as cmt\n```\n\nSeveral submodules are also important to include:\n```Python\nfrom pycmtensor.expressions import Beta # Beta class for model parameters\nfrom pycmtensor.models import MNLogit   # model library\nfrom pycmtensor.optimizers import Adam  # Optimizers\nfrom pycmtensor.results import Results  # for generating results\n```\n\nFor a full list of submodules and description, refer to [API Reference](/autoapi/index)\n\n### Simple example: Swissmetro dataset\n\nUsing the [swissmetro dataset](https://biogeme.epfl.ch/data.html) from Biogeme, we define a simple MNL model. \n\nNote:The following is a replication of the results from Biogeme using the `Adam` optimization algorithm. For further examples including the ResLogit model, refer **here**.\n\n\n1. Import the dataset and perform some data santiation\n\t```Python\n\tswissmetro = pd.read_csv("swissmetro.dat", sep="\\t")\n\tdb = cmt.Database(name="swissmetro", pandasDatabase=swissmetro, choiceVar="CHOICE")\n\tglobals().update(db.variables)\n\t# additional steps to format database\n\tdb.data["CHOICE"] -= 1 # set the first choice to 0\n\tdb.choices = sorted(db.data["CHOICE"].unique()) # save original choices\n\tdb.autoscale(\n\t\tvariables=[\'TRAIN_CO\', \'TRAIN_TT\', \'CAR_CO\', \'CAR_TT\', \'SM_CO\', \'SM_TT\'], \n\t\tdefault=100., \n\t) # automatically scales features by 1/100.\n\t```\n\n\t``cmt.Database()`` loads the dataset and defines Tensor Variables automatically.\n\n2. Initialize the model parameters\n\t```Python\n\tb_cost = Beta("b_cost", 0.0, None, None, 0)\n\tb_time = Beta("b_time", 0.0, None, None, 0)\n\tasc_train = Beta("asc_train", 0.0, None, None, 0)\n\tasc_car = Beta("asc_car", 0.0, None, None, 0)\n\tasc_sm = Beta("asc_sm", 0.0, None, None, 1)  # set to 1 to keep it fixed\n\t``` \n\n3. Specify the utility functions and availability conditions\n\t```Python\n\tU_1 = b_cost * db["TRAIN_CO"] + b_time * db["TRAIN_TT"] + asc_train\n\tU_2 = b_cost * db["SM_CO"] + b_time * db["SM_TT"] + asc_sm\n\tU_3 = b_cost * db["CAR_CO"] + b_time * db["CAR_TT"] + asc_car\n\tU = [U_1, U_2, U_3]\n\tAV = [db["TRAIN_AV"], db["SM_AV"], db["CAR_AV"]]\n\t```\n\n4. Specify the model ``MNLogit``\n\t```Python\n\tmymodel = MNLogit(u=U, av=AV, database=db, name="mymodel")\n\tmymodel.add_params(locals())  # load Betas into the model\n\t```\n\n5. Define the training hyperparameters\n\t```Python\n\tmymodel.config["patience"] = 20000\n\tmymodel.config["max_epoch"] = 499\n\tmymodel.config["base_lr"] = 0.0012\n\tmymodel.config["max_lr"] = 0.002\n\tmymodel.config["learning_scheduler"] = "CyclicLR"\n\tmymodel.config["cyclic_lr_step_size"] = 8\n\tmymodel.config["cyclic_lr_mode"] = "triangular2"\n\t```\n\n6. Call the training function and save the trained model\n\t```Python\n\tmodel = cmt.train(mymodel, database=db, optimizer=Adam)  # we use the Adam Optimizer\n\t```\n\n7. Generate the statistics and correlation matrices\n\t```Python\n\tresult = Results(model, db, show_weights=True)\n\tresult.print_beta_statistics()\n\tresult.print_correlation_matrix()\n\t```\n\n\tSample output: \n\n\t\tOptimization complete with accuracy of 62.618%. Max loglikelihood reached @ epoch 195.\t\n\n\t\tResults for model: mymodel\n\t\tBuild time: 00:00:13\n\t\tEstimation time: 00:00:28\n\t\tEstimation rate: 13.781 epochs/s\n\t\tSeed value: 999\n\t\tNumber of Beta parameters: 4\n\t\tSample size: 6768\n\t\tExcluded data: 3960\n\t\tInit loglikelihood: -6964.663\n\t\tFinal loglikelihood: -5590.672\n\t\tFinal loglikelihood reached at: epoch 195\n\t\tLikelihood ratio test: 2747.982\n\t\tAccuracy: 62.618%\n\t\tRho square: 0.197\n\t\tRho bar square: 0.197\n\t\tAkaike Information Criterion: 11189.34\n\t\tBayesian Information Criterion: 11216.62\n\t\tFinal gradient norm: 0.111\n\n\t\tStatistical Analysis:\n\t\t              Value   Std err     t-test   p-value Rob. Std err Rob. t-test Rob. p-value\n\t\tasc_car    0.111877  0.042071   2.659267  0.007831     0.038512    2.905005     0.003672\n\t\tasc_train -0.624174   0.05471 -11.408845       0.0     0.014402   -43.33787          0.0\n\t\tb_cost     0.002601  0.002547    1.02136  0.307084     0.003616    0.719242     0.471992\n\t\tb_time     -1.16109  0.054086 -21.467576       0.0     0.005372 -216.155293          0.0\n\n\t\tCorrelation matrix:\n\t\t             b_cost    b_time  asc_train   asc_car\n\t\tb_cost     1.000000 -0.105761   0.154368  0.283711\n\t\tb_time    -0.105761  1.000000  -0.724388 -0.659056\n\t\tasc_train  0.154368 -0.724388   1.000000  0.606882\n\t\tasc_car    0.283711 -0.659056   0.606882  1.000000\n\n8. Plot the training performance and accuracy\n\n\t![](https://github.com/mwong009/pycmtensor/blob/master/docs/_static/viz/fig.png)\n\n9. Compute the elasticities\n\n\t![](https://github.com/mwong009/pycmtensor/blob/master/docs/_static/viz/els.png)\n\n10. Visualize the computation graph\n\t```Python\n\timport aesara.d3viz as d3v\n\tfrom aesara import printing\n\tprinting.pydotprint(mymodel.cost, "graph.png")\n\t```\n\n\t![](https://github.com/mwong009/pycmtensor/blob/master/docs/_static/viz/print.png)\n\n---\n\n## Credits\n\nPyCMTensor was inspired by [Biogeme](https://biogeme.epfl.ch) and aims to provide deep learning modelling tools for transport modellers and researchers.\n\nThis package was generated with [Cookiecutter](https://github.com/audreyr/cookiecutter) and the [audreyr/cookiecutter-pypackage](https://github.com/audreyr/cookiecutter-pypackage) project template.\n',
    'author': 'Melvin Wong',
    'author_email': 'm.j.w.wong@tue.nl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/mwong009/pycmtensor',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<=3.10',
}


setup(**setup_kwargs)
