# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['colorcontroller']

package_data = \
{'': ['*'], 'colorcontroller': ['readmepics/*']}

install_requires = \
['Pillow>=8.4.0,<9.0.0',
 'cycler>=0.11.0,<0.12.0',
 'fonttools>=4.28.3,<5.0.0',
 'kiwisolver>=1.3.2,<2.0.0',
 'matplotlib>=3.5.0,<4.0.0',
 'numpy>=1.21.4,<2.0.0',
 'packaging>=21.3,<22.0',
 'pandas>=1.3.4,<2.0.0',
 'pyparsing>=3.0.6,<4.0.0',
 'python-dateutil>=2.8.2,<3.0.0',
 'pytz>=2021.3,<2022.0',
 'setuptools-scm>=6.3.2,<7.0.0',
 'six>=1.16.0,<2.0.0',
 'tomli>=1.2.2,<2.0.0']

setup_kwargs = {
    'name': 'colorcontroller',
    'version': '0.1.2',
    'description': 'ColorController is a Pythonic interface for managing colors using either english-language names or one of several standard color values.',
    'long_description': '# ColorController.py\n\n# Table of Contents\n1. [Encode color data in various formats.](#1-encode-color-data-in-various-formats)\n   - [1.1: Create a ColorController object using a familiar, english-language color name, and print out its properties.](#example-11-create-a-colorcontroller-object-using-a-familiar-english-language-color-name-and-print-out-its-properties)\n   - [1.2: Show a color](#example-12-show-a-color)\n   - [1.3: Create a ColorController object using a hex code.](#example-13-create-a-colorcontroller-object-using-a-hex-code)\n   - [1.4: Create a ColorController object using an RGB triplet](#example-14-create-a-colorcontroller-object-using-an-rgb-triplet)\n   - [1.5: Create a ColorController object using an HSV triplet, and print out its properties.](#example-15-create-a-colorcontroller-object-using-an-hsv-triplet-and-print-out-its-properties)\n2. [Modify a color using simple, convenient methods.](#2-modify-a-color-using-simple-convenient-methods)\n3. [Invert a color](#3-invert-a-color)\n4. [Access a rich set of color values and color names](#4-access-a-rich-set-of-color-values-and-color-names-prepared-by-martin-krzywinski-conveniently-stored-in-a-pandas-dataframe)\n\n\n\nWelcome to the ColorController Python library! \n\nMy name is Tal Zaken, and I wrote this library for use in a natural language processing project \nthat aims to take in free-form text, and spit out color data which somehow relates to the text\'s \ncontent. \n\nEnough about that. Here are some things that you can do with ColorController:\n\n## 1. Encode color data in various formats.\n#### Example 1.1: Create a ColorController object using a familiar, english-language color name, and print out its properties.\n\nYou can set a color using a very large library of color names. \nSee the colornames.txt document contained herein, with enormous thanks to \n[Martin Krzywinski](http://mkweb.bcgsc.ca/colornames). \n\nThe following code:\n\n```python\nfrom ColorController import ColorController\n\ncolor = ColorController(name=\'hazel\')\n\nprint(f"Name: {color.name}",\n      f"Hex Code: {color.hex_code}",\n      f"RGB: {color.rgb}",\n      f"HSV: {color.hsv}",\n      sep=\'\\n\')\n```\noutputs:\n```\nName: hazel\nHex Code: [\'#8E7618\']\nRGB: [(142, 118, 24)]\nHSV: [(0.133, 0.831, 142)]\n```\nFurther, you can change all the ColorController\'s properties by changing any one of them. \n\nBy example:\n```python\ncolor.name = \'blue\'\n\nprint(f"Name: {color.name}", \n      f"Hex Code: {color.hex_code}", \n      f"RGB: {color.rgb}", \n      f"HSV: {color.hsv}", \n      sep=\'\\n\')\n```\nYou will see that all properties have updated:\n```\nName: blue\nHex Code: [\'#00008B\', \'#0000CD\', \'#0000EE\', \'#0000FF\', \'#0018A8\', \'#0087BD\', \'#0093AF\', \'#0247FE\', \'#0343DF\', \'#1F75FE\', \'#2242C7\', \'#333399\']\nRGB: [(0, 0, 139), (0, 0, 205), (0, 0, 238), (0, 0, 255), (0, 24, 168), (0, 135, 189), (0, 147, 175), (2, 71, 254), (3, 67, 223), (31, 117, 254), (34, 66, 199), (51, 51, 153)]\nHSV: [(0.667, 1.0, 139), (0.667, 1.0, 205), (0.667, 1.0, 238), (0.667, 1.0, 255), (0.643, 1.0, 168), (0.548, 1.0, 189), (0.527, 1.0, 175), (0.621, 0.992, 254), (0.618, 0.987, 223), (0.602, 0.878, 254), (0.634, 0.829, 199), (0.667, 0.667, 153)]\n```\nNotably, the colornames.txt file has numerous entries that all share the name "blue." This is true of many colors.\nBecause color is thought to be a culturally relative phenomenon, I have chosen to return all hex codes that match a given name. \nYou will notice a similar phenomenon occurs for color names when you set a color using hex code, RGB, or HSV. \nThis is because there are sometimes many names that all describe the same color. \n\n#### Example 1.2: Show a color.\nWe\'ve had a lot of talk about colors so far, but we haven\'t even seen any colors yet! \nLet\'s solve that now, and do away with these lengthy print statements:\n```python\ncolor.hex_code =\'#ffbff9\'\n\ncolor.show_color()\n```\nShows:\n\n![pale_orchid](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/pale_orchid.PNG?raw=true "pale_orchid")\n\nThat said, the ColorController object is biased toward whatever you, the user, set it to be. \nIf you explicitly set a name, then that will be the singular name of your object.\nSimilarly, if you explicitly set a hex code, then that will be the value of your hex code. \nIf you leave  a leading # off of your hex code, \nthen everything will still work, but that will be the hex code value. For example...\n\n#### Example 1.3: Create a ColorController object using a hex code.\n```python\ncolor = ColorController(hex_code=\'#990000\')\n\ncolor.show_color()\n```\nShows:\n\n\n![[\'crimson_red\', \'stizza\', \'ou_crimson_red\', \'usc_cardinal\']](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/crimson_red.PNG?raw=true "[\'crimson_red\', \'stizza\', \'ou_crimson_red\', \'usc_cardinal\']")\n\n\nWhile:\n```python\ncolor.hex_code = \'990000\'\n\ncolor.show_color()\n```\nShows very similar results:\n\n![[\'crimson_red\', \'stizza\', \'ou_crimson_red\', \'usc_cardinal\']](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/crimson_red2.PNG?raw=true "[\'crimson_red\', \'stizza\', \'ou_crimson_red\', \'usc_cardinal\']")\n\n\n#### Example 1.4: Create a ColorController object using an RGB triplet.\nYou can also pass a 3-tuple whose values are each contained in range(0,256) to the rgb property. For example:\n```python\ncolor = ColorController(rgb=(10, 255, 230))\n\ncolor.show_color()\n```\nShows:\n\n![[bright_aqua]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/bright_aqua.PNG?raw=true "bright_aqua")\n\n#### Example 1.5: Create a ColorController object using an HSV triplet, and print out its properties.\nLastly, you can also pass a 3-tuple whose first two values are a floating point number between 0 and 1 inclusive, and whose third value falls in range(0, 256):\n```python\ncolor = ColorController(hsv=(0.25, 1, 255))\n\ncolor.show_color()\n```\nShows:\n\n![[chartreuse]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/chartreuse.PNG?raw=true "chartreuse")\n\n\nNOTE: While this is the HSV value format that comes included with the colorsys python standard library, it doesn\'t seem to be a very common format elsewhere. \nTo match formats used in other locations, see the following functions:\n```python\ndef colorsys_hsv_to_hsv360(colorsys_hsv=tuple):\n    """Takes an HSV triplet as provided by colorsys, and converts it to match the\n    notation used in colornames.txt"""\n\ndef hsv360_to_hsvdistance(hsv360=tuple):\n    """Takes an HSV triplet as provided by colorsys_hsv_to_hsv360(), and converts it to match the\n    notation used in the function for calculating distance between colors."""  \n```\n\n\n## 2. Modify a color using simple, convenient methods.\n#### Example 2.1: Darken a color.\nYou can darken a color using the darken_color() method. For example:\n```python\nfrom ColorController import ColorController\n\ncolor = ColorController(name=\'forest\')\n\ncolor.show_color()\n\ncolor.darken_color()\n\ncolor.show_color()\n```\nWill show the following in series:\n\n![[forest]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/forest.PNG?raw=true "forest")\n![[hunter_green]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/hunter_green.PNG?raw=true "hunter_green")\n\nYou can also pass in a darkening_value between zero and one, to set the percent darker you\'d like to go. For instance:\n```python\ncolor = ColorController(name=\'cocoa\')\n\ncolor.show_color()\n\ncolor.darken_color(.3)\n\ncolor.show_color()\n```\nShows the following in series:\n\n![[cocoa]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/cocoa.PNG?raw=true "cocoa")\n![[nutmeg]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/nutmeg.PNG?raw=true "nutmeg")\n\nIn the above example, note that our color object was first initiated by the name property, \nmeaning that there are lists of associated hex, rgb, and hsv values stored in their respective properties.\nWhen we lighten the color, we have to select one index from these lists to operate on. \nThe default behavior is to operate on the first index. See the `unlist()` function in helpers.py.\n\n\n#### Example 2.2: Lighten a color.\n\nYou can also pass in a lightening_value between zero and one, to set the percent lighter you\'d like to go. For instance:\n```python\ncolor = ColorController(hex_code=\'#6c3461\')\n\ncolor.show_color()\n\ncolor.lighten_color(.5)\n\ncolor.show_color()\n```\nShows the following in series:\n\n![[grape]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/grape.PNG?raw=true "grape")\n![[grayish_fuchsia]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/grayish_fuchsia.PNG?raw=true "grayish_fuchsia")\n\n\n#### Example 2.3: Brighten a color.\n\n### 3. Invert a color.\nExample:\n```python\nfrom ColorController.conversions import invert_rgb\nfrom ColorController import ColorController\n\n\ncolor = ColorController(hex_code=\'#9ffeb0\')\ncolor.show_color()\n\nr, g, b = color.rgb\ncolor.rgb = invert_rgb(r, g, b)\ncolor.show_color()\n```\nShows the following in series:\n\n![[mint]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/mint.PNG?raw=true "mint")\n![[deep_orchid]](https://github.com/tal-z/TextToColor/blob/main/ColorController/readmepics/deep_orchid.PNG?raw=true "deep_orchid")\n\n\n\n\n### 4. Access a rich set of color values and color names (prepared by Martin Krzywinski), conveniently stored in a Pandas DataFrame.\nExample:\n\n```python\nfrom ColorController.ColorController import colors_df\n\nprint(colors_df.iloc[5000])\n```\nOutputs:\n```\nIDX                                                                  5000\nNAME                                                    light_apple_green\nrgb                                                                   rgb\nR                                                                     220\nG                                                                     231\nB                                                                     139\nhex                                                                   hex\nHEX                                                               #DCE78B\nhsv                                                                   hsv\nh                                                                      67\ns                                                                      40\nv                                                                      91\nxyz                                                                   xyz\nX                                                                    0.63\nY                                                                    0.74\nZ                                                                    0.35\nlab                                                                   lab\nL                                                                      89\nA                                                                     -17\nB                                                                      44\nlch                                                                   lch\nL                                                                      89\nC                                                                      47\nH                                                                     112\ncmyk                                                                 cmyk\nC                                                                       4\nM                                                                       0\nY                                                                      36\nK                                                                       9\nNEIGHBOUR_STR           PMS586[775][226,229,132](3.6):hypnotic[4592][2...\nNUM_NEIGHBOURS_MAXDE                                                    4\nWORD_TAGS               [light, PMS586, hypnotic, jonquil, green, lime...\nName: 5000, dtype: object\n```\n\n## Known Bugs:\n  - I don\'t know of any right now, but I\'m sure they exist!\n\n## Ideas\n  - tint and shade methods instead of or in addition to lighten/darken?\n  - more unit tests\n  - module to mix colors\n    - this will require converting to LAB or another subtractive space, so more getter/setter methods\n\n## Resources:\n  - What is color?: https://www.crayola.com/for-educators/resources-landing/articles/color-what-is-color.aspx\n  - unofficial crayola colors: https://www.w3schools.com/colors/colors_crayola.asp\n  - color names database: http://mkweb.bcgsc.ca/colornames/\n  - interactive color code tool: https://www.hexcolortool.com/#3cec71\n  - NLTK Information extraction chapter: http://www.nltk.org/book/ch07.html\n  - colorsys source code: https://github.com/python/cpython/blob/3.9/Lib/colorsys.py\n  - webcolors source code: https://github.com/ubernostrum/webcolors/blob/trunk/src/webcolors.py\n    - Note: while I\'m not using webcolors in this program, I\'m looking at their hex conversion algorithms to better understand the concept and see how it gets implemented.\n  - explanation of calculating distance in hsv space: https://stackoverflow.com/questions/35113979/calculate-distance-between-colors-in-hsv-space\n  - Making colors lighter or darker: https://www.color-hex.com/blog/making-colors-lighter-or-darker-4',
    'author': 'Tal Zaken',
    'author_email': 'talzaken@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/tal-z/ColorController',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<3.11',
}


setup(**setup_kwargs)
