"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
require("@aws-cdk/assert/jest");
const defaults = require("@aws-solutions-constructs/core");
const s3 = require("aws-cdk-lib/aws-s3");
const sns = require("aws-cdk-lib/aws-sns");
const core_1 = require("@aws-solutions-constructs/core");
test('All get methods return non-null objects', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const app = new lib_1.S3ToSns(stack, 'test-s3-sns', {});
    expect(app.snsTopic !== null);
    expect(app.s3Bucket !== null);
    expect(app.s3LoggingBucket !== null);
    expect(app.encryptionKey !== null);
    expect(app.s3BucketInterface !== null);
});
test('construct creates default event notification', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {});
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            TopicConfigurations: [
                {
                    Events: ['s3:ObjectCreated:*'],
                    TopicArn: {
                        Ref: "tests3snsSnsTopicF02F6BD0"
                    }
                }
            ]
        }
    });
});
test('construct uses existingBucketObj property', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const buildS3BucketResponse = defaults.buildS3Bucket(stack, {
        bucketProps: {
            bucketName: 'existing-bucket-name'
        }
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        existingBucketObj: buildS3BucketResponse.bucket
    });
    expect(stack).toHaveResourceLike("AWS::S3::Bucket", {
        BucketName: 'existing-bucket-name'
    });
});
test('construct uses existing topic and key', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = defaults.buildEncryptionKey(stack, {
        description: 'existing-key-description'
    });
    const buildTopicResponse = defaults.buildTopic(stack, {
        encryptionKey: cmk,
        topicProps: {
            topicName: 'existing-topic-name'
        }
    });
    const testConstruct = new lib_1.S3ToSns(stack, 'test-s3-sns', {
        existingTopicObj: buildTopicResponse.topic,
        existingTopicEncryptionKey: cmk
    });
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).toBeDefined();
    expect(stack).toHaveResourceLike("AWS::SNS::Topic", {
        TopicName: 'existing-topic-name'
    });
    expect(stack).toHaveResourceLike("AWS::KMS::Key", {
        Description: 'existing-key-description'
    });
    // Make sure the construct did not create any other topics or keys created
    expect(stack).toCountResources('AWS::KMS::Key', 1);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
});
test('construct uses specific event types and filters', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        s3EventTypes: [s3.EventType.OBJECT_REMOVED],
        s3EventFilters: [
            {
                prefix: 'the/place',
                suffix: '*.mp3'
            }
        ]
    });
    expect(stack).toHaveResource("Custom::S3BucketNotifications", {
        NotificationConfiguration: {
            TopicConfigurations: [
                {
                    Events: [
                        's3:ObjectRemoved:*'
                    ],
                    Filter: {
                        Key: {
                            FilterRules: [
                                {
                                    Name: 'suffix',
                                    Value: '*.mp3'
                                },
                                {
                                    Name: 'prefix',
                                    Value: 'the/place'
                                }
                            ]
                        }
                    },
                    TopicArn: {
                        Ref: "tests3snsSnsTopicF02F6BD0"
                    }
                }
            ]
        }
    });
});
test('Topic is encrypted with imported CMK when set on encryptionKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = defaults.buildEncryptionKey(stack, {
        description: 'existing-key-description'
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        encryptionKey: cmk
    });
    core_1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SNS::Topic', 'existing-key-description');
});
test('Topic is encrypted with provided encryptionKeyProps', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        encryptionKeyProps: {
            description: 'existing-key-description'
        }
    });
    core_1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SNS::Topic', 'existing-key-description');
});
test('Topic is encrypted with imported CMK when set on topicProps.masterKey prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const cmk = defaults.buildEncryptionKey(stack, {
        description: 'existing-key-description'
    });
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        topicProps: {
            masterKey: cmk
        }
    });
    core_1.expectKmsKeyAttachedToCorrectResource(stack, 'AWS::SNS::Topic', 'existing-key-description');
});
test('Topic is encrypted by default with Customer-managed KMS key when no other encryption properties are set', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {});
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::GetAtt": [
                "tests3snsEncryptionKey6C553584",
                "Arn"
            ]
        }
    });
    expect(stack).toCountResources('AWS::KMS::Key', 1);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
});
test('Topic is encrypted with SQS-managed KMS Key when enable encryption flag is false', () => {
    const stack = new aws_cdk_lib_1.Stack();
    new lib_1.S3ToSns(stack, 'test-s3-sns', {
        enableEncryptionWithCustomerManagedKey: false
    });
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        KmsMasterKeyId: {
            "Fn::Join": [
                "",
                [
                    "arn:",
                    {
                        Ref: "AWS::Partition"
                    },
                    ":kms:",
                    {
                        Ref: "AWS::Region"
                    },
                    ":",
                    {
                        Ref: "AWS::AccountId"
                    },
                    ":alias/aws/sns"
                ]
            ]
        }
    });
    expect(stack).toCountResources('AWS::KMS::Key', 0);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
});
test('Construct does not override unencrypted topic when passed in existingTopicObj prop', () => {
    const stack = new aws_cdk_lib_1.Stack();
    const existingTopicObj = new sns.Topic(stack, 'Topic', {
        topicName: 'existing-topic-name'
    });
    const testConstruct = new lib_1.S3ToSns(stack, 'test-s3-sns', {
        existingTopicObj
    });
    expect(testConstruct.snsTopic).toBeDefined();
    expect(testConstruct.encryptionKey).not.toBeDefined();
    expect(stack).toCountResources('AWS::KMS::Key', 0);
    expect(stack).toCountResources('AWS::SNS::Topic', 1);
    expect(stack).toHaveResource("AWS::SNS::Topic", {
        TopicName: 'existing-topic-name'
    });
});
//# sourceMappingURL=data:application/json;base64,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