"""Test command implementation for running pytest tests."""

import click
from click.testing import CliRunner

from lintro.utils.tool_executor import run_lint_tools_simple

# Constants
DEFAULT_PATHS: list[str] = ["."]
DEFAULT_EXIT_CODE: int = 0
DEFAULT_ACTION: str = "test"


def _ensure_pytest_prefix(option_fragment: str) -> str:
    """Normalize tool option fragments to use the pytest prefix.

    Args:
        option_fragment: Raw option fragment from --tool-options.

    Returns:
        str: Fragment guaranteed to start with ``pytest:``.
    """
    fragment = option_fragment.strip()
    if not fragment:
        return fragment

    lowered = fragment.lower()
    if lowered.startswith("pytest:"):
        _, rest = fragment.split(":", 1)
        return f"pytest:{rest}"
    return f"pytest:{fragment}"


@click.command("test")
@click.argument("paths", nargs=-1, type=click.Path(exists=True))
@click.option(
    "--exclude",
    type=str,
    help="Comma-separated list of patterns to exclude from testing",
)
@click.option(
    "--include-venv",
    is_flag=True,
    help="Include virtual environment directories in testing",
)
@click.option(
    "--output",
    type=click.Path(),
    help="Output file path for writing results",
)
@click.option(
    "--output-format",
    type=click.Choice(["plain", "grid", "markdown", "html", "json", "csv"]),
    default="grid",
    help="Output format for displaying results",
)
@click.option(
    "--group-by",
    type=click.Choice(["file", "code", "none", "auto"]),
    default="file",
    help="How to group issues in the output",
)
@click.option(
    "--verbose",
    is_flag=True,
    help="Show verbose output",
)
@click.option(
    "--raw-output",
    is_flag=True,
    help="Show raw tool output instead of formatted output",
)
@click.option(
    "--tool-options",
    type=str,
    help="Tool-specific options in the format option=value,option=value",
)
@click.option(
    "--enable-docker",
    is_flag=True,
    default=False,
    help="Enable Docker-specific tests",
)
@click.option(
    "--list-plugins",
    is_flag=True,
    default=False,
    help="List all installed pytest plugins",
)
@click.option(
    "--check-plugins",
    is_flag=True,
    default=False,
    help=(
        "Check if required plugins are installed "
        "(use with --tool-options pytest:required_plugins=plugin1,plugin2)"
    ),
)
@click.option(
    "--collect-only",
    is_flag=True,
    default=False,
    help="List tests without executing them",
)
@click.option(
    "--fixtures",
    is_flag=True,
    default=False,
    help="List all available fixtures",
)
@click.option(
    "--fixture-info",
    type=str,
    default=None,
    help="Show detailed information about a specific fixture",
)
@click.option(
    "--markers",
    is_flag=True,
    default=False,
    help="List all available markers",
)
@click.option(
    "--parametrize-help",
    is_flag=True,
    default=False,
    help="Show help for parametrized tests",
)
def test_command(
    paths,
    exclude,
    include_venv,
    output,
    output_format,
    group_by,
    verbose,
    raw_output,
    tool_options,
    enable_docker,
    list_plugins,
    check_plugins,
    collect_only,
    fixtures,
    fixture_info,
    markers,
    parametrize_help,
) -> None:
    """Run tests using pytest.

    Args:
        paths: tuple: List of file/directory paths to test.
        exclude: str | None: Comma-separated patterns of files/dirs to exclude.
        include_venv: bool: Whether to include virtual environment directories.
        output: str | None: Path to output file for results.
        output_format: str: Format for displaying results (grid, json, etc).
        group_by: str: How to group issues in output (file, code, etc).
        verbose: bool: Whether to show verbose output during execution.
        raw_output: bool: Whether to show raw tool output instead of formatted output.
        tool_options: str | None: Tool-specific options.
        enable_docker: bool: Whether to enable Docker-specific tests.
        list_plugins: bool: Whether to list installed pytest plugins.
        check_plugins: bool: Whether to check required plugins.
        collect_only: bool: Whether to list tests without executing.
        fixtures: bool: Whether to list available fixtures.
        fixture_info: str | None: Name of fixture to get info for.
        markers: bool: Whether to list available markers.
        parametrize_help: bool: Whether to show parametrization help.

    Raises:
        SystemExit: Process exit with the aggregated exit code.
    """
    # Add default paths if none provided
    if not paths:
        paths = DEFAULT_PATHS

    # Build tool options with pytest prefix
    tool_option_parts: list[str] = []

    # Add Docker enable option if flag is set
    if enable_docker:
        tool_option_parts.append("pytest:run_docker_tests=True")

    # Add special mode flags
    boolean_flags: list[tuple[bool, str]] = [
        (list_plugins, "pytest:list_plugins=True"),
        (check_plugins, "pytest:check_plugins=True"),
        (collect_only, "pytest:collect_only=True"),
        (fixtures, "pytest:list_fixtures=True"),
        (markers, "pytest:list_markers=True"),
        (parametrize_help, "pytest:parametrize_help=True"),
    ]

    for flag_value, option_string in boolean_flags:
        if flag_value:
            tool_option_parts.append(option_string)

    # Handle fixture_info as special case (requires non-empty value)
    if fixture_info:
        tool_option_parts.append(f"pytest:fixture_info={fixture_info}")

    if tool_options:
        # Prefix with "pytest:" for pytest tool
        # Parse options carefully to handle values containing commas
        # Format: key=value,key=value where values can contain commas
        prefixed_options: list[str] = []
        parts = tool_options.split(",")
        i = 0

        while i < len(parts):
            current_part = parts[i].strip()
            if not current_part:
                i += 1
                continue

            # Check if this part looks like a complete option (contains =)
            # or starts with pytest prefix (already namespaced)
            if "=" in current_part or current_part.lower().startswith("pytest:"):
                normalized_part = _ensure_pytest_prefix(current_part)
                prefixed_options.append(normalized_part)
                i += 1
            else:
                # This part doesn't have =, might be a value continuation
                # Merge with previous part if it exists and had an =
                if prefixed_options and "=" in prefixed_options[-1]:
                    # Merge with previous option's value
                    prefixed_options[-1] = f"{prefixed_options[-1]},{current_part}"
                else:
                    # Standalone option without =, prefix it
                    prefixed_options.append(_ensure_pytest_prefix(current_part))
                i += 1

        tool_option_parts.append(",".join(prefixed_options))

    combined_tool_options: str | None = (
        ",".join(tool_option_parts) if tool_option_parts else None
    )

    # Run with pytest tool
    exit_code: int = run_lint_tools_simple(
        action=DEFAULT_ACTION,
        paths=list(paths),
        tools="pytest",
        tool_options=combined_tool_options,
        exclude=exclude,
        include_venv=include_venv,
        group_by=group_by,
        output_format=output_format,
        verbose=verbose,
        raw_output=raw_output,
        output_file=output,
    )

    # Exit with code only
    raise SystemExit(exit_code)


# Exclude from pytest collection - this is a Click command, not a test function
test_command.__test__ = False


def test(
    paths,
    exclude,
    include_venv,
    output,
    output_format,
    group_by,
    verbose,
    raw_output=False,
    tool_options=None,
) -> None:
    """Programmatic test function for backward compatibility.

    Args:
        paths: tuple: List of file/directory paths to test.
        exclude: str | None: Comma-separated patterns of files/dirs to exclude.
        include_venv: bool: Whether to include virtual environment directories.
        output: str | None: Path to output file for results.
        output_format: str: Format for displaying results.
        group_by: str: How to group issues in output.
        verbose: bool: Whether to show verbose output during execution.
        raw_output: bool: Whether to show raw tool output instead of formatted output.
        tool_options: str | None: Tool-specific options.

    Returns:
        None: This function does not return a value.
    """
    # Build arguments for the click command
    args: list[str] = []
    if paths:
        args.extend(paths)
    if exclude:
        args.extend(["--exclude", exclude])
    if include_venv:
        args.append("--include-venv")
    if output:
        args.extend(["--output", output])
    if output_format:
        args.extend(["--output-format", output_format])
    if group_by:
        args.extend(["--group-by", group_by])
    if verbose:
        args.append("--verbose")
    if raw_output:
        args.append("--raw-output")
    if tool_options:
        args.extend(["--tool-options", tool_options])

    runner = CliRunner()
    result = runner.invoke(test_command, args)

    if result.exit_code != DEFAULT_EXIT_CODE:
        import sys

        sys.exit(result.exit_code)
    return None
