import * as cloudfront from '@aws-cdk/aws-cloudfront';
import * as iam from '@aws-cdk/aws-iam';
import * as s3 from '@aws-cdk/aws-s3';
import * as cdk from '@aws-cdk/core';
import { ISource } from './source';
export interface BucketDeploymentProps {
    /**
     * The sources from which to deploy the contents of this bucket.
     */
    readonly sources: ISource[];
    /**
     * The S3 bucket to sync the contents of the zip file to.
     */
    readonly destinationBucket: s3.IBucket;
    /**
     * Key prefix in the destination bucket.
     *
     * @default "/" (unzip to root of the destination bucket)
     */
    readonly destinationKeyPrefix?: string;
    /**
     * If this is set to "false", the destination files will be deleted when the
     * resource is deleted or the destination is updated.
     *
     * NOTICE: if this is set to "false" and destination bucket/prefix is updated,
     * all files in the previous destination will first be deleted and then
     * uploaded to the new destination location. This could have availablity
     * implications on your users.
     *
     * @default true - when resource is deleted/updated, files are retained
     */
    readonly retainOnDelete?: boolean;
    /**
     * The CloudFront distribution using the destination bucket as an origin.
     * Files in the distribution's edge caches will be invalidated after
     * files are uploaded to the destination bucket.
     *
     * @default - No invalidation occurs
     */
    readonly distribution?: cloudfront.IDistribution;
    /**
     * The file paths to invalidate in the CloudFront distribution.
     *
     * @default - All files under the destination bucket key prefix will be invalidated.
     */
    readonly distributionPaths?: string[];
    /**
     * The amount of memory (in MiB) to allocate to the AWS Lambda function which
     * replicates the files from the CDK bucket to the destination bucket.
     *
     * If you are deploying large files, you will need to increase this number
     * accordingly.
     *
     * @default 128
     */
    readonly memoryLimit?: number;
    /**
     * Execution role associated with this function
     *
     * @default - A role is automatically created
     */
    readonly role?: iam.IRole;
    /**
     * User-defined object metadata to be set on all objects in the deployment
     * @default - No user metadata is set
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#UserMetadata
     */
    readonly metadata?: UserDefinedObjectMetadata;
    /**
     * System-defined cache-control metadata to be set on all objects in the deployment.
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly cacheControl?: CacheControl[];
    /**
     * System-defined cache-disposition metadata to be set on all objects in the deployment.
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly contentDisposition?: string;
    /**
     * System-defined content-encoding metadata to be set on all objects in the deployment.
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly contentEncoding?: string;
    /**
     * System-defined content-language metadata to be set on all objects in the deployment.
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly contentLanguage?: string;
    /**
     * System-defined content-type metadata to be set on all objects in the deployment.
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly contentType?: string;
    /**
     * System-defined expires metadata to be set on all objects in the deployment.
     * @default - The objects in the distribution will not expire.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly expires?: Expires;
    /**
     * System-defined x-amz-server-side-encryption metadata to be set on all objects in the deployment.
     * @default - Server side encryption is not used.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly serverSideEncryption?: ServerSideEncryption;
    /**
     * System-defined x-amz-storage-class metadata to be set on all objects in the deployment.
     * @default - Default storage-class for the bucket is used.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly storageClass?: StorageClass;
    /**
     * System-defined x-amz-website-redirect-location metadata to be set on all objects in the deployment.
     * @default - No website redirection.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly websiteRedirectLocation?: string;
    /**
     * System-defined x-amz-server-side-encryption-aws-kms-key-id metadata to be set on all objects in the deployment.
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
     */
    readonly serverSideEncryptionAwsKmsKeyId?: string;
    /**
     * System-defined x-amz-server-side-encryption-customer-algorithm metadata to be set on all objects in the deployment.
     * Warning: This is not a useful parameter until this bug is fixed: https://github.com/aws/aws-cdk/issues/6080
     * @default - Not set.
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/ServerSideEncryptionCustomerKeys.html#sse-c-how-to-programmatically-intro
     */
    readonly serverSideEncryptionCustomerAlgorithm?: string;
}
export declare class BucketDeployment extends cdk.Construct {
    constructor(scope: cdk.Construct, id: string, props: BucketDeploymentProps);
    private renderSingletonUuid;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
export declare class CacheControl {
    readonly value: any;
    static mustRevalidate(): CacheControl;
    static noCache(): CacheControl;
    static noTransform(): CacheControl;
    static setPublic(): CacheControl;
    static setPrivate(): CacheControl;
    static proxyRevalidate(): CacheControl;
    static maxAge(t: cdk.Duration): CacheControl;
    static sMaxAge(t: cdk.Duration): CacheControl;
    static fromString(s: string): CacheControl;
    private constructor();
}
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
export declare enum ServerSideEncryption {
    AES_256 = "AES256",
    AWS_KMS = "aws:kms"
}
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
export declare enum StorageClass {
    STANDARD = "STANDARD",
    REDUCED_REDUNDANCY = "REDUCED_REDUNDANCY",
    STANDARD_IA = "STANDARD_IA",
    ONEZONE_IA = "ONEZONE_IA",
    INTELLIGENT_TIERING = "INTELLIGENT_TIERING",
    GLACIER = "GLACIER",
    DEEP_ARCHIVE = "DEEP_ARCHIVE"
}
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
export declare class Expires {
    readonly value: any;
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d: Date): Expires;
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t: number): Expires;
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t: cdk.Duration): Expires;
    static fromString(s: string): Expires;
    private constructor();
}
export interface UserDefinedObjectMetadata {
    /**
     * Arbitrary metadata key-values
     * Keys must begin with `x-amzn-meta-` (will be added automatically if not provided)
     * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#UserMetadata
     */
    readonly [key: string]: string;
}
