import datetime
import pytest

@pytest.fixture
def sinex_data():
    """ Dictionary with the same structure as returned from a sinex_site parser
    
    This data structure does not contain information about station coordinates
    
    Contains two examples with stations zimm and hrao
    """
    data = dict()
    
    data["zimm"] = {'site_id': {'site_code': 'zimm',
              'point_code': 'A',
              'domes': '14001',
              'marker': 'M004',
              'obs_code': 'P',
              'description': 'Zimmerwald, Switzerlan',
              'approx_lon': 7.465277777777778,
              'approx_lat': 46.87708333333334,
              'approx_height': 956.4},
             'site_receiver': [{'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1993, 5, 1, 0, 0),
               'end_time': datetime.datetime(1997, 8, 6, 0, 0),
               'receiver_type': 'TRIMBLE 4000SSE',
               'serial_number': '3310A',
               'firmware': '6.12'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1997, 8, 6, 0, 0),
               'end_time': datetime.datetime(1999, 8, 16, 0, 0),
               'receiver_type': 'TRIMBLE 4000SSI',
               'serial_number': '3310A',
               'firmware': '7.25'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1999, 8, 16, 0, 0),
               'end_time': datetime.datetime(2003, 8, 12, 14, 0),
               'receiver_type': 'TRIMBLE 4000SSI',
               'serial_number': '3310A',
               'firmware': '7.29'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2003, 8, 12, 14, 0),
               'end_time': datetime.datetime(2006, 2, 22, 8, 0),
               'receiver_type': 'TRIMBLE 4700',
               'serial_number': '20224',
               'firmware': 'Nav  1.30 /'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2006, 2, 22, 14, 0),
               'end_time': datetime.datetime(2008, 4, 1, 14, 10),
               'receiver_type': 'TRIMBLE NETRS',
               'serial_number': '45262',
               'firmware': 'Nav  1.15 /'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2008, 4, 1, 14, 10),
               'end_time': datetime.datetime(2014, 5, 7, 7, 20),
               'receiver_type': 'TRIMBLE NETRS',
               'serial_number': '45262',
               'firmware': 'Nav  1.20'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2014, 5, 7, 7, 20),
               'end_time': datetime.datetime(2015, 4, 27, 7, 15),
               'receiver_type': 'TRIMBLE NETRS',
               'serial_number': '45262',
               'firmware': '1.32'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2015, 4, 27, 7, 15),
               'end_time': datetime.datetime(2015, 6, 19, 6, 30),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '4.85'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2015, 6, 19, 6, 30),
               'end_time': datetime.datetime(2015, 8, 11, 8, 45),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.01'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2015, 8, 11, 8, 45),
               'end_time': datetime.datetime(2015, 12, 16, 9, 20),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.03'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2015, 12, 16, 9, 20),
               'end_time': datetime.datetime(2017, 2, 6, 8, 20),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.10'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2017, 2, 6, 8, 20),
               'end_time': datetime.datetime(2017, 2, 8, 11, 8),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.20'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2017, 2, 8, 11, 8),
               'end_time': datetime.datetime(2017, 4, 11, 12, 5),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.10'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2017, 4, 11, 12, 5),
               'end_time': datetime.datetime(2018, 12, 18, 9, 45),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.22'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2018, 12, 18, 9, 45),
               'end_time': datetime.datetime(2020, 6, 8, 7, 15),
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.37'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2020, 6, 8, 7, 15),
               'end_time': None,
               'receiver_type': 'TRIMBLE NETR9',
               'serial_number': '5429R',
               'firmware': '5.45'}],
             'site_antenna': [{'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1993, 5, 1, 0, 0),
               'end_time': datetime.datetime(1998, 11, 6, 0, 0),
               'antenna_type': 'TRM14532.00',
               'serial_number': '3311A',
               'radome_type': 'NONE'},
              {'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1998, 11, 6, 0, 0),
               'end_time': None,
               'antenna_type': 'TRM29659.00',
               'serial_number': '99390',
               'radome_type': 'NONE'}],
             'site_eccentricity': [{'site_code': 'zimm',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1993, 5, 1, 0, 0),
               'end_time': None,
               'vector_type': 'UNE',
               'vector_1': 0.0,
               'vector_2': 0.0,
               'vector_3': 0.0}]}
    
    data["hrao"] = {'site_id': {'site_code': 'hrao',
              'point_code': 'A',
              'domes': '30302',
              'marker': 'M004',
              'obs_code': 'P',
              'description': 'Krugersdorp, South Afr',
              'approx_lon': 27.686972222222224,
              'approx_lat': -25.89011111111111,
              'approx_height': 1414.3},
             'site_receiver': [{'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1996, 9, 5, 0, 0),
               'end_time': datetime.datetime(1998, 1, 14, 0, 0),
               'receiver_type': 'ROGUE SNR-8000',
               'serial_number': 'T365',
               'firmware': '3.2 link 03'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1998, 1, 14, 0, 0),
               'end_time': datetime.datetime(1999, 5, 25, 12, 30),
               'receiver_type': 'ROGUE SNR-12 RM',
               'serial_number': '254',
               'firmware': '3.2.32.4'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1999, 5, 25, 12, 30),
               'end_time': datetime.datetime(2000, 4, 16, 0, 0),
               'receiver_type': 'ROGUE SNR-12 RM',
               'serial_number': '254',
               'firmware': '3.2.32.8'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2000, 4, 26, 12, 30),
               'end_time': datetime.datetime(2006, 12, 28, 0, 0),
               'receiver_type': 'ASHTECH Z-XII3',
               'serial_number': 'LP019',
               'firmware': 'CC00 1s soc'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2007, 4, 26, 0, 0),
               'end_time': datetime.datetime(2007, 12, 11, 16, 22),
               'receiver_type': 'ASHTECH UZ-12',
               'serial_number': 'ZR520',
               'firmware': 'CQ00'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2008, 2, 14, 9, 0),
               'end_time': datetime.datetime(2009, 1, 10, 10, 59),
               'receiver_type': 'ASHTECH UZ-12',
               'serial_number': 'ZR220',
               'firmware': 'CQ00'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2009, 1, 15, 0, 0),
               'end_time': datetime.datetime(2012, 3, 30, 17, 15),
               'receiver_type': 'ASHTECH UZ-12',
               'serial_number': 'UC120',
               'firmware': 'CN00'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2012, 4, 12, 0, 0),
               'end_time': datetime.datetime(2014, 7, 23, 1, 30),
               'receiver_type': 'ASHTECH Z-XII3',
               'serial_number': 'LP008',
               'firmware': 'CD00'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2014, 7, 23, 1, 35),
               'end_time': datetime.datetime(2014, 8, 21, 16, 39),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.5.5'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2014, 8, 21, 16, 40),
               'end_time': datetime.datetime(2014, 10, 22, 22, 0),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.5.6'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2014, 10, 22, 22, 15),
               'end_time': datetime.datetime(2015, 1, 21, 20, 39),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.5.7b0'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2015, 1, 21, 20, 40),
               'end_time': datetime.datetime(2016, 2, 19, 17, 5),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.5.8b0-12-'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2016, 2, 19, 17, 15),
               'end_time': datetime.datetime(2018, 4, 10, 17, 0),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.6.4'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2018, 4, 10, 17, 15),
               'end_time': datetime.datetime(2018, 12, 27, 17, 11),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.7.2'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2018, 12, 27, 17, 15),
               'end_time': datetime.datetime(2019, 5, 23, 22, 51),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.7.4'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2019, 5, 23, 22, 51),
               'end_time': datetime.datetime(2020, 3, 24, 20, 52),
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.7.6'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2020, 3, 24, 20, 52),
               'end_time': None,
               'receiver_type': 'JAVAD TRE_G3TH DELTA',
               'serial_number': '01707',
               'firmware': '3.7.9'}],
             'site_antenna': [{'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1996, 9, 5, 0, 0),
               'end_time': datetime.datetime(1998, 1, 14, 0, 0),
               'antenna_type': 'AOAD/M_T',
               'serial_number': '200',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1998, 1, 14, 0, 0),
               'end_time': datetime.datetime(1999, 2, 20, 0, 0),
               'antenna_type': 'AOAD/M_T',
               'serial_number': '201',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1999, 2, 20, 0, 0),
               'end_time': datetime.datetime(1999, 7, 13, 7, 0),
               'antenna_type': 'AOAD/M_T',
               'serial_number': '127',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1999, 7, 13, 7, 0),
               'end_time': datetime.datetime(2003, 10, 17, 8, 55),
               'antenna_type': 'AOAD/M_T',
               'serial_number': '201',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2003, 10, 17, 9, 0),
               'end_time': datetime.datetime(2004, 11, 19, 10, 30),
               'antenna_type': 'AOAD/M_T',
               'serial_number': '388',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2004, 11, 19, 10, 30),
               'end_time': datetime.datetime(2005, 4, 24, 0, 0),
               'antenna_type': 'ASH701945C_M',
               'serial_number': 'CR520',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2005, 5, 3, 9, 0),
               'end_time': datetime.datetime(2006, 2, 13, 12, 30),
               'antenna_type': 'ASH701945C_M',
               'serial_number': 'CR520',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2006, 2, 14, 10, 30),
               'end_time': datetime.datetime(2006, 2, 21, 18, 0),
               'antenna_type': 'ASH701945C_M',
               'serial_number': 'CR620',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2006, 2, 23, 0, 0),
               'end_time': datetime.datetime(2006, 12, 28, 0, 0),
               'antenna_type': 'ASH701945C_M',
               'serial_number': 'CR519',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2007, 4, 26, 0, 0),
               'end_time': datetime.datetime(2007, 12, 11, 16, 22),
               'antenna_type': 'ASH701945C_M',
               'serial_number': 'CR520',
               'radome_type': 'NONE'},
              {'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(2008, 2, 14, 9, 0),
               'end_time': None,
               'antenna_type': 'ASH701945E_M',
               'serial_number': 'CR620',
               'radome_type': 'NONE'}],
             'site_eccentricity': [{'site_code': 'hrao',
               'point_code': 'A',
               'soln': '----',
               'obs_code': 'P',
               'start_time': datetime.datetime(1996, 9, 5, 0, 0),
               'end_time': None,
               'vector_type': 'UNE',
               'vector_1': 0.0814,
               'vector_2': 0.0,
               'vector_3': 0.0}]}

    return data

@pytest.fixture
def sinex_data_site_coord():
    """ Dictionary with the same structure as returned from a sinex_site parser
    
    This data structure contains information about station coordinates
    
    Contains two examples with station kiri
    """
    data = dict()
    data["kiri"] = {'site_id': {'site_code': 'KIRI',
          'point_code': 'A',
          'domes': '50305',
          'marker': 'M001',
          'obs_code': 'C',
          'description': 'BETIO ISLAND - KIRIBAT',
          'approx_lon': 172.92288888888888,
          'approx_lat': 1.3545555555555557,
          'approx_height': 36.2},
         'site_receiver': [{'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2002, 8, 3, 0, 0),
           'end_time': datetime.datetime(2002, 10, 8, 0, 0),
           'receiver_type': 'ASHTECH UZ-12',
           'serial_number': 'ZR220',
           'firmware': ''},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2002, 10, 8, 0, 0),
           'end_time': datetime.datetime(2003, 9, 26, 0, 0),
           'receiver_type': 'ASHTECH UZ-12',
           'serial_number': '14506',
           'firmware': ''},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2003, 10, 5, 0, 0),
           'end_time': datetime.datetime(2005, 6, 1, 0, 0),
           'receiver_type': 'ASHTECH UZ-12',
           'serial_number': 'ZR220',
           'firmware': ''},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2005, 6, 1, 0, 0),
           'end_time': datetime.datetime(2006, 11, 3, 11, 42),
           'receiver_type': 'ASHTECH UZ-12',
           'serial_number': 'UC220',
           'firmware': ''},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2006, 11, 3, 11, 42),
           'end_time': datetime.datetime(2008, 3, 15, 0, 0),
           'receiver_type': 'ASHTECH UZ-12',
           'serial_number': 'UC220',
           'firmware': 'ZC00'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2008, 3, 15, 0, 0),
           'end_time': datetime.datetime(2011, 11, 19, 22, 30),
           'receiver_type': 'ASHTECH UZ-12',
           'serial_number': 'ZR220',
           'firmware': 'CJ00'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2011, 11, 20, 1, 0),
           'end_time': datetime.datetime(2012, 8, 31, 2, 44),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '4.42'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2012, 8, 31, 2, 44),
           'end_time': datetime.datetime(2015, 4, 21, 0, 0),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '4.61'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2015, 4, 21, 0, 0),
           'end_time': datetime.datetime(2015, 6, 19, 0, 0),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '4.93'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2015, 6, 19, 0, 0),
           'end_time': datetime.datetime(2016, 1, 7, 0, 0),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.01'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2016, 1, 7, 0, 0),
           'end_time': datetime.datetime(2016, 12, 16, 0, 0),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.10'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2016, 12, 16, 0, 0),
           'end_time': datetime.datetime(2017, 4, 27, 0, 0),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.15'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2017, 4, 27, 0, 0),
           'end_time': datetime.datetime(2019, 4, 9, 0, 30),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.22'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2019, 4, 9, 0, 30),
           'end_time': datetime.datetime(2020, 3, 2, 6, 49),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.37'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2020, 3, 2, 6, 49),
           'end_time': datetime.datetime(2020, 3, 3, 0, 38),
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.42'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2020, 3, 3, 0, 38),
           'end_time': None,
           'receiver_type': 'TRIMBLE NETR9',
           'serial_number': '5038K',
           'firmware': '5.44'}],
         'site_antenna': [{'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2002, 8, 3, 0, 0),
           'end_time': datetime.datetime(2011, 11, 20, 1, 0),
           'antenna_type': 'ASH701945C_M',
           'serial_number': 'CR620',
           'radome_type': 'SCIS'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2011, 11, 20, 1, 0),
           'end_time': datetime.datetime(2014, 11, 18, 0, 0),
           'antenna_type': 'TRM59800.00',
           'serial_number': '50373',
           'radome_type': 'NONE'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2014, 11, 18, 0, 0),
           'end_time': datetime.datetime(2015, 4, 28, 0, 0),
           'antenna_type': 'TRM59800.00',
           'serial_number': '4844A',
           'radome_type': 'NONE'},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2015, 4, 28, 0, 0),
           'end_time': None,
           'antenna_type': 'TRM59800.00',
           'serial_number': '54213',
           'radome_type': 'NONE'}],
         'site_eccentricity': [{'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2002, 8, 3, 0, 0),
           'end_time': datetime.datetime(2011, 11, 20, 1, 0),
           'vector_type': 'UNE',
           'vector_1': 0.0,
           'vector_2': 0.0,
           'vector_3': 0.0},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2011, 11, 20, 1, 0),
           'end_time': datetime.datetime(2014, 11, 18, 0, 0),
           'vector_type': 'UNE',
           'vector_1': 0.0008,
           'vector_2': 0.0,
           'vector_3': 0.0},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2014, 11, 18, 0, 0),
           'end_time': datetime.datetime(2015, 4, 28, 0, 0),
           'vector_type': 'UNE',
           'vector_1': 0.003,
           'vector_2': 0.0,
           'vector_3': 0.0},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '----',
           'obs_code': 'C',
           'start_time': datetime.datetime(2015, 4, 28, 0, 0),
           'end_time': None,
           'vector_type': 'UNE',
           'vector_1': 0.0038,
           'vector_2': 0.0,
           'vector_3': 0.0}],
         'solution_epochs': [{'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'obs_code': 'C',
           'start_epoch': datetime.datetime(2002, 8, 3, 0, 0),
           'end_epoch': datetime.datetime(2011, 11, 20, 23, 59, 30),
           'mean_epoch': datetime.datetime(2007, 3, 28, 11, 59, 45)},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'obs_code': 'C',
           'start_epoch': datetime.datetime(2011, 11, 21, 0, 0),
           'end_epoch': datetime.datetime(2014, 11, 17, 23, 59, 30),
           'mean_epoch': datetime.datetime(2013, 5, 20, 11, 59, 45)},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'obs_code': 'C',
           'start_epoch': datetime.datetime(2014, 11, 18, 0, 0),
           'end_epoch': datetime.datetime(2015, 4, 27, 23, 59, 30),
           'mean_epoch': datetime.datetime(2015, 2, 6, 11, 59, 45)},
          {'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'obs_code': 'C',
           'start_epoch': datetime.datetime(2015, 4, 28, 0, 0),
           'end_epoch': datetime.datetime(2020, 4, 12, 12, 0),
           'mean_epoch': datetime.datetime(2017, 10, 19, 18, 0)}],
         'solution_estimate': [{'param_idx': 2437,
           'param_name': 'STAX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': -6327822.41344055,
           'estimate_std': 0.00021532},
          {'param_idx': 2438,
           'param_name': 'STAY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 785604.5037294,
           'estimate_std': 6.037e-05},
          {'param_idx': 2439,
           'param_name': 'STAZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 149769.243423874,
           'estimate_std': 5.0163e-05},
          {'param_idx': 2440,
           'param_name': 'VELX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.00915878328902547,
           'estimate_std': 5.0722e-05},
          {'param_idx': 2441,
           'param_name': 'VELY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0672007804490354,
           'estimate_std': 1.4785e-05},
          {'param_idx': 2442,
           'param_name': 'VELZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '1',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0312437481266807,
           'estimate_std': 1.2096e-05},
          {'param_idx': 2443,
           'param_name': 'STAX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': -6327822.40683967,
           'estimate_std': 0.00051379},
          {'param_idx': 2444,
           'param_name': 'STAY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 785604.499418105,
           'estimate_std': 0.00014701},
          {'param_idx': 2445,
           'param_name': 'STAZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 149769.237641736,
           'estimate_std': 0.00011541},
          {'param_idx': 2446,
           'param_name': 'VELX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.009159150665851,
           'estimate_std': 5.0769e-05},
          {'param_idx': 2447,
           'param_name': 'VELY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0672009068537374,
           'estimate_std': 1.4965e-05},
          {'param_idx': 2448,
           'param_name': 'VELZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '2',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.031243585437167,
           'estimate_std': 1.2312e-05},
          {'param_idx': 2449,
           'param_name': 'STAX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': -6327822.39473399,
           'estimate_std': 0.0019184},
          {'param_idx': 2450,
           'param_name': 'STAY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 785604.499453199,
           'estimate_std': 0.00047128},
          {'param_idx': 2451,
           'param_name': 'STAZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 149769.235876838,
           'estimate_std': 0.00035477},
          {'param_idx': 2452,
           'param_name': 'VELX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.00915950854733218,
           'estimate_std': 5.0817e-05},
          {'param_idx': 2453,
           'param_name': 'VELY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0672011192002233,
           'estimate_std': 1.5144e-05},
          {'param_idx': 2454,
           'param_name': 'VELZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '3',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0312437640507729,
           'estimate_std': 1.2523e-05},
          {'param_idx': 2455,
           'param_name': 'STAX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': -6327822.40890791,
           'estimate_std': 0.00042754},
          {'param_idx': 2456,
           'param_name': 'STAY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 785604.505731186,
           'estimate_std': 0.00012964},
          {'param_idx': 2457,
           'param_name': 'STAZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm',
           'constraint': '2',
           'estimate': 149769.239739762,
           'estimate_std': 0.00010695},
          {'param_idx': 2458,
           'param_name': 'VELX',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.00915986633078941,
           'estimate_std': 5.0865e-05},
          {'param_idx': 2459,
           'param_name': 'VELY',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0672013306335172,
           'estimate_std': 1.5316e-05},
          {'param_idx': 2460,
           'param_name': 'VELZ',
           'site_code': 'KIRI',
           'point_code': 'A',
           'soln': '4',
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'unit': 'm/y',
           'constraint': '2',
           'estimate': 0.0312439382692589,
           'estimate_std': 1.2724e-05}]}

    return data


@pytest.fixture
def ssc_data():
    """ Dictionary with the same structure as returned from a ssc_site parser
    
    Contains two examples with stations GRAS and BORR
    """
    data = dict()
    
    data["GRAS"] = {'site_num': '10002',
         'antenna_num': 'M006',
         'name': 'GRAS',
         'tech': 'GPS',
         'antenna_id': 'GRAS',
         'soln': 4,
         'pos_vel': {1: {'STAX': 4581690.826,
           'STAY': 556114.926,
           'STAZ': 4389360.845,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(1996, 1, 1, 0, 0),
           'end': datetime.datetime(1996, 5, 6, 23, 59, 30),
           'VELX': -0.0137,
           'VELY': 0.0189,
           'VELZ': 0.0115,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001},
          2: {'STAX': 4581690.83,
           'STAY': 556114.924,
           'STAZ': 4389360.844,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(1996, 10, 4, 0, 0),
           'end': datetime.datetime(2003, 4, 21, 23, 59, 30),
           'VELX': -0.0137,
           'VELY': 0.0189,
           'VELZ': 0.0115,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001},
          3: {'STAX': 4581690.835,
           'STAY': 556114.927,
           'STAZ': 4389360.849,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(2003, 4, 24, 0, 0),
           'end': datetime.datetime(2004, 10, 19, 23, 59, 30),
           'VELX': -0.0137,
           'VELY': 0.0189,
           'VELZ': 0.0115,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001},
          4: {'STAX': 4581690.831,
           'STAY': 556114.93,
           'STAZ': 4389360.851,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(2004, 10, 22, 0, 0),
           'end': datetime.datetime(2021, 2, 20, 23, 59, 30),
           'VELX': -0.0137,
           'VELY': 0.0189,
           'VELZ': 0.0115,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001}}}

    data["BORR"] = {'site_num': '13480',
         'antenna_num': 'M001',
         'name': 'BORR',
         'tech': 'GPS',
         'antenna_id': 'BORR',
         'soln': 3,
         'pos_vel': {1: {'STAX': 4899519.031,
           'STAY': -7115.451,
           'STAZ': 4069961.71,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(2007, 9, 26, 0, 0),
           'end': datetime.datetime(2013, 12, 28, 23, 59, 30),
           'VELX': -0.0109,
           'VELY': 0.0203,
           'VELZ': 0.0121,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001},
          2: {'STAX': 4899519.033,
           'STAY': -7115.452,
           'STAZ': 4069961.711,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(2013, 12, 29, 0, 0),
           'end': datetime.datetime(2016, 6, 18, 23, 59, 30),
           'VELX': -0.0109,
           'VELY': 0.0203,
           'VELZ': 0.0121,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001},
          3: {'STAX': 4899519.029,
           'STAY': -7115.455,
           'STAZ': 4069961.71,
           'sigma_X': 0.001,
           'sigma_Y': 0.001,
           'sigma_Z': 0.001,
           'ref_epoch': datetime.datetime(2010, 1, 1, 0, 0),
           'start': datetime.datetime(2016, 6, 26, 0, 0),
           'end': datetime.datetime(2021, 2, 20, 23, 59, 30),
           'VELX': -0.0109,
           'VELY': 0.0203,
           'VELZ': 0.0121,
           'sigma_VX': 0.0001,
           'sigma_VY': 0.0001,
           'sigma_VZ': 0.0001}}}

    return data
