"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceSpecDefinition = exports.Protocol = exports.Service = exports.ServiceType = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new ServiceSpecDefinition(props.spec);
        this.apiObject = new k8s.Service(this, 'Pod', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this.spec._toKube() }),
        });
    }
}
exports.Service = Service;
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
/**
 * A description of a service.
 */
class ServiceSpecDefinition {
    constructor(props = {}) {
        var _a, _b, _c;
        this.clusterIP = props.clusterIP;
        this.externalIPs = (_a = props.externalIPs) !== null && _a !== void 0 ? _a : [];
        this.type = (_b = props.type) !== null && _b !== void 0 ? _b : ServiceType.CLUSTER_IP;
        this._ports = [];
        this._selector = {};
        for (const portAndOptions of (_c = props.ports) !== null && _c !== void 0 ? _c : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector() {
        return this._selector;
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port, options = {}) {
        this._ports.push({ port, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0) {
            throw new Error('A service must be configured with a port');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                port: port.port,
                targetPort: port.targetPort,
                nodePort: port.nodePort,
            });
        }
        return {
            clusterIP: this.clusterIP,
            externalIPs: this.externalIPs,
            type: this.type,
            selector: this._selector,
            ports: ports,
        };
    }
}
exports.ServiceSpecDefinition = ServiceSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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