"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DeploymentSpecDefinition = exports.Deployment = void 0;
const k8s = require("./imports/k8s");
const constructs_1 = require("constructs");
const service_1 = require("./service");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
const pod_1 = require("./pod");
const cdk8s_1 = require("cdk8s");
/**
*
* A Deployment provides declarative updates for Pods and ReplicaSets.
*
* You describe a desired state in a Deployment, and the Deployment Controller changes the actual
* state to the desired state at a controlled rate. You can define Deployments to create new ReplicaSets, or to remove
* existing Deployments and adopt all their resources with new Deployments.
*
* > Note: Do not manage ReplicaSets owned by a Deployment. Consider opening an issue in the main Kubernetes repository if your use case is not covered below.
*
* Use Case
* ---------
*
* The following are typical use cases for Deployments:
*
* - Create a Deployment to rollout a ReplicaSet. The ReplicaSet creates Pods in the background.
*   Check the status of the rollout to see if it succeeds or not.
* - Declare the new state of the Pods by updating the PodTemplateSpec of the Deployment.
*   A new ReplicaSet is created and the Deployment manages moving the Pods from the old ReplicaSet to the new one at a controlled rate.
*   Each new ReplicaSet updates the revision of the Deployment.
* - Rollback to an earlier Deployment revision if the current state of the Deployment is not stable.
*   Each rollback updates the revision of the Deployment.
* - Scale up the Deployment to facilitate more load.
* - Pause the Deployment to apply multiple fixes to its PodTemplateSpec and then resume it to start a new rollout.
* - Use the status of the Deployment as an indicator that a rollout has stuck.
* - Clean up older ReplicaSets that you don't need anymore.
*
**/
class Deployment extends base_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.spec = new DeploymentSpecDefinition(props.spec);
        this.apiObject = new k8s.Deployment(this, 'Pod', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this.spec._toKube(this) }),
        });
    }
    /**
     * Expose a deployment via a service.
     *
     * This is equivalent to running `kubectl expose deployment <deployment-name>`.
     *
     * @param options - Options.
     */
    expose(options) {
        var _a;
        const containers = this.spec.podSpecTemplate.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        // create a label and attach it to the deployment pods
        const selector = 'cdk8s.deployment';
        const matcher = constructs_1.Node.of(this).uniqueId;
        const service = new service_1.Service(this, 'Service', {
            spec: {
                type: (_a = options.serviceType) !== null && _a !== void 0 ? _a : service_1.ServiceType.CLUSTER_IP,
            },
        });
        service.spec.addSelector(selector, matcher);
        service.spec.serve(options.port, {
            // just a PoC, we assume the first container is the main one.
            // TODO: figure out what the correct thing to do here.
            targetPort: containers[0].port,
        });
        return service;
    }
}
exports.Deployment = Deployment;
/**
 * DeploymentSpec is the specification of the desired behavior of the Deployment.
 */
class DeploymentSpecDefinition {
    constructor(props = {}) {
        var _a;
        this.replicas = (_a = props.replicas) !== null && _a !== void 0 ? _a : 1;
        this.podSpecTemplate = new pod_1.PodSpecDefinition(props.podSpecTemplate);
        this.podMetadataTemplate = new cdk8s_1.ApiObjectMetadataDefinition(props.podMetadataTemplate);
        this._labelSelector = {};
    }
    /**
     * Configure a label selector to this deployment.
     * Pods that have the label will be selected by deployments configured with this spec.
     *
     * @param key - The label key.
     * @param value - The label value.
     */
    selectByLabel(key, value) {
        this._labelSelector[key] = value;
    }
    /**
     * The labels this deployment will match against in order to select pods.
     *
     * Returns a a copy. Use `selectByLabel()` to add labels.
     */
    get labelSelector() {
        return { ...this._labelSelector };
    }
    /**
     * @internal
     */
    _toKube(deployment) {
        // automatically select pods in this deployment
        const selector = 'cdk8s.deployment';
        const matcher = constructs_1.Node.of(deployment).uniqueId;
        this.podMetadataTemplate.addLabel(selector, matcher);
        this.selectByLabel(selector, matcher);
        return {
            replicas: this.replicas,
            template: {
                metadata: this.podMetadataTemplate.toJson(),
                spec: this.podSpecTemplate._toKube(),
            },
            selector: {
                matchLabels: this._labelSelector,
            },
        };
    }
}
exports.DeploymentSpecDefinition = DeploymentSpecDefinition;
//# sourceMappingURL=data:application/json;base64,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