"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = void 0;
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a by reading a specific key inside a secret.
     *
     * @param secret - The secret.
     * @param key - The key.
     * @param options - Additional options.
     */
    static fromSecret(secret, key, options = {}) {
        const source = {
            secretKeyRef: {
                name: secret.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        var _a, _b, _c, _d;
        this.name = (_a = props.name) !== null && _a !== void 0 ? _a : 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._env = (_b = props.env) !== null && _b !== void 0 ? _b : {};
        this.workingDir = props.workingDir;
        this.mounts = (_c = props.volumeMounts) !== null && _c !== void 0 ? _c : [];
        this.imagePullPolicy = (_d = props.imagePullPolicy) !== null && _d !== void 0 ? _d : ImagePullPolicy.ALWAYS;
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Add an environment value to the container. The variable value can come
     * from various dynamic sources such a secrets of config maps.
     *
     * @see EnvValue.fromXXX
     *
     * @param name - The variable name.
     * @param value - The variable value.
     */
    addEnv(name, value) {
        this._env[name] = value;
    }
    /**
     * The environment variables for this container.
     *
     * Returns a copy. To add environment variables use `addEnv()`.
     */
    get env() {
        return { ...this._env };
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param volume - The volume to mount.
     */
    mount(path, volume, options = {}) {
        this.mounts.push({ path, volume, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports,
            volumeMounts,
            command: this.command,
            workingDir: this.workingDir,
            env: renderEnv(this._env),
        };
    }
}
exports.Container = Container;
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
function renderEnv(env) {
    const result = new Array();
    for (const [name, v] of Object.entries(env)) {
        result.push({
            name,
            value: v.value,
            valueFrom: v.valueFrom,
        });
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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