#import inspect
import os
from os import listdir
from os.path import join, isfile, isdir
#from html.parser import HTMLParser
import json
import gzip 
from datetime import datetime, timedelta
import pandas as pd
#from random import random
import matplotlib.pyplot as plt
#import numpy as np
import requests
from math import ceil
from co2eq.flight_utils import AirportDB, CityDB, FlightDB, GoClimateDB, Flight, logger



plt.rcParams.update({'figure.max_open_warning': 0})



class Meeting:

  def __init__( self, name:str, location:dict, attendee_list:str = None, \
                base_output_dir=None, conf={},  airportDB=True, \
                cityDB=True, flightDB=True, goclimateDB=True ):
    """ Meeting class

    Since conf is used to generate some DB (goclimateDB), DB can either
    be provided as object of set to True to indicate these need to be generated.

    Args:
      name (str): the name of the meeting
      conf (dict): configuration parameters
      airportDB: the airportDB object ( AirportDB ) or True to indicate the DB
        is generated by FlightDB.
        By default it is set to True.
      cityDB: the cityDB object ( CityDB ) or True to indicate the DB is
        generated by FlightDB.
        By default it is set to True.
      goclimateDB: the cityDB object ( GoClimateDB ) or True to indicate the DB
        is generated by FlightDB.
        By default it is set to True. Note that this is the only DB that needs
        the configuration parameters.

    """
    self.name = name
    self.conf = conf
    if airportDB is True:
      self.airportDB = AirportDB()
    else:
      self.airportDB = airportDB
    if  cityDB is True:
      self.cityDB = CityDB( conf, airportDB=self.airportDB )
    else:
      self.cityDB = cityDB
    if goclimateDB is True:
      self.goclimateDB=GoClimateDB( conf )
    else:
      self.goclimateDB = goclimateDB
    ## location is a dictionary with keys iata, city, state, country)
    self.location = location
    ## flightDB and designate it by the destination airport
    if flightDB is True:
      flightDB = FlightDB( conf, cityDB=self.cityDB, airportDB=self.airportDB, \
                           goclimateDB=self.goclimateDB)
    self.flightDB = flightDB
    # print( f"meeting iata location: {self.iata_location}" )
    if base_output_dir is None:
      base_output_dir = self.conf[ 'OUTPUT_DIR' ]
    self.output_dir = join( base_output_dir, self.name )

    if isdir( self.output_dir ) is False:
      os.makedirs( self.output_dir )
    self.logger = logger( conf, __name__ )     
    self.attendee_list = attendee_list

  def get_attendee_list( self ):
    """ return a python list of attendees """
    if isinstance( self.attendee_list, list ):
      attendee_list = self.attendee_list
    elif isfile( self.attendee_list ) is True:
      try:
        with gzip.open( self.attendee_list, 'rt', encoding="utf8") as f:
          attendee_list = json.loads( f.read() )
      except: 
        with open( self.attendee_list, 'rt', encoding="utf8") as f:
          attendee_list = json.loads( f.read() )
    else:
      raise ValueError( f"Unable to generate attendee_list from {self.attendee_list}" )
    return attendee_list 

  def sanity_check_args( self, mode=None, cluster_key=None, co2eq=None  ):
    """ perfoms sanity check for inputs 
 
    """
    if mode in [ 'distance', 'flight' ] and co2eq is None :
      raise ValueError( f"{kwargs}: co2eq cannot be None with mode set to 'distance' or 'flight'" )
    if mode in [ 'attendee' ]:
      co2eq = None
    return mode, cluster_key, co2eq 

  def kwargs_to_file_name( self, suffix:str, extension:str, kwargs:dict ) -> str:
    """ build a file name string from kwargs """
    ## output file_name
    file_name = suffix
    ## removing None values
    for k in  [ k for k in kwargs.keys() if kwargs[ k ] is None ]:
      del kwargs[ k ]
    ## co2eq indicates how co2 is computed, as such it is only meaningful 
    ## when used with the mode 'flight' or 'distance'
    if kwargs[ 'mode' ] == 'attendee':
      try:
        del kwargs[ 'co2eq' ]
      except KeyError:
        pass
    for key, value in kwargs.items():
      file_name += f"-{key}_{value}"
    return file_name + f".{extension}"

  def file_name_to_kwargs( self, file_name ): 
    """ retrieves kwargs from a file name """
    
    cluster_key = None
    co2eq = None
    mode = None
    for seg in file_name.split('-'):
      if 'mode' in seg:
        mode = seg.split( '_' )[1]
      elif 'cluster_key' in seg:
        cluster_key_split = seg.split( '_' )[2:]
        if isinstance( cluster_key_split, list ) :
          cluster_key = ""
          for word in cluster_key_split:
            cluster_key += f" {word}"
        else:
          cluster_key = cluster_key_split
      elif 'co2eq_' in seg:
        co2eq = seg.split( '_' )[1]
 
    return { 'mode' : mode,
             'cluster_key' : cluster_key,
             'co2eq' : co2eq }

  def build_co2eq_data( self, mode='flight', cluster_key=None, co2eq='myclimate' ) -> dict :
    """ co2 equivalent based on real flights including multiple segments)

    The possible modes are 'attendee', 'flight', 'distance'. 
    'flight' and 'distance' always output a number of CO2eq in kg
    'attendee' always outputs a number of attendees
    cluster_key can be any key mention in the participant object withthe additional key 'flight_segment_number'.

    Todo:
    * current distance does not work with goclimate as goclimate seems to only
      take iata airport code, while we provide iata city code.
    """

    mode, cluster_key, co2eq = self.sanity_check_args( mode=mode, \
                                                        cluster_key=cluster_key, co2eq=co2eq )
    kwargs = { 'mode' : mode, 'cluster_key' : cluster_key, 'co2eq' : co2eq }

    data_file = join( self.output_dir, self.kwargs_to_file_name( 'co2eq', 'json.gz', kwargs ) )
    if isfile( data_file ) is True:
      with gzip.open( data_file, 'rt', encoding="utf8" ) as f:
        return json.loads( f.read() )

    ## initializing data
    if cluster_key is None:
      if mode in [ 'attendee' ] :
        y = { 'total attendee' : 0 }  ## return a number
      elif mode in [ 'flight', 'distance' ] :
        y = { 'total co2eq' : 0 }  ## return a number
      else:
        raise ValueError( f"unknown value mode: {mode}" )
    else:
      y = {} ## returns a dict { cluster_value: co2_eq }

    attendee_nbr = 0
    for attendee in self.get_attendee_list( ):
      attendee_nbr +=1 
      self.logger.debug( f"  - attendee {attendee_nbr}: {attendee}" )
      attendee_iata_city = self.cityDB.representative_city( attendee )[ 'iata' ]
      meeting_iata_city = self.cityDB.representative_city( self.location )[ 'iata' ]
      self.logger.debug( f"  - Flight from {attendee_iata_city} to  {meeting_iata_city}" )
      if attendee_iata_city == meeting_iata_city :
        continue
      if attendee_iata_city in [ '', None ]:
        self.logger.debug( "Unable to find city for {attendee}" )
        continue

      if mode == 'attendee' :
        if cluster_key == 'flight_segment_number':
          flight = self.flightDB.select_flight( attendee_iata_city , meeting_iata_city  )
      elif mode in [ 'flight' ]:
        try: 
          flight = self.flightDB.force_select_flight( attendee_iata_city , meeting_iata_city  )
        except :
          if 'country' in attendee.keys() and 'city' not in attendee.keys() and 'state' not in attendee.keys() :
            print( f"\n-----------------------------\n"\
                   f"Unable to retrieve flight: FROM: {attendee_iata_city} "\
                   f"TO: {meeting_iata_city}\n" ) 
            self.debug_country_info( attendee[ 'country' ] )
          raise ValueError( "unable top retrieve flight") 
      elif mode == 'distance':
        segment_list = [ [ attendee_iata_city , meeting_iata_city  ],  \
                         [ attendee_iata_city , meeting_iata_city  ] ]
        ## co2_eq is computed at instantiation
        flight = Flight( segment_list=segment_list, cityDB=self.cityDB, \
                         airportDB=self.airportDB, goclimateDB=self.goclimateDB)
        flight = flight.export()

      ## clustering key k -- attendees are clustered according to k
      if cluster_key is None:
        if mode == 'attendee' :
          key = 'total attendee'
        else:
          key = 'total co2eq'
      elif cluster_key == 'flight_segment_number':
        attendee[ 'flight_segment_number' ]  = len( flight[ 'segment_list' ] )
        key = attendee[ cluster_key ]
      else:
        key = attendee[ cluster_key ]

      ## computing y depending on the co2eq input
      if mode in  [ 'attendee' ] :
        try:
          y [ key ] += 1
        except:
          y [ key ] = 1
      elif mode in [ 'distance', 'flight' ]:
        try:
          y [ key ] += flight[ 'co2eq' ][ co2eq ]
        except:
          y [ key ] = flight[ 'co2eq' ][ co2eq ]

    with gzip.open( data_file, 'wt', encoding="utf8" ) as f:
      f.write( json.dumps( y, indent=2 ) )
    return y

  def reduce_and_transpose( self, column_dict, column_keys ):
    """ format the data for a stack diagram

    Args:
      column_dict (dict): is a dictionary that contains the data associated to the column.
    column_dict = { column_label_1 : column_data_1 , ..., column_label_n : column_data_n }
    column_data contains is a dict { column_key_1 : value_1, ..., column_key_m : value_m }

    each column_data is reduced to the 'column_keys' most important keys, other
    values are concatenated into 'others'

    """
    ## sorting keys according to values
    ## merged_column_dict is not necessary when the dictionaries have the same keys
    ## but this is a more generic way to do this.
    ## https://www.pythoncentral.io/how-to-sort-python-dictionaries-by-key-or-value/
    merged_column_dict = {}
    for column_data in column_dict.values():
      for key in column_data.keys():
        try:
          merged_column_dict[ key ] += column_data[ key ]
        except KeyError:
          merged_column_dict[ key ] = column_data[ key ]

    sorted_keys = sorted(merged_column_dict, key=merged_column_dict.__getitem__, reverse=True)
    if column_keys is None or len( sorted_keys ) <= column_keys :
      plot_sorted_keys = sorted_keys
    else:
      plot_sorted_keys = sorted_keys[ :column_keys ]
      plot_sorted_keys.append('Others')

    for column_label in column_dict.keys():
      column_dict[ column_label ][ 'Others' ] = 0
      column_key_list = list( column_dict[ column_label ].keys() )
      for key in  column_key_list :
        if key not in plot_sorted_keys:
          column_dict[ column_label ][ 'Others' ] += column_dict[ column_label ][ key ]
          del column_dict[ column_label ][ key ]
    ## stack diagrams are line_data stack together
    ## line_data = [ y(X1[0]), y(X2[0]), y(X3[0]) y(X4[0])]
    ## with yi the y value that corresponds to xi
    ## line_list  = [ line_data_0, line_data_1, ..., ]
    line_list = [] ## column_dict data reordered to be plot
    for key in plot_sorted_keys:
      line_data = []
      for column_label in column_dict.keys():
        try:
          line_data.append( column_dict[ column_label ][ key ] )
        except KeyError :
          line_data.append( 0 )
      line_list.append( line_data )
    return line_list, plot_sorted_keys

  def plot_stack( self, line_list:list, stack_label:list=None, column_label:list=None,\
                  column_range:list=None, title:str="", figsize=(6,4), ax=None):
    if column_range is None:
      column_range = range( len( line_list[ 0 ] ) )
    if column_label is None:
      column_label = [ "" for column in  line_list[ 0 ] ]

    if ax is None:
      fig, ax = plt.subplots(figsize=figsize )
    for line_data in line_list:
      if stack_label is None:
        ax.bar( column_range,  line_data )
      else:
        line_index = line_list.index( line_data )
        if line_index == 0:
          bottom = [ 0 for i in column_range ]
        else:
          bottom = [ bottom[ i ] + line_list[ line_index - 1 ][i] for i in column_range ]
        ax.bar( column_range,  line_data, bottom=bottom, label=stack_label[ line_index ] )

    if stack_label is not None and ( isinstance( stack_label, list ) and len( stack_label ) >= 2) :  
      ax.legend( labels=stack_label )
    ax.set_xticks( column_range )
    ax.set_xticklabels( column_label )
    ax.set_title( title, pad=20 )
    return fig, ax

  def plot_co2eq_default_kwargs( self, mode, cluster_key:str, cluster_nbr:int, co2eq:str ) -> list :
    """ returns the default arguments, i.e. when there values is set to None 
      
     This function is only used by plot_co2eq and operates two functions:
     1) coordinates the various arguments that are non independent -- typically co2eq is only valid for mode. 
     2) converts arguments into lists, when multiple diagrams are represented on the same figure. Typically the default call should return all measurements.     
     """

    kwargs = locals()
    del kwargs[ 'self' ]
    if mode is None: # by defult the focus is on co2eq
      mode_list = [ 'flight', 'distance' ]
    elif isinstance( mode, list ):
      mode_list = mode
    elif isinstance( mode, str ):
      mode_list = [ mode ]
    else: 
      raise ValueError( f"unexpected value {mode} for mode" )
    if ( 'attendee' in mode_list and 'flight' in mode_list ) or \
       ( 'attendee' in mode_list and 'distance' in mode_list ) :
      raise ValueError( f"unexpected mode_list: {mode_list}" \
                        f"'attendee' is not compatble with 'flight' or 'distance' mode" )
    mode_str = mode_list[ 0 ]
    for m in mode_list[ 1:]:
      mode_str += f"_{m}"
    kwargs[ 'mode' ] = mode_str

    if co2eq is None:
      if 'flight' in mode_list or 'distance' in mode_list :
        co2eq_list = [ 'myclimate', 'goclimate' ]
        kwargs[ 'co2eq' ] = 'myclimate_goclimate'
      else:
        co2eq_list = [ ]
    elif isinstance( co2eq , list ) :
      co2eq_list = co2eq
    elif isinstance( co2eq , str ) :
      co2eq_list = [ co2eq ]
    if len( co2eq_list ) != 0:
      co2eq_str = co2eq_list[ 0 ]
      for c in co2eq_list[1:]:
        co2eq_str += f"_{c}"
      kwargs[ 'co2eq' ] = co2eq_str
    return mode_list, cluster_key, cluster_nbr, co2eq_list, kwargs

  def plot_title( self, mode, cluster_key ):
    """return title and ylabel associated to the figure """

    if mode in [ 'flight', 'distance' ] :
      title = r'$CO_2$ Equivalent (kg)'
      y_label = r"$CO_2$ Equivalent (kg)"
    elif mode == 'attendee' : 
      title = 'Attendee Number'
      y_label = 'Attendee Number'

    if cluster_key is not None:
      title = title + f" by {cluster_key}"

    return title, y_label

  def plot_co2eq( self, mode=None, cluster_key=None, cluster_nbr=None, co2eq=None) :
    """ plots data 

    mode 'attendee' represents on y a number of person while 'flight' and 'distance' represents CO2. 
    matplotlib enables to plot these values -- with different scales -- on a same figure.
    we do not provide such facilities as the clustering may also have different ordering leading to a figure with two axes and two legends. This woudl provide little gains over using two distinct figures. 

    """
##    https://pylibraries.com/matplotlib/stacked-bar-charts-with-matplotlip-pyplot/
##    https://www.statology.org/matplotlib-stacked-bar-chart/
##
    
    mode_list, cluster_key, cluster_nbr, co2eq_list, kwargs = \
      self.plot_co2eq_default_kwargs( mode, cluster_key, cluster_nbr, co2eq )
    fig_file = join( self.output_dir, self.kwargs_to_file_name( 'co2eq', 'svg', kwargs  ) )
    if isfile( fig_file ) is True:
      return

    ## building the CO2 equivalent data.
    ## in our case, the data is a dictionary for each mode
##    column_dict = {} ## associates for each labels co2eq dict associated to each column
    ## associates for each label the dictionary associated to each column
    ## we use twi different dictionaries as they are using different scales
    ## and need to be plot separately 
    col_dict = {}  
    col_label = []
    for mode in mode_list : #, 'distance' ]:
      if mode in [ 'flight', 'distance' ]:
        for co2eq in co2eq_list:
          y = self.build_co2eq_data( mode=mode, cluster_key=cluster_key, co2eq=co2eq )
          col_dict[ ( mode, co2eq ) ] = y
          col_label.append( f"{mode}\n{co2eq}" ) 
      elif mode in [ 'attendee' ]:
        y = self.build_co2eq_data( mode=mode, cluster_key=cluster_key, co2eq=co2eq )
        col_dict[ mode ] = y
        col_label.append( mode ) 

    line_list, stack_label = self.reduce_and_transpose( col_dict, cluster_nbr )

    title, y_label = self.plot_title( mode_list[0], cluster_key )
    title = title + f" for {self.name} ({len( self.get_attendee_list( ) )} participants)"

    fig, ax = self.plot_stack( line_list, stack_label=stack_label, \
                               column_label=col_label, title=title )
    ax.set_ylabel( y_label ) 

    if len( stack_label ) >= 2:
      plt.legend(bbox_to_anchor=(1, 1.1), loc='upper left')
    plt.tight_layout()
    plt.savefig( fig_file, bbox_inches='tight' )


  def debug_country_info( self, country ):
    """ provides information to understand why a flight cannot be retrived 

    The ability to retrieve a flight depends on:
      1) the correctness and coherence of the iata_city_codes-2015.json, iata_city_airport_map.json and countryinfo
      2) the ability of Amadeus to retrieve an itinerary
      3) Many other reasons...
    This function helps to find why a flight cannot be retrieved. 
    It is limited to the case where only the country has been provided.
    """
##    countryDB = CountryDB()
    ## Step 1: checking the capital returned by country info
    country_info = self.cityDB.countryDB.get_country_info( country )
    print( f"-- step 1: Capital of {country_info.name()} ({country}) is "\
           f"{country_info.capital()}\n"\
           f"           coordinates of the capital are {country_info.capital_latlng()}\n"\
           f"           Please check capital and coordinates are correct.\n"\
           f"           If these information are not correct update the\n"\
           f"           countryinfo data.")
 
    ## Step 2: checking the the matching cities from the IATA city code DB
    iata_capital_list = self.cityDB.get_city( country=country, name=country_info.capital() )
    if len( iata_capital_list ) > 0:
      iata_capital = iata_capital_list[ 0 ]
    else:
      iata_capital = None
    print( f"-- step 2: Searching IATA city by matching the capital\n"\
           f"           name returns: the following city {iata_capital}.\n"\
           f"           A response is only expected when there is a match\n"\
           f"           Between names. It may not happen. In case a match is\n"\
           f"           expected, please consider carefully changing the\n"\
           f"           'capital' in the countryinfo data or the 'name'\n"\
           f"           in the iata_city_codes-2015.json file." )
    ## Step 3: checking the city returned by evaluating the distances
    iata_rep_city = self.cityDB.country_representative_city( country )
    print( f"-- step 3: Searching capital via distance returns {iata_rep_city}\n"\
           f"           There cases where the capital has been overwritten \n"\
           f"           by a larger city (see ISO3166_REPRESENTATIVE_CITY).\n"\
           f"           If the result is surprising  here are possible\n"\
           f"           reasons:\n"\
           f"             a) Coordinate of the expected city in the file\n"\
           f"                iata_city_codes-2015.json are not correct.\n"\
           f"                In this case, upadte the coordinates in \n"\
           f"                iata_city_codes-2015.json\n"\
           f"             b) The distance method only considers cities with airports\n"\
           f"                The city in question is not associated to sufficiently\n"\
           f"                large airports. This could be that the IATA city airport\n"\
           f"                In iata_city_codes-2015.json does not corresponds to the\n"\
           f"                IATA city airport of iata_city_airport_map.\n"\
           f"             c) Amadeux is unable to retrieve flights itinerary from that airport" )
 
    coordinates = country_info.capital_latlng()
    capital = { 'latitude' : coordinates[ 0 ], 'longitude' : coordinates[ 1 ] }
    if iata_capital is not None:
      print( f"Distance from {country_info.capital()} of IATA city found by name matching\n"\
             f"(step 2 ) : {self.cityDB.dist( capital, iata_capital )}\n" )
    print( f"Distance from {country_info.capital()} of IATA city found by shortest distance\n"\
           f"(step 3) : {self.cityDB.dist( capital, iata_rep_city )}\n" )
    ## Step3.5 checking airports associated to the IATA 
    if iata_capital is not None:
      print( f"-- step 4: Does the IATA city (matching capital name) has airports ?\n"\
             f"           {self.cityDB.has_airports( iata_capital ) }\n"\
             f"           False may indicate the IATA code of the city is not\n"\
             f"           recognized by the iata_city_airport_map.\n"\
             f"           True may indicate everything is correct but Amadeux is unable\n"\
             f"           to retrieve an itinerary. If that is the case, this may be adjusted\n"\
             f"           by updating ISO3166_REPRESENTATIVE_CITY.\n")

    print( f"Please check the log in {self.conf[ 'log' ]} "\
           f"for further details. You can do tail -f {self.conf[ 'log' ]}.\n\n"\
           f"If you have not detected something abnormal, it may be that AMADEUS "\
           f"was not able to retrieve the flight at this time and that the next "\
           f" attempt will work. Keep finger crossed and retry." )


  def cluster_dict( self, mode='flight', cluster_key=None, co2eq=None ) -> dict:
    """ returns the dictionary associated to a cluster_key. """
    ## we need to define a common function to retrieve the file ( see build_co2eq_data )
    mode, cluster_key, co2eq = self.sanity_check_args( mode=mode, \
                                                        cluster_key=cluster_key, co2eq=co2eq )
    kwargs = { 'mode' : mode, 'cluster_key' : cluster_key, 'co2eq' : co2eq } 
    data_file = join( self.output_dir, self.kwargs_to_file_name( 'co2eq', 'json.gz', kwargs ) )
    if isfile( data_file ) is True:
      with gzip.open( data_file, 'rt', encoding="utf8" ) as f:
        json_data = json.loads( f.read() )
    else: 
      json_data = self.build_co2eq_data( mode=mode, cluster_key=cluster_key, co2eq=co2eq )
    return json_data

  def select_attendee_list( self, cluster_key=None, cluster_value=None):
    """ returns the list of participants with matching key / values 

    attendee = { 'cluster_key_A' : cluster_value_A, 'cluster_key_B' : cluster_value_B  }


    """
    attendee_list = []
    for attendee in self.attendee_list :
      if cluster_key is None:
        attendee_list.append( attendee ) 
      elif cluster_key in attendee.keys() :
        if cluster_value is None:
          attendee_list.append( attendee ) 
        elif attendee[ cluster_key ] == cluster_value :
          attendee_list.append( attendee ) 
        else: 
          continue
      else: 
        continue
    return attendee_list   

  def md( self, banner="" , toc=True):
    """ generates the md page with all figures"""

    svg_dict = { 'flight' : "", 'distance' : "", 'attendee' : "" }
    mode_list = []
    cluster_key_list = []
    fig_list = []
    for f in listdir( self.output_dir ):
      if 'svg' not  in f:
        continue
      arg_dict = self.file_name_to_kwargs( f )
      if arg_dict[ 'mode' ] is None: ## only mandatory parameter
        continue
      arg_dict[ 'md_text' ] =  f"![]({f})\n"
      if arg_dict[ 'mode' ] not in mode_list:
        mode_list.append( arg_dict[ 'mode' ] )
      elif arg_dict[ 'cluster_key' ]  not in cluster_key_list:
        cluster_key_list.append( arg_dict[ 'cluster_key' ] )
      fig_list.append( arg_dict )
    if None in cluster_key_list:
      cluster_key_list.remove( None )
      cluster_key_list.insert( 0, None )

    toc_md = "Table of Contents\n\n"
    md = ""
    for mode in [ 'flight', 'distance', 'attendee' ]: ## mode
      if mode not in mode_list:
        continue
      if mode in [ 'flight', 'distance' ]:
        title = f"CO2 Estimation in {mode} mode"
      elif mode == 'attendee':
        title = f"Number of Attendees"
      toc_md += f"* [{title}](#{mode})\n"
      md += f"<div id=\"{mode}\"></div>\n## {title}\n\n"
      for cluster_key in cluster_key_list:
        if cluster_key not in cluster_key_list :
          continue
        if cluster_key is None:
          title = f"No Clustering"
        else:
          title = f"Clustering by {cluster_key}"
        toc_md += f"  * [{title}](#{mode}-{cluster_key})\n"
        md += f"<div id=\"{mode}-{cluster_key}\"></div>\n### {title}\n\n"
        for fig in fig_list:
          if fig[ 'mode' ] == mode and fig[ 'cluster_key' ] == cluster_key:
            md += fig[ 'md_text' ]
        md += "\n"

    with open( join( self.output_dir, "index.md"), 'wt', encoding='utf8' ) as f:
      header = f"# {self.name} Data\n\n"
      if toc is False:
        f.write( f"{header}\n{banner}\n{md}" )
      else: 
        f.write( f"{header}\n{banner}\n{toc_md}\n{md}" )




## inherite from Meeting
class MeetingList(Meeting):

  def __init__( self, name, conf={}, meeting_list=None, flightDB=True, airportDB=True, cityDB=True, goclimateDB=True  ):
    """ instantiates a MeetingList object 

    Args: 
      meeting_list (list): a list of meeting dictionary
        { 'name' : meeting_name, 
          'location' : { 
            'country' : meeting_country, 
            'city' : meeting_city,
            'iata' : the meeting iata_airport # especially useful when city has a small airport
           }                                  # and we prefer to select a larger airport      
        }
        meeting_list could also be a list of Meeting objects, but this use is not the prefrred way.
    """ 

    self.name = name
    self.conf = conf
    self.meeting_list = meeting_list
    self.output_dir = join( conf[ 'OUTPUT_DIR' ], self.name )
    if isdir( self.output_dir ) is False:
      os.makedirs( self.output_dir )
    if airportDB is True:
      airportDB = AirportDB()
      self.airportDB = airportDB
    if  cityDB is True:
      cityDB = CityDB( conf, airportDB=self.airportDB )
    self.cityDB = cityDB
    if goclimateDB is True:
      goclimateDB=GoClimateDB( conf )
    self.goclimateDB = goclimateDB
    if flightDB is True:
      flightDB = FlightDB( conf, cityDB=self.cityDB, airportDB=self.airportDB, \
                           goclimateDB=self.goclimateDB)
    self.flightDB = flightDB
    self.logger = logger( conf, __name__ )     

  def get_meeting( self, meeting ):
    """ return a meetin object from the content of meeting_list

    Args:
      meeting (dict) : a meeting_dict object (see meeting_list). Eventually a Meeting object.
    """
    if isinstance( meeting, Meeting ):
      return meeting
    else:
      name = meeting[ 'name' ]
      location = meeting[ 'location' ]
      if 'attendee_list' in meeting.keys():
        attendee_list = meeting[ 'attendee_list' ]
      else: 
        attendee_list = None
      return Meeting( name, location, attendee_list=attendee_list, \
                      conf=self.conf, flightDB=self.flightDB, \
                      airportDB=self.airportDB, cityDB=self.cityDB, \
                      goclimateDB=self.goclimateDB )

  def plot_co2eq( self, mode=None, cluster_key=None, cluster_nbr=None, \
                  co2eq=None, figsize=(10,4), column_label=None, \
                  xticks_rotation='vertical', adjust_bottom=0.4, ) :

    kwargs = { 'mode' : mode, 'cluster_key' : cluster_key, \
               'cluster_nbr' : cluster_nbr, 'co2eq' : co2eq }

    ## checking mode values
    if mode not in [ 'flight', 'distance', 'attendee' ]:
      raise ValueError( f"Unacceptable value {mode} for mode. Accepted values are 'attendee', 'flight' and 'distance'" )
    ## in mode attendee co2eq is set to None
    if mode == 'attendee' and co2eq != None:
      raise ValueError( f"Unacceptable value {co2eq} for co2eq with mode {mode}. Must be None in mode 'attendee'" )
    if co2eq not in [ 'goclimate', 'myclimate', None ]:
      raise ValueError( f"Unacceptable value {co2eq} for co2eq. Accepted values are 'goclimate', 'myclimate', None." )
    if kwargs[ 'cluster_key' ] is not None and not isinstance( kwargs[ 'cluster_key' ], str ):
      raise ValueError( f"Unacceptable arguments. cluster_key must be string or None. {kwargs}" )
    if kwargs[ 'cluster_nbr' ] is not None and not isinstance( kwargs[ 'cluster_nbr' ], int ):
      raise ValueError( f"Unacceptable arguments. cluster_nbr must be int or None. {kwargs}" )

    fig_file = join( self.output_dir, self.kwargs_to_file_name( 'co2eq', 'svg', kwargs  ) )
    if isfile( fig_file ) is True:
      return

    column_dict = {}
    for meeting in self.meeting_list:
      meeting = self.get_meeting( meeting )
      column_dict[ meeting.name ] = meeting.build_co2eq_data( mode=mode, cluster_key=cluster_key,\
                                                              co2eq=co2eq )
    line_list, stack_label = self.reduce_and_transpose( column_dict, cluster_nbr )
    
    title, y_label = self.plot_title( mode, cluster_key )
    if mode in [ 'distance', 'flight' ]:
      title = title + f" measured in {mode} mode with {co2eq}"
    
    fig, ax = self.plot_stack( line_list, stack_label=stack_label, \
                               column_label=column_label, title=title, \
                               figsize=figsize )
    ax.set_ylabel( y_label ) 
    plt.xticks(rotation = xticks_rotation)
    fig.subplots_adjust(bottom=adjust_bottom)
    if len( stack_label ) >= 2:
      plt.legend(bbox_to_anchor=(1, 1.1), loc='upper left')
    plt.tight_layout()
    plt.savefig( fig_file, bbox_inches='tight' )
#    plt.show()

  def plot_all( self ):
    for meeting in self.meeting_list:
      meeting = self.get_meeting( meeting )
      self.logger.info( f"{meeting.name}: Processing plot_co2eq" )
      meeting.plot_co2eq( )
      ## evaluation the cluster_keys by reading those of the first 
      ## attendee
      cluster_key_list = list( meeting.get_attendee_list()[0].keys() )
      cluster_key_list.append( 'flight_segment_number' )
      cluster_key_list.append( None )
      for cluster_key in  cluster_key_list :
        meeting.plot_co2eq( mode=None, cluster_key=cluster_key, cluster_nbr=15)
##      meeting.plot_co2eq( mode=None, cluster_key='organization', cluster_nbr=15)
##      meeting.plot_co2eq( mode=None, cluster_key='presence' )
##      meeting.plot_co2eq( mode=None, cluster_key='country', cluster_nbr=15 )
##      meeting.plot_co2eq( mode=None, cluster_key='flight_segment_number',  cluster_nbr=15 )
    self.logger.info( f"\nprocessing {self.name }\n" )
    if isinstance( self.meeting_list[0], dict ):
      column_label = []
      for meeting in self.meeting_list:
        label = ""
        try :
          label = f"{meeting[ 'location' ][ 'city' ]} "
        except KeyError:
          pass
        label = f"{label}{meeting[ 'name' ]}"
        column_label.append( label )
    else:
      column_label = None
    figsize=(10,4) ## width, heigh (inch) of the figure
    adjust_bottom = 0.4 ## make sure we have enough space to read the xticks
    xticks_rotation='vertical' ## xticks orientation
    plot_kwargs = { 'cluster_nbr' : 15, 'figsize' : figsize, 'column_label' : column_label, \
                    'adjust_bottom' : adjust_bottom, 'xticks_rotation' : xticks_rotation }
    for mode in [ 'flight' ]:
      for co2eq in [ 'myclimate', 'goclimate' ]:
         for cluster_key in  cluster_key_list :
           self.plot_co2eq( mode=mode, cluster_key=cluster_key, co2eq=co2eq, **plot_kwargs )
    for mode in [ 'attendee' ]:
       for cluster_key in  cluster_key_list :
         self.plot_co2eq( mode=mode, cluster_key=cluster_key, **plot_kwargs )
##        self.plot_co2eq( cluster_key=None, co2eq=co2eq, **plot_kwargs )
##        self.plot_co2eq( cluster_key='organization', cluster_nbr=15, co2eq=co2eq, **plot_kwargs )
##        self.plot_co2eq( cluster_key='presence', co2eq=co2eq, **plot_kwargs )
##        self.plot_co2eq( cluster_key='country', cluster_nbr=15, co2eq=co2eq, **plot_kwargs )
##        self.plot_co2eq( cluster_key='flight_segment_number', cluster_nbr=15, co2eq=co2eq, **plot_kwargs )



  def banner_cell( self, meeting_list_url:str, meeting_name:str ) -> str :
    """ returns the cell of the banner """

    if meeting_list_url[ -1 ] != '/' :
      meeting_list_url += '/'
    meeting_url = meeting_list_url + meeting_name
##   return f"[{meeting_name}]({meeting_url})"
    return f"<a href='{meeting_url}'>{meeting_name}</a>"

## in meetingList
## meeting_list is in the object
  def banner_md( self, meeting_list_url:str, col_nbr:int =10, home_url:str =None ) -> str :
    """ returns the banner for a md page - though html is used. 

    The banner contains a link to the main meeting list page as well as 
    every individual meetings. 
    Optionally, another home url may be added, for example when the meeting 
    list pages are part of another web site.  

    Args:
      meeting_list_url (str) : the url associated to the meeting list. The pages
        urls are derived as follows: the meetinglist name is appened to 
        form the meeting list url. Each meeting names are appended to the url. 
        For the IETF it is like if we had the following subdirectories: 
        IETF and IETF99, IETF100, ...
      col_nbr (int) : the number of meeting names that can be shown on the web page. 
        When multiple meetings are considered, these may b eprinted on multiple lines.
        To be aligned these are placed in a table of col_nbr columns.
    """
    if home_url != None:
      add_cell = 2
    else :
      add_cell = 1
    
    row_nbr = ceil( ( len( self.meeting_list ) + add_cell ) / col_nbr )
    ## meeting list including meetingList
#    meeting_list = [ self.banner_cell( meeting_list_url, m ) for m in self.meeting_list ]
    meeting_list = []
    for m in self.meeting_list :
      meeting_obj = self.get_meeting( m )
      meeting_list.append( self.banner_cell( meeting_list_url, meeting_obj.name ) )
    meeting_list.insert( 0, self.banner_cell( meeting_list_url, self.name ) )
    if home_url != None:
      meeting_list.insert( 0, f"<a href='{home_url}' style='font-size: 30px; text-decoration: none' >⌂</a>" )

    ## note that modifying the indentiation of the html block makes it 
    ## mis-interpreted by jekyl and so produces a bad html rendering. 
    begin_table = \
    """
  <html>
  <style>
  table, th, td {
    border: 0px none;
    padding: 0px;
  }
  </style>
  <body>
    <table style="width:100%">
    """
    end_table = \
    """
    </table>
  </body>
  </html>
    """
    banner = begin_table
    for row in range( row_nbr ):
      banner += "      <tr>\n"
      for col in range( col_nbr ):
        try:
          banner += f"<td>{meeting_list[ row * col_nbr + col ]}</td>\n"
        except IndexError:
          banner += f"<td> </td>\n"
      banner += "      </tr>\n"
 
    banner += end_table
    return banner

  def www_md( self, meeting_list_url, col_nbr=10, toc=True, home_url=None):
    """ generates all md pages for the IETF meetings """
    banner = self.banner_md( meeting_list_url, col_nbr=col_nbr, home_url=home_url )
    self.md( banner, toc=toc )
    for meeting_name in self.meeting_list:
      meeting = self.get_meeting( meeting_name )
      meeting.md( banner, toc=toc ) 

def get_flight( conf, origin, destination ):
  """ return a flight from origin to destination

  The function tries with default values provided by FlightDB and in case no
  offer is provided performs another lookup with different dates.
  In our cases, the dates are 5 days latter.
  """
  cityDB = CityDB( conf, airportDB=airportDB )
  airportDB = AirportDB()
  goclimateDB = GoClimateDB( conf )
  flightDB = FlightDB( conf, cityDB=cityDB, airportDB=airportDB, goclimateDB=goclimateDB)
  try:
    flight = flightDB.select_flight( origin, destination )
  except ( ValueError ) :
    ## retry with other dates - in this case 5 days later
    departure_date = flightDB.departure_date
    return_date = flightDB.return_date
    alt_departure = datetime.strptime( departure_date + 'T16:41:24+0200', "%Y-%m-%dT%H:%M:%S%z")
    alt_departure = ( alt_departure + timedelta( days=5 ) ).isoformat()
    alt_return = datetime.strptime( return_date + 'T16:41:24+0200', "%Y-%m-%dT%H:%M:%S%z")
    alt_return = ( alt_return + timedelta( days=5 ) ).isoformat()
    flight = flightDB.select_flight( origin, destination, departure_date=alt_departure, return_date=alt_return )
  return flight

