# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['pyinterboleto',
 'pyinterboleto.common',
 'pyinterboleto.consulta',
 'pyinterboleto.emissao',
 'pyinterboleto.utils']

package_data = \
{'': ['*']}

install_requires = \
['Unidecode>=1.2.0,<2.0.0',
 'dacite>=1.6.0,<2.0.0',
 'requests>=2.26.0,<3.0.0',
 'tabulate>=0.8.9,<0.9.0']

setup_kwargs = {
    'name': 'pyinterboleto',
    'version': '1.0.3',
    'description': 'Biblioteca para facilitar o manuseio de boletos de contas PJ no Banco Inter.',
    'long_description': '# PyInterBoleto\n\nBiblioteca para facilitar o manuseio de boletos de contas PJ no Banco Inter.\n\n[![PyPI version](https://badge.fury.io/py/pyinterboleto.svg)](https://badge.fury.io/py/pyinterboleto)\n[![codecov](https://codecov.io/gh/feslima/pyinterboleto/branch/main/graph/badge.svg?token=T2SJ0P8KPG)](https://codecov.io/gh/feslima/pyinterboleto)\n\n***\n\n- [PyInterBoleto](#pyinterboleto)\n- [Instalação](#instalação)\n- [Documentação](#documentação)\n- [Utilização básica](#utilização-básica)\n  - [Configuração de autenticação](#configuração-de-autenticação)\n  - [Emissão de boleto](#emissão-de-boleto)\n  - [Consultas](#consultas)\n    - [Consulta detalhada de boletos individuais](#consulta-detalhada-de-boletos-individuais)\n    - [Consulta de coleção de boletos](#consulta-de-coleção-de-boletos)\n    - [Resgate de PDFs de boletos individuais](#resgate-de-pdfs-de-boletos-individuais)\n  - [Baixa de boleto](#baixa-de-boleto)\n- [Testagem](#testagem)\n\n## Instalação\n\nBasta executar o comando via pip (ou poetry):\n\n```shell\npip install pyinterboleto\n# ou via poetry\npoetry add pyinterboleto\n```\n\n## Documentação\n\nA maioria das classes, métodos, funções e estruturas de dados já contém uma\ndocumentação habilitada para uso em IDEs no estilo\n[numpy docstring](https://numpydoc.readthedocs.io/en/latest/format.html).\n\nFoi optado por não fazer uma documentação exclusiva (no estilo readthedocs)\ndevido a ser uma biblioteca relativamente pequena.\n\nSendo assim, o pacote está organizado em três submódulos principais:\n**emissão** (os dados necessários na emissão são organizados em estruturas\nmenores. São elas dados de: [emissão](src/pyinterboleto/emissao/emissor.py),\n[pagador](src/pyinterboleto/emissao/pagador.py),\n[desconto](src/pyinterboleto/emissao/desconto.py),\n[multa](src/pyinterboleto/emissao/multa.py),\n[mora](src/pyinterboleto/emissao/mora.py) e\n[mensagem](src/pyinterboleto/emissao/mensagem.py)), **consulta**\n([detalhada](src/pyinterboleto/consulta/detalhado.py),\n[coleção](src/pyinterboleto/consulta/lista.py) e\n[PDF](src/pyinterboleto/consulta/pdf.py)) e\n[**cancelamento**](src/pyinterboleto/baixa.py) de boletos.\n\nEm cada um desses links se encontram as funções e objetos com suas\nrespectivas documentações, caso seja preciso mais informações.\n\n## Utilização básica\n\nA classe principal que tem todas as funcionalidades requeridas para a API se\nchama [**`Boleto`**](src/pyinterboleto/boleto.py). Através dela que todas as\noperações de emissão, consulta e baixa feitas.\n\nNo entanto, como consta na documentação do Banco Inter, para se ter acesso a\nAPI é preciso emitir a chave e o certificado de acesso desta. Antes de\nutilizar este pacote, certifique-se que você já possui estes arquivos.\n\nFeito isto, alguns exemplos de manuseio são mostrados nas seções à seguir.\n***\n\n### Configuração de autenticação\n\nAntes de fazer qualquer requisição à API do Inter é preciso antes definir o\n[objeto de configuração](src/pyinterboleto/utils/requests.py) para\nautenticação e autorização:\n\n```python\n>>> from pathlib import Path\n>>> from datetime import date, timedelta\n>>> from prettyprinter import pprint, install_extras\n>>> from pyinterboleto import RequestConfigs\n>>>\n>>> install_extras()\n>>>\n>>> # definição da configuração de autenticação\n>>> direc = Path(\'caminho/para/pasta/com/certificados\')\n>>> cert = direc / \'Inter API_Certificado.crt\'\n>>> key = direc / \'Inter API_Chave.key\'\n>>> # client_id e client_secret são obtidos de acordo com a documentação do Inter\n>>> client_id = \'valor-do-id-uuid\'\n>>> client_secret = \'valor-do-secret-uuid\'\n>>> scopes = (ScopeEnum.BOLETO_COBRANCA_WRITE,)\n>>> configs = RequestConfigs(client_id=client_id, client_secret=client_secret, scopes=scopes, certificate=cert, key=key)\n```\n\n### Emissão de boleto\n\n**Os dados a seguir são fictícios. Não os utilize do jeito que estão!**\n\n```python\n>>> from pyinterboleto import Boleto, Emissao, Pagador, Beneficiario\n>>> boleto = Boleto(configs) # configs vem da seção configuração\n>>>\n>>> pagador = Pagador(\n...     cpfCnpj="12.345.678/0001-12",\n...     tipoPessoa=TipoPessoa.JURIDICA,\n...     nome="Alguma Empresa LTDA",\n...     endereco="Qulaquer um",\n...     cidade="Também do Brasil",\n...     uf="SP",\n...     cep="12345-678",\n... )\n>>> beneficiario = Beneficiario(\n...     cpfCnpj="123.456.789-01",\n...     tipoPessoa=TipoPessoa.FISICA,\n...     nome="Algum Nome de Pessoa",\n...     endereco="Algum lugar",\n...     bairro="Qualquer",\n...     cidade="Do Brasil",\n...     uf="SP",\n...     cep="12345-678",\n... )\n>>> dados = Emissao(\n...     pagador=pagador,\n...     beneficiario=beneficiario,\n...     seuNumero="000001",\n...     valorNominal=10.01,\n...     dataVencimento="2023-01-01",\n...     numDiasAgenda=25,\n... )\n>>> result = boleto.emitir(emissao)\n>>> print(result)\n{\'seuNumero\': \'00001\', \'nossoNumero\': \'00123456789\',\n \'codigoBarras\': \'00000000000000000000000000000000000000000000\',\n \'linhaDigitavel\': \'00000000000000000000000000000000000000000000000\'}\n>>>\n```\n\n### Consultas\n\nHá três categorias de consultas disponíveis: detalhamento individual de\nboletos, coleção de boletos e resgate de arquivos .pdf.\n\n#### Consulta detalhada de boletos individuais\n\nÉ preciso saber o número de identificação do título antes de fazer esta\nrequisição. Este número pode ser obtido quando a emissão do título é\nbem sucedida (chave `nossoNumero` do dicionário de resposta na seção anterior)\nou quando se faz a filtragem de coleções de boletos.\n\n```python\n>>> boleto = Boleto(configs)\n>>> num_boleto = \'00123456789\' # numero de identificação do título pelo Inter\n>>> detail = boleto.consulta_detalhada(num_boleto)\n>>> pprint(detail)\n```\n\n#### Consulta de coleção de boletos\n\nAs datas de início e final da filtragem são obrigatórias,\n[há outras definições de elementos de filtragem opcionais](src/pyinterboleto/consulta/lista.py).\n\n```python\n>>> from datetime import date, timedelta\n>>> boleto = Boleto(configs)\n>>> inicial = date.today() - timedelta(days=30)\n>>> final = date.today()\n>>> lista = boleto.consulta_lista(inicial, final)\n>>> pprint(lista)\n```\n\n#### Resgate de PDFs de boletos individuais\n\nAssim como na consulta detalhada individual, é preciso saber o número de\nidentificação do título antes de fazer a requisição. Há dois modos de\ncomo o PDF é armazendo: em memória ou salvo diretamento em um arquivo especificado.\n\n```python\n>>> from pathlib import Path\n>>> boleto = Boleto(configs)\n>>> num_boleto = \'00123456789\'\n>>> # Armazenado em um buffer de bytes na memória\n>>> pdf = boleto.consulta_pdf(num_boleto)\n>>>\n>>> # salva em um arquivo chamado \'boleto.pdf\' no diretório atual\n>>> filename = Path().resolve() / \'boleto.pdf\'\n>>> boleto.consulta_pdf(num_boleto, filename)\n```\n\n***\n\n### Cancelamento de boleto\n\nTambém é preciso saber o número de identificação do título. Os tipos de\nbaixas são definidos no enum [`MotivoCancelamentoEnum`](src/pyinterboleto/baixa.py).\n\n```python\n>>> from pyinterboleto import MotivoCancelamentoEnum\n>>> boleto = Boleto(configs)\n>>> num_boleto = \'00123456789\'\n>>> boleto.cancelar_boleto(num_boleto, MotivoCancelamentoEnum.A_PEDIDO_DO_CLIENTE)\n```\n\n***\n\n### Testes\n\nComo a API do Inter não possui ambiente de sandboxing, optei por\n"mockar" todas as repostas de acordo com a documentação oficial\ndeles. Isso não isenta a biblioteca de bugs, tendo em vista que a\nprópria documentação deles apresenta algumas inconsistências.\n\nPara realizar os testes localmente, clone o repositório e instale as\ndependências de desenvolvimento:\n\n```shell\n# pode usar o gerenciador que quiser (e.g. poetry, conda, etc.)\npip install -r requirements-dev.txt\n\n# poetry install --no-root (para o caso de usar poetry).\n```\n\nPara rodar os tests:\n\n```shell\npytest\n\n# ou no caso de poetry:\npoetry run poe cov_local\n```\n',
    'author': 'Felipe Souza Lima',
    'author_email': 'feslima93@gmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/feslima/pyinterboleto',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
