import shutil
from pathlib import Path

from sphinxcontrib.yamcs import autogen, lexers
from sphinxcontrib.yamcs.javadoc import javadoc_role
from sphinxcontrib.yamcs.options import OptionsDirective
from sphinxcontrib.yamcs.proto import (
    ProtoDirective,
    RouteDirective,
    RPCDirective,
    ServiceDirective,
    WebSocketDirective,
)
from sphinxcontrib.yamcs.protoparse import ProtoParser


def config_inited(app, config):
    """
    Autogenerate GPB documents.
    """
    if config.yamcs_api_protobin:
        with open(config.yamcs_api_protobin, "rb") as f:
            data = f.read()

        destdir = Path(app.srcdir, app.config.yamcs_api_destdir)

        # Careful attempt at deleting past autogenerated files.
        # We must account for the use case where non-autogenerated
        # files are present in the same directory, therefore this is
        # not a simple remove-all of destdir.
        autogenfile = Path(destdir, ".autogen")
        if autogenfile.exists():
            with autogenfile.open("r") as f:
                for line in f.readlines():
                    if line.strip():
                        to_be_deleted = Path(destdir, line.strip())
                        if to_be_deleted.exists():
                            if to_be_deleted.is_file():
                                to_be_deleted.unlink()
                            elif to_be_deleted.is_dir():
                                shutil.rmtree(to_be_deleted)
            autogenfile.unlink()
        destdir.mkdir(exist_ok=True)

        parser = ProtoParser(data)
        title = app.config.yamcs_api_title
        additional_docs = app.config.yamcs_api_additional_docs

        autogen.generate(parser, destdir, title, additional_docs)


def env_before_read_docs(app, env, docnames):
    """
    Cache a ProtoParser for use by any directives.
    """
    if app.config.yamcs_api_protobin:
        with open(app.config.yamcs_api_protobin, "rb") as f:
            data = f.read()

        env.protoparser = ProtoParser(data)


def setup(app):
    app.add_config_value("yamcs_api_protobin", None, "env")
    app.add_config_value("yamcs_api_destdir", "http-api", "env")
    app.add_config_value("yamcs_api_title", "HTTP API", "env")
    app.add_config_value("yamcs_api_additional_docs", [], "env")

    app.add_directive("options", OptionsDirective)
    app.add_directive("proto", ProtoDirective)
    app.add_directive("rpc", RPCDirective)
    app.add_directive("route", RouteDirective)
    app.add_directive("service", ServiceDirective)
    app.add_directive("websocket", WebSocketDirective)

    app.add_lexer("uritemplate", lexers.URITemplateLexer)
    app.add_lexer("urivariable", lexers.URIVariableLexer)

    app.add_role("javadoc", javadoc_role)

    app.connect("config-inited", config_inited)
    app.connect("env-before-read-docs", env_before_read_docs)
