"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TableClass = exports.StreamViewType = exports.ProjectionType = exports.BillingMode = exports.AttributeType = exports.Table = exports.TableEncryption = exports.Operation = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const dynamodb_canned_metrics_generated_1 = require("./dynamodb-canned-metrics.generated");
const dynamodb_generated_1 = require("./dynamodb.generated");
const perms = require("./perms");
const replica_provider_1 = require("./replica-provider");
const scalable_table_attribute_1 = require("./scalable-table-attribute");
// keep this import separate from other imports to reduce chance for merge conflicts with v2-main
// eslint-disable-next-line no-duplicate-imports, import/order
const core_2 = require("@aws-cdk/core");
const HASH_KEY_TYPE = 'HASH';
const RANGE_KEY_TYPE = 'RANGE';
// https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
const MAX_LOCAL_SECONDARY_INDEX_COUNT = 5;
/**
 * Supported DynamoDB table operations.
 */
var Operation;
(function (Operation) {
    /** GetItem */
    Operation["GET_ITEM"] = "GetItem";
    /** BatchGetItem */
    Operation["BATCH_GET_ITEM"] = "BatchGetItem";
    /** Scan */
    Operation["SCAN"] = "Scan";
    /** Query */
    Operation["QUERY"] = "Query";
    /** GetRecords */
    Operation["GET_RECORDS"] = "GetRecords";
    /** PutItem */
    Operation["PUT_ITEM"] = "PutItem";
    /** DeleteItem */
    Operation["DELETE_ITEM"] = "DeleteItem";
    /** UpdateItem */
    Operation["UPDATE_ITEM"] = "UpdateItem";
    /** BatchWriteItem */
    Operation["BATCH_WRITE_ITEM"] = "BatchWriteItem";
    /** TransactWriteItems */
    Operation["TRANSACT_WRITE_ITEMS"] = "TransactWriteItems";
    /** TransactGetItems */
    Operation["TRANSACT_GET_ITEMS"] = "TransactGetItems";
    /** ExecuteTransaction */
    Operation["EXECUTE_TRANSACTION"] = "ExecuteTransaction";
    /** BatchExecuteStatement */
    Operation["BATCH_EXECUTE_STATEMENT"] = "BatchExecuteStatement";
    /** ExecuteStatement */
    Operation["EXECUTE_STATEMENT"] = "ExecuteStatement";
})(Operation = exports.Operation || (exports.Operation = {}));
/**
 * What kind of server-side encryption to apply to this table.
 */
var TableEncryption;
(function (TableEncryption) {
    /**
     * Server-side KMS encryption with a master key owned by AWS.
     */
    TableEncryption["DEFAULT"] = "AWS_OWNED";
    /**
     * Server-side KMS encryption with a customer master key managed by customer.
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     *
     * > **NOTE**: if `encryptionKey` is not specified and the `Table` construct creates
     * > a KMS key for you, the key will be created with default permissions. If you are using
     * > CDKv2, these permissions will be sufficient to enable the key for use with DynamoDB tables.
     * > If you are using CDKv1, make sure the feature flag `@aws-cdk/aws-kms:defaultKeyPolicies`
     * > is set to `true` in your `cdk.json`.
     */
    TableEncryption["CUSTOMER_MANAGED"] = "CUSTOMER_MANAGED";
    /**
     * Server-side KMS encryption with a master key managed by AWS.
     */
    TableEncryption["AWS_MANAGED"] = "AWS_MANAGED";
})(TableEncryption = exports.TableEncryption || (exports.TableEncryption = {}));
class TableBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        this.regionalArns = new Array();
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:PutItem", "dynamodb:GetItem", ...)
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [
                this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                })),
            ],
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement associated with this table's stream to an
     * IAM principal's policy.
     *
     * If `encryptionKey` is present, appropriate grants to the key needs to be added
     * separately using the `table.encryptionKey.grant*` methods.
     *
     * @param grantee The principal (no-op if undefined)
     * @param actions The set of actions to allow (i.e. "dynamodb:DescribeStream", "dynamodb:GetRecords", ...)
     */
    grantStream(grantee, ...actions) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: [this.tableStreamArn],
            scope: this,
        });
    }
    /**
     * Permits an IAM principal all data read operations from this table:
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan, DescribeTable.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantReadData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.DESCRIBE_TABLE);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, tableActions });
    }
    /**
     * Permits an IAM Principal to list streams attached to current dynamodb table.
     *
     * @param grantee The principal (no-op if undefined)
     */
    grantTableListStreams(grantee) {
        if (!this.tableStreamArn) {
            throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Permits an IAM principal all stream data read operations for this
     * table's stream:
     * DescribeStream, GetRecords, GetShardIterator, ListStreams.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantStreamRead(grantee) {
        this.grantTableListStreams(grantee);
        return this.combinedGrant(grantee, { keyActions: perms.KEY_READ_ACTIONS, streamActions: perms.READ_STREAM_DATA_ACTIONS });
    }
    /**
     * Permits an IAM principal all data write operations to this table:
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem, DescribeTable.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantWriteData(grantee) {
        const tableActions = perms.WRITE_DATA_ACTIONS.concat(perms.DESCRIBE_TABLE);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits an IAM principal to all data read/write operations to this table.
     * BatchGetItem, GetRecords, GetShardIterator, Query, GetItem, Scan,
     * BatchWriteItem, PutItem, UpdateItem, DeleteItem, DescribeTable
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantReadWriteData(grantee) {
        const tableActions = perms.READ_DATA_ACTIONS.concat(perms.WRITE_DATA_ACTIONS).concat(perms.DESCRIBE_TABLE);
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions });
    }
    /**
     * Permits all DynamoDB operations ("dynamodb:*") to an IAM principal.
     *
     * Appropriate grants will also be added to the customer-managed KMS key
     * if one was configured.
     *
     * @param grantee The principal to grant access to
     */
    grantFullAccess(grantee) {
        const keyActions = perms.KEY_READ_ACTIONS.concat(perms.KEY_WRITE_ACTIONS);
        return this.combinedGrant(grantee, { keyActions, tableActions: ['dynamodb:*'] });
    }
    /**
     * Return the given named metric for this Table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/DynamoDB',
            metricName,
            dimensionsMap: {
                TableName: this.tableName,
            },
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the consumed read capacity units this table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricConsumedReadCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedReadCapacityUnitsSum, props);
    }
    /**
     * Metric for the consumed write capacity units this table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricConsumedWriteCapacityUnits(props) {
        return this.cannedMetric(dynamodb_canned_metrics_generated_1.DynamoDBMetrics.consumedWriteCapacityUnitsSum, props);
    }
    /**
     * Metric for the system errors this table
     *
     * @deprecated use `metricSystemErrorsForOperations`.
     */
    metricSystemErrors(props) {
        var _b, _c, _d, _e;
        if (!((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) && !((_c = props === null || props === void 0 ? void 0 : props.dimensionsMap) === null || _c === void 0 ? void 0 : _c.Operation)) {
            // 'Operation' must be passed because its an operational metric.
            throw new Error("'Operation' dimension must be passed for the 'SystemErrors' metric.");
        }
        const dimensionsMap = {
            TableName: this.tableName,
            ...(_d = props === null || props === void 0 ? void 0 : props.dimensions) !== null && _d !== void 0 ? _d : {},
            ...(_e = props === null || props === void 0 ? void 0 : props.dimensionsMap) !== null && _e !== void 0 ? _e : {},
        };
        return this.metric('SystemErrors', { statistic: 'sum', ...props, dimensionsMap });
    }
    /**
     * Metric for the user errors. Note that this metric reports user errors across all
     * the tables in the account and region the table resides in.
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricUserErrors(props) {
        if (props === null || props === void 0 ? void 0 : props.dimensions) {
            throw new Error("'dimensions' is not supported for the 'UserErrors' metric");
        }
        // overriding 'dimensions' here because this metric is an account metric.
        // see 'UserErrors' in https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/metrics-dimensions.html
        return this.metric('UserErrors', { statistic: 'sum', ...props, dimensionsMap: {} });
    }
    /**
     * Metric for the conditional check failed requests this table
     *
     * By default, the metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricConditionalCheckFailedRequests(props) {
        return this.metric('ConditionalCheckFailedRequests', { statistic: 'sum', ...props });
    }
    /**
     * How many requests are throttled on this table
     *
     * Default: sum over 5 minutes
     *
     * @deprecated Do not use this function. It returns an invalid metric. Use `metricThrottledRequestsForOperation` instead.
     */
    metricThrottledRequests(props) {
        return this.metric('ThrottledRequests', { statistic: 'sum', ...props });
    }
    /**
     * How many requests are throttled on this table, for the given operation
     *
     * Default: sum over 5 minutes
     */
    metricThrottledRequestsForOperation(operation, props) {
        return new cloudwatch.Metric({
            ...dynamodb_canned_metrics_generated_1.DynamoDBMetrics.throttledRequestsSum({ Operation: operation, TableName: this.tableName }),
            ...props,
        }).attachTo(this);
    }
    /**
     * Metric for the successful request latency this table.
     *
     * By default, the metric will be calculated as an average over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricSuccessfulRequestLatency(props) {
        var _b, _c, _d, _e, _f;
        if (!((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) && !((_c = props === null || props === void 0 ? void 0 : props.dimensionsMap) === null || _c === void 0 ? void 0 : _c.Operation)) {
            throw new Error("'Operation' dimension must be passed for the 'SuccessfulRequestLatency' metric.");
        }
        const dimensionsMap = {
            TableName: this.tableName,
            Operation: (_e = (_d = props.dimensionsMap) === null || _d === void 0 ? void 0 : _d.Operation) !== null && _e !== void 0 ? _e : (_f = props.dimensions) === null || _f === void 0 ? void 0 : _f.Operation,
        };
        return new cloudwatch.Metric({
            ...dynamodb_canned_metrics_generated_1.DynamoDBMetrics.successfulRequestLatencyAverage(dimensionsMap),
            ...props,
            dimensionsMap,
        }).attachTo(this);
    }
    /**
     * Metric for the system errors this table.
     *
     * This will sum errors across all possible operations.
     * Note that by default, each individual metric will be calculated as a sum over a period of 5 minutes.
     * You can customize this by using the `statistic` and `period` properties.
     */
    metricSystemErrorsForOperations(props) {
        var _b, _c;
        if ((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) {
            throw new Error("The Operation dimension is not supported. Use the 'operations' property.");
        }
        const operations = (_c = props === null || props === void 0 ? void 0 : props.operations) !== null && _c !== void 0 ? _c : Object.values(Operation);
        const values = this.createMetricsForOperations('SystemErrors', operations, { statistic: 'sum', ...props });
        const sum = new cloudwatch.MathExpression({
            expression: `${Object.keys(values).join(' + ')}`,
            usingMetrics: { ...values },
            color: props === null || props === void 0 ? void 0 : props.color,
            label: 'Sum of errors across all operations',
            period: props === null || props === void 0 ? void 0 : props.period,
        });
        return sum;
    }
    /**
     * Create a map of metrics that can be used in a math expression.
     *
     * Using the return value of this function as the `usingMetrics` property in `cloudwatch.MathExpression` allows you to
     * use the keys of this map as metric names inside you expression.
     *
     * @param metricName The metric name.
     * @param operations The list of operations to create metrics for.
     * @param props Properties for the individual metrics.
     * @param metricNameMapper Mapper function to allow controlling the individual metric name per operation.
     */
    createMetricsForOperations(metricName, operations, props, metricNameMapper) {
        var _b;
        const metrics = {};
        const mapper = metricNameMapper !== null && metricNameMapper !== void 0 ? metricNameMapper : (op => op.toLowerCase());
        if ((_b = props === null || props === void 0 ? void 0 : props.dimensions) === null || _b === void 0 ? void 0 : _b.Operation) {
            throw new Error('Invalid properties. Operation dimension is not supported when calculating operational metrics');
        }
        for (const operation of operations) {
            const metric = this.metric(metricName, {
                ...props,
                dimensionsMap: {
                    TableName: this.tableName,
                    Operation: operation,
                    ...props === null || props === void 0 ? void 0 : props.dimensions,
                },
            });
            const operationMetricName = mapper(operation);
            const firstChar = operationMetricName.charAt(0);
            if (firstChar === firstChar.toUpperCase()) {
                // https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax
                throw new Error(`Mapper generated an illegal operation metric name: ${operationMetricName}. Must start with a lowercase letter`);
            }
            metrics[operationMetricName] = metric;
        }
        return metrics;
    }
    /**
     * Adds an IAM policy statement associated with this table to an IAM
     * principal's policy.
     * @param grantee The principal (no-op if undefined)
     * @param opts Options for keyActions, tableActions and streamActions
     */
    combinedGrant(grantee, opts) {
        if (opts.tableActions) {
            const resources = [this.tableArn,
                core_1.Lazy.string({ produce: () => this.hasIndex ? `${this.tableArn}/index/*` : core_1.Aws.NO_VALUE }),
                ...this.regionalArns,
                ...this.regionalArns.map(arn => core_1.Lazy.string({
                    produce: () => this.hasIndex ? `${arn}/index/*` : core_1.Aws.NO_VALUE,
                }))];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.tableActions,
                resourceArns: resources,
                scope: this,
            });
            if (this.encryptionKey && opts.keyActions) {
                this.encryptionKey.grant(grantee, ...opts.keyActions);
            }
            return ret;
        }
        if (opts.streamActions) {
            if (!this.tableStreamArn) {
                throw new Error(`DynamoDB Streams must be enabled on the table ${this.node.path}`);
            }
            const resources = [this.tableStreamArn];
            const ret = iam.Grant.addToPrincipal({
                grantee,
                actions: opts.streamActions,
                resourceArns: resources,
                scope: this,
            });
            return ret;
        }
        throw new Error(`Unexpected 'action', ${opts.tableActions || opts.streamActions}`);
    }
    cannedMetric(fn, props) {
        return new cloudwatch.Metric({
            ...fn({ TableName: this.tableName }),
            ...props,
        }).attachTo(this);
    }
}
/**
 * Provides a DynamoDB table.
 */
class Table extends TableBase {
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id, {
            physicalName: props.tableName,
        });
        this.keySchema = new Array();
        this.attributeDefinitions = new Array();
        this.globalSecondaryIndexes = new Array();
        this.localSecondaryIndexes = new Array();
        this.secondaryIndexSchemas = new Map();
        this.nonKeyAttributes = new Set();
        this.tableScaling = {};
        this.indexScaling = new Map();
        this.globalReplicaCustomResources = new Array();
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_TableProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.constructor);
            }
            throw error;
        }
        const { sseSpecification, encryptionKey } = this.parseEncryption(props);
        let streamSpecification;
        if (props.replicationRegions) {
            if (props.stream && props.stream !== StreamViewType.NEW_AND_OLD_IMAGES) {
                throw new Error('`stream` must be set to `NEW_AND_OLD_IMAGES` when specifying `replicationRegions`');
            }
            streamSpecification = { streamViewType: StreamViewType.NEW_AND_OLD_IMAGES };
            this.billingMode = (_b = props.billingMode) !== null && _b !== void 0 ? _b : BillingMode.PAY_PER_REQUEST;
        }
        else {
            this.billingMode = (_c = props.billingMode) !== null && _c !== void 0 ? _c : BillingMode.PROVISIONED;
            if (props.stream) {
                streamSpecification = { streamViewType: props.stream };
            }
        }
        this.validateProvisioning(props);
        this.table = new dynamodb_generated_1.CfnTable(this, 'Resource', {
            tableName: this.physicalName,
            keySchema: this.keySchema,
            attributeDefinitions: this.attributeDefinitions,
            globalSecondaryIndexes: core_1.Lazy.any({ produce: () => this.globalSecondaryIndexes }, { omitEmptyArray: true }),
            localSecondaryIndexes: core_1.Lazy.any({ produce: () => this.localSecondaryIndexes }, { omitEmptyArray: true }),
            pointInTimeRecoverySpecification: props.pointInTimeRecovery != null ? { pointInTimeRecoveryEnabled: props.pointInTimeRecovery } : undefined,
            billingMode: this.billingMode === BillingMode.PAY_PER_REQUEST ? this.billingMode : undefined,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
            sseSpecification,
            streamSpecification,
            tableClass: props.tableClass,
            timeToLiveSpecification: props.timeToLiveAttribute ? { attributeName: props.timeToLiveAttribute, enabled: true } : undefined,
            contributorInsightsSpecification: props.contributorInsightsEnabled !== undefined ? { enabled: props.contributorInsightsEnabled } : undefined,
            kinesisStreamSpecification: props.kinesisStream ? { streamArn: props.kinesisStream.streamArn } : undefined,
        });
        this.table.applyRemovalPolicy(props.removalPolicy);
        this.encryptionKey = encryptionKey;
        this.tableArn = this.getResourceArnAttribute(this.table.attrArn, {
            service: 'dynamodb',
            resource: 'table',
            resourceName: this.physicalName,
        });
        this.tableName = this.getResourceNameAttribute(this.table.ref);
        if (props.tableName) {
            this.node.addMetadata('aws:cdk:hasPhysicalName', this.tableName);
        }
        this.tableStreamArn = streamSpecification ? this.table.attrStreamArn : undefined;
        this.scalingRole = this.makeScalingRole();
        this.addKey(props.partitionKey, HASH_KEY_TYPE);
        this.tablePartitionKey = props.partitionKey;
        if (props.sortKey) {
            this.addKey(props.sortKey, RANGE_KEY_TYPE);
            this.tableSortKey = props.sortKey;
        }
        if (props.replicationRegions && props.replicationRegions.length > 0) {
            this.createReplicaTables(props.replicationRegions, props.replicationTimeout, props.waitForReplicationToFinish);
        }
    }
    /**
     * Permits an IAM Principal to list all DynamoDB Streams.
     * @deprecated Use {@link #grantTableListStreams} for more granular permission
     * @param grantee The principal (no-op if undefined)
     */
    static grantListStreams(grantee) {
        try {
            jsiiDeprecationWarnings.print("@aws-cdk/aws-dynamodb.Table#grantListStreams", "Use {@link #grantTableListStreams} for more granular permission");
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.grantListStreams);
            }
            throw error;
        }
        return iam.Grant.addToPrincipal({
            grantee,
            actions: ['dynamodb:ListStreams'],
            resourceArns: ['*'],
        });
    }
    /**
     * Creates a Table construct that represents an external table via table name.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableName The table's name.
     */
    static fromTableName(scope, id, tableName) {
        return Table.fromTableAttributes(scope, id, { tableName });
    }
    /**
     * Creates a Table construct that represents an external table via table arn.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param tableArn The table's ARN.
     */
    static fromTableArn(scope, id, tableArn) {
        return Table.fromTableAttributes(scope, id, { tableArn });
    }
    /**
     * Creates a Table construct that represents an external table.
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs A `TableAttributes` object.
     */
    static fromTableAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_TableAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromTableAttributes);
            }
            throw error;
        }
        class Import extends TableBase {
            constructor(_tableArn, tableName, tableStreamArn) {
                var _b, _c;
                super(scope, id);
                this.hasIndex = ((_b = attrs.globalIndexes) !== null && _b !== void 0 ? _b : []).length > 0 ||
                    ((_c = attrs.localIndexes) !== null && _c !== void 0 ? _c : []).length > 0;
                this.tableArn = _tableArn;
                this.tableName = tableName;
                this.tableStreamArn = tableStreamArn;
                this.encryptionKey = attrs.encryptionKey;
            }
        }
        let name;
        let arn;
        const stack = core_1.Stack.of(scope);
        if (!attrs.tableName) {
            if (!attrs.tableArn) {
                throw new Error('One of tableName or tableArn is required!');
            }
            arn = attrs.tableArn;
            const maybeTableName = stack.splitArn(attrs.tableArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeTableName) {
                throw new Error('ARN for DynamoDB table must be in the form: ...');
            }
            name = maybeTableName;
        }
        else {
            if (attrs.tableArn) {
                throw new Error('Only one of tableArn or tableName can be provided');
            }
            name = attrs.tableName;
            arn = stack.formatArn({
                service: 'dynamodb',
                resource: 'table',
                resourceName: attrs.tableName,
            });
        }
        return new Import(arn, name, attrs.tableStreamArn);
    }
    /**
     * Add a global secondary index of table.
     *
     * @param props the property of global secondary index
     */
    addGlobalSecondaryIndex(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_GlobalSecondaryIndexProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addGlobalSecondaryIndex);
            }
            throw error;
        }
        this.validateProvisioning(props);
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const gsiKeySchema = this.buildIndexKeySchema(props.partitionKey, props.sortKey);
        const gsiProjection = this.buildIndexProjection(props);
        this.globalSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: gsiKeySchema,
            projection: gsiProjection,
            provisionedThroughput: this.billingMode === BillingMode.PAY_PER_REQUEST ? undefined : {
                readCapacityUnits: props.readCapacity || 5,
                writeCapacityUnits: props.writeCapacity || 5,
            },
        });
        this.secondaryIndexSchemas.set(props.indexName, {
            partitionKey: props.partitionKey,
            sortKey: props.sortKey,
        });
        this.indexScaling.set(props.indexName, {});
    }
    /**
     * Add a local secondary index of table.
     *
     * @param props the property of local secondary index
     */
    addLocalSecondaryIndex(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_LocalSecondaryIndexProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addLocalSecondaryIndex);
            }
            throw error;
        }
        // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
        if (this.localSecondaryIndexes.length >= MAX_LOCAL_SECONDARY_INDEX_COUNT) {
            throw new RangeError(`a maximum number of local secondary index per table is ${MAX_LOCAL_SECONDARY_INDEX_COUNT}`);
        }
        this.validateIndexName(props.indexName);
        // build key schema and projection for index
        const lsiKeySchema = this.buildIndexKeySchema(this.tablePartitionKey, props.sortKey);
        const lsiProjection = this.buildIndexProjection(props);
        this.localSecondaryIndexes.push({
            indexName: props.indexName,
            keySchema: lsiKeySchema,
            projection: lsiProjection,
        });
        this.secondaryIndexSchemas.set(props.indexName, {
            partitionKey: this.tablePartitionKey,
            sortKey: props.sortKey,
        });
    }
    /**
     * Enable read capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings
     */
    autoScaleReadCapacity(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_EnableScalingProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.autoScaleReadCapacity);
            }
            throw error;
        }
        if (this.tableScaling.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        return this.tableScaling.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'ReadScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for this table
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleWriteCapacity(props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_EnableScalingProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.autoScaleWriteCapacity);
            }
            throw error;
        }
        if (this.tableScaling.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this table');
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        this.tableScaling.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, 'WriteScaling', {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}`,
            dimension: 'dynamodb:table:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
        for (const globalReplicaCustomResource of this.globalReplicaCustomResources) {
            globalReplicaCustomResource.node.addDependency(this.tableScaling.scalableWriteAttribute);
        }
        return this.tableScaling.scalableWriteAttribute;
    }
    /**
     * Enable read capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexReadCapacity(indexName, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_EnableScalingProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.autoScaleGlobalSecondaryIndexReadCapacity);
            }
            throw error;
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableReadAttribute) {
            throw new Error('Read AutoScaling already enabled for this index');
        }
        return attributePair.scalableReadAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}ReadScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:ReadCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Enable write capacity scaling for the given GSI
     *
     * @returns An object to configure additional AutoScaling settings for this attribute
     */
    autoScaleGlobalSecondaryIndexWriteCapacity(indexName, props) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_dynamodb_EnableScalingProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.autoScaleGlobalSecondaryIndexWriteCapacity);
            }
            throw error;
        }
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            throw new Error('AutoScaling is not available for tables with PAY_PER_REQUEST billing mode');
        }
        const attributePair = this.indexScaling.get(indexName);
        if (!attributePair) {
            throw new Error(`No global secondary index with name ${indexName}`);
        }
        if (attributePair.scalableWriteAttribute) {
            throw new Error('Write AutoScaling already enabled for this index');
        }
        return attributePair.scalableWriteAttribute = new scalable_table_attribute_1.ScalableTableAttribute(this, `${indexName}WriteScaling`, {
            serviceNamespace: appscaling.ServiceNamespace.DYNAMODB,
            resourceId: `table/${this.tableName}/index/${indexName}`,
            dimension: 'dynamodb:index:WriteCapacityUnits',
            role: this.scalingRole,
            ...props,
        });
    }
    /**
     * Get schema attributes of table or index.
     *
     * @returns Schema of table or index.
     */
    schema(indexName) {
        if (!indexName) {
            return {
                partitionKey: this.tablePartitionKey,
                sortKey: this.tableSortKey,
            };
        }
        let schema = this.secondaryIndexSchemas.get(indexName);
        if (!schema) {
            throw new Error(`Cannot find schema for index: ${indexName}. Use 'addGlobalSecondaryIndex' or 'addLocalSecondaryIndex' to add index`);
        }
        return schema;
    }
    /**
     * Validate the table construct.
     *
     * @returns an array of validation error message
     */
    validate() {
        const errors = new Array();
        if (!this.tablePartitionKey) {
            errors.push('a partition key must be specified');
        }
        if (this.localSecondaryIndexes.length > 0 && !this.tableSortKey) {
            errors.push('a sort key of the table must be specified to add local secondary indexes');
        }
        if (this.globalReplicaCustomResources.length > 0 && this.billingMode === BillingMode.PROVISIONED) {
            const writeAutoScaleAttribute = this.tableScaling.scalableWriteAttribute;
            if (!writeAutoScaleAttribute) {
                errors.push('A global Table that uses PROVISIONED as the billing mode needs auto-scaled write capacity. ' +
                    'Use the autoScaleWriteCapacity() method to enable it.');
            }
            else if (!writeAutoScaleAttribute._scalingPolicyCreated) {
                errors.push('A global Table that uses PROVISIONED as the billing mode needs auto-scaled write capacity with a policy. ' +
                    'Call one of the scaleOn*() methods of the object returned from autoScaleWriteCapacity()');
            }
        }
        return errors;
    }
    /**
     * Validate read and write capacity are not specified for on-demand tables (billing mode PAY_PER_REQUEST).
     *
     * @param props read and write capacity properties
     */
    validateProvisioning(props) {
        if (this.billingMode === BillingMode.PAY_PER_REQUEST) {
            if (props.readCapacity !== undefined || props.writeCapacity !== undefined) {
                throw new Error('you cannot provision read and write capacity for a table with PAY_PER_REQUEST billing mode');
            }
        }
    }
    /**
     * Validate index name to check if a duplicate name already exists.
     *
     * @param indexName a name of global or local secondary index
     */
    validateIndexName(indexName) {
        if (this.secondaryIndexSchemas.has(indexName)) {
            // a duplicate index name causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`a duplicate index name, ${indexName}, is not allowed`);
        }
    }
    /**
     * Validate non-key attributes by checking limits within secondary index, which may vary in future.
     *
     * @param nonKeyAttributes a list of non-key attribute names
     */
    validateNonKeyAttributes(nonKeyAttributes) {
        if (this.nonKeyAttributes.size + nonKeyAttributes.length > 100) {
            // https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/Limits.html#limits-secondary-indexes
            throw new RangeError('a maximum number of nonKeyAttributes across all of secondary indexes is 100');
        }
        // store all non-key attributes
        nonKeyAttributes.forEach(att => this.nonKeyAttributes.add(att));
    }
    buildIndexKeySchema(partitionKey, sortKey) {
        this.registerAttribute(partitionKey);
        const indexKeySchema = [
            { attributeName: partitionKey.name, keyType: HASH_KEY_TYPE },
        ];
        if (sortKey) {
            this.registerAttribute(sortKey);
            indexKeySchema.push({ attributeName: sortKey.name, keyType: RANGE_KEY_TYPE });
        }
        return indexKeySchema;
    }
    buildIndexProjection(props) {
        var _b, _c;
        if (props.projectionType === ProjectionType.INCLUDE && !props.nonKeyAttributes) {
            // https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-dynamodb-projectionobject.html
            throw new Error(`non-key attributes should be specified when using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.projectionType !== ProjectionType.INCLUDE && props.nonKeyAttributes) {
            // this combination causes validation exception, status code 400, while trying to create CFN stack
            throw new Error(`non-key attributes should not be specified when not using ${ProjectionType.INCLUDE} projection type`);
        }
        if (props.nonKeyAttributes) {
            this.validateNonKeyAttributes(props.nonKeyAttributes);
        }
        return {
            projectionType: (_b = props.projectionType) !== null && _b !== void 0 ? _b : ProjectionType.ALL,
            nonKeyAttributes: (_c = props.nonKeyAttributes) !== null && _c !== void 0 ? _c : undefined,
        };
    }
    findKey(keyType) {
        return this.keySchema.find(prop => prop.keyType === keyType);
    }
    addKey(attribute, keyType) {
        const existingProp = this.findKey(keyType);
        if (existingProp) {
            throw new Error(`Unable to set ${attribute.name} as a ${keyType} key, because ${existingProp.attributeName} is a ${keyType} key`);
        }
        this.registerAttribute(attribute);
        this.keySchema.push({
            attributeName: attribute.name,
            keyType,
        });
        return this;
    }
    /**
     * Register the key attribute of table or secondary index to assemble attribute definitions of TableResourceProps.
     *
     * @param attribute the key attribute of table or secondary index
     */
    registerAttribute(attribute) {
        const { name, type } = attribute;
        const existingDef = this.attributeDefinitions.find(def => def.attributeName === name);
        if (existingDef && existingDef.attributeType !== type) {
            throw new Error(`Unable to specify ${name} as ${type} because it was already defined as ${existingDef.attributeType}`);
        }
        if (!existingDef) {
            this.attributeDefinitions.push({
                attributeName: name,
                attributeType: type,
            });
        }
    }
    /**
     * Return the role that will be used for AutoScaling
     */
    makeScalingRole() {
        // Use a Service Linked Role.
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        return iam.Role.fromRoleArn(this, 'ScalingRole', core_1.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/dynamodb.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_DynamoDBTable',
        }));
    }
    /**
     * Creates replica tables
     *
     * @param regions regions where to create tables
     */
    createReplicaTables(regions, timeout, waitForReplicationToFinish) {
        const stack = core_1.Stack.of(this);
        if (!core_1.Token.isUnresolved(stack.region) && regions.includes(stack.region)) {
            throw new Error('`replicationRegions` cannot include the region where this stack is deployed.');
        }
        const provider = replica_provider_1.ReplicaProvider.getOrCreate(this, { timeout });
        // Documentation at https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/V2gt_IAM.html
        // is currently incorrect. AWS Support recommends `dynamodb:*` in both source and destination regions
        const onEventHandlerPolicy = new SourceTableAttachedPolicy(this, provider.onEventHandler.role);
        const isCompleteHandlerPolicy = new SourceTableAttachedPolicy(this, provider.isCompleteHandler.role);
        // Permissions in the source region
        this.grant(onEventHandlerPolicy, 'dynamodb:*');
        this.grant(isCompleteHandlerPolicy, 'dynamodb:DescribeTable');
        let previousRegion;
        let previousRegionCondition;
        for (const region of new Set(regions)) { // Remove duplicates
            // Use multiple custom resources because multiple create/delete
            // updates cannot be combined in a single API call.
            const currentRegion = new core_1.CustomResource(this, `Replica${region}`, {
                serviceToken: provider.provider.serviceToken,
                resourceType: 'Custom::DynamoDBReplica',
                properties: {
                    TableName: this.tableName,
                    Region: region,
                    SkipReplicationCompletedWait: waitForReplicationToFinish == null
                        ? undefined
                        // CFN changes Custom Resource properties to strings anyways,
                        // so let's do that ourselves to make it clear in the handler this is a string, not a boolean
                        : (!waitForReplicationToFinish).toString(),
                },
            });
            currentRegion.node.addDependency(onEventHandlerPolicy.policy, isCompleteHandlerPolicy.policy);
            this.globalReplicaCustomResources.push(currentRegion);
            // Deploy time check to prevent from creating a replica in the region
            // where this stack is deployed. Only needed for environment agnostic
            // stacks.
            let createReplica;
            if (core_1.Token.isUnresolved(stack.region)) {
                createReplica = new core_1.CfnCondition(this, `StackRegionNotEquals${region}`, {
                    expression: core_1.Fn.conditionNot(core_1.Fn.conditionEquals(region, core_1.Aws.REGION)),
                });
                const cfnCustomResource = currentRegion.node.defaultChild;
                cfnCustomResource.cfnOptions.condition = createReplica;
            }
            // Save regional arns for grantXxx() methods
            this.regionalArns.push(stack.formatArn({
                region,
                service: 'dynamodb',
                resource: 'table',
                resourceName: this.tableName,
            }));
            // We need to create/delete regions sequentially because we cannot
            // have multiple table updates at the same time. The `isCompleteHandler`
            // of the provider waits until the replica is in an ACTIVE state.
            if (previousRegion) {
                if (previousRegionCondition) {
                    // we can't simply use a Dependency,
                    // because the previousRegion is protected by the "different region" Condition,
                    // and you can't have Fn::If in DependsOn.
                    // Instead, rely on Ref adding a dependency implicitly!
                    const previousRegionCfnResource = previousRegion.node.defaultChild;
                    const currentRegionCfnResource = currentRegion.node.defaultChild;
                    currentRegionCfnResource.addMetadata('DynamoDbReplicationDependency', core_1.Fn.conditionIf(previousRegionCondition.logicalId, previousRegionCfnResource.ref, core_1.Aws.NO_VALUE));
                }
                else {
                    currentRegion.node.addDependency(previousRegion);
                }
            }
            previousRegion = currentRegion;
            previousRegionCondition = createReplica;
        }
        // Permissions in the destination regions (outside of the loop to
        // minimize statements in the policy)
        onEventHandlerPolicy.grantPrincipal.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['dynamodb:*'],
            resources: this.regionalArns,
        }));
    }
    /**
     * Whether this table has indexes
     */
    get hasIndex() {
        return this.globalSecondaryIndexes.length + this.localSecondaryIndexes.length > 0;
    }
    /**
     * Set up key properties and return the Table encryption property from the
     * user's configuration.
     */
    parseEncryption(props) {
        var _b;
        let encryptionType = props.encryption;
        if (encryptionType != null && props.serverSideEncryption != null) {
            throw new Error('Only one of encryption and serverSideEncryption can be specified, but both were provided');
        }
        if (props.serverSideEncryption && props.encryptionKey) {
            throw new Error('encryptionKey cannot be specified when serverSideEncryption is specified. Use encryption instead');
        }
        if (encryptionType === undefined) {
            encryptionType = props.encryptionKey != null
                // If there is a configured encryptionKey, the encryption is implicitly CUSTOMER_MANAGED
                ? TableEncryption.CUSTOMER_MANAGED
                // Otherwise, if severSideEncryption is enabled, it's AWS_MANAGED; else undefined (do not set anything)
                : props.serverSideEncryption ? TableEncryption.AWS_MANAGED : undefined;
        }
        if (encryptionType !== TableEncryption.CUSTOMER_MANAGED && props.encryptionKey) {
            throw new Error('`encryptionKey cannot be specified unless encryption is set to TableEncryption.CUSTOMER_MANAGED (it was set to ${encryptionType})`');
        }
        if (encryptionType === TableEncryption.CUSTOMER_MANAGED && props.replicationRegions) {
            throw new Error('TableEncryption.CUSTOMER_MANAGED is not supported by DynamoDB Global Tables (where replicationRegions was set)');
        }
        switch (encryptionType) {
            case TableEncryption.CUSTOMER_MANAGED:
                const encryptionKey = (_b = props.encryptionKey) !== null && _b !== void 0 ? _b : new kms.Key(this, 'Key', {
                    description: `Customer-managed key auto-created for encrypting DynamoDB table at ${this.node.path}`,
                    enableKeyRotation: true,
                });
                return {
                    sseSpecification: { sseEnabled: true, kmsMasterKeyId: encryptionKey.keyArn, sseType: 'KMS' },
                    encryptionKey,
                };
            case TableEncryption.AWS_MANAGED:
                // Not specifying "sseType: 'KMS'" here because it would cause phony changes to existing stacks.
                return { sseSpecification: { sseEnabled: true } };
            case TableEncryption.DEFAULT:
                return { sseSpecification: { sseEnabled: false } };
            case undefined:
                // Not specifying "sseEnabled: false" here because it would cause phony changes to existing stacks.
                return { sseSpecification: undefined };
            default:
                throw new Error(`Unexpected 'encryptionType': ${encryptionType}`);
        }
    }
}
exports.Table = Table;
_a = JSII_RTTI_SYMBOL_1;
Table[_a] = { fqn: "@aws-cdk/aws-dynamodb.Table", version: "1.158.0" };
/**
 * Data types for attributes within a table
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.NamingRulesDataTypes.html#HowItWorks.DataTypes
 */
var AttributeType;
(function (AttributeType) {
    /** Up to 400KiB of binary data (which must be encoded as base64 before sending to DynamoDB) */
    AttributeType["BINARY"] = "B";
    /** Numeric values made of up to 38 digits (positive, negative or zero) */
    AttributeType["NUMBER"] = "N";
    /** Up to 400KiB of UTF-8 encoded text */
    AttributeType["STRING"] = "S";
})(AttributeType = exports.AttributeType || (exports.AttributeType = {}));
/**
 * DynamoDB's Read/Write capacity modes.
 */
var BillingMode;
(function (BillingMode) {
    /**
     * Pay only for what you use. You don't configure Read/Write capacity units.
     */
    BillingMode["PAY_PER_REQUEST"] = "PAY_PER_REQUEST";
    /**
     * Explicitly specified Read/Write capacity units.
     */
    BillingMode["PROVISIONED"] = "PROVISIONED";
})(BillingMode = exports.BillingMode || (exports.BillingMode = {}));
/**
 * The set of attributes that are projected into the index
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_Projection.html
 */
var ProjectionType;
(function (ProjectionType) {
    /** Only the index and primary keys are projected into the index. */
    ProjectionType["KEYS_ONLY"] = "KEYS_ONLY";
    /** Only the specified table attributes are projected into the index. The list of projected attributes is in `nonKeyAttributes`. */
    ProjectionType["INCLUDE"] = "INCLUDE";
    /** All of the table attributes are projected into the index. */
    ProjectionType["ALL"] = "ALL";
})(ProjectionType = exports.ProjectionType || (exports.ProjectionType = {}));
/**
 * When an item in the table is modified, StreamViewType determines what information
 * is written to the stream for this table.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/APIReference/API_StreamSpecification.html
 */
var StreamViewType;
(function (StreamViewType) {
    /** The entire item, as it appears after it was modified, is written to the stream. */
    StreamViewType["NEW_IMAGE"] = "NEW_IMAGE";
    /** The entire item, as it appeared before it was modified, is written to the stream. */
    StreamViewType["OLD_IMAGE"] = "OLD_IMAGE";
    /** Both the new and the old item images of the item are written to the stream. */
    StreamViewType["NEW_AND_OLD_IMAGES"] = "NEW_AND_OLD_IMAGES";
    /** Only the key attributes of the modified item are written to the stream. */
    StreamViewType["KEYS_ONLY"] = "KEYS_ONLY";
})(StreamViewType = exports.StreamViewType || (exports.StreamViewType = {}));
/**
 * DynamoDB's table class.
 *
 * @see https://docs.aws.amazon.com/amazondynamodb/latest/developerguide/HowItWorks.TableClasses.html
 */
var TableClass;
(function (TableClass) {
    /** Default table class for DynamoDB. */
    TableClass["STANDARD"] = "STANDARD";
    /** Table class for DynamoDB that reduces storage costs compared to existing DynamoDB Standard tables. */
    TableClass["STANDARD_INFREQUENT_ACCESS"] = "STANDARD_INFREQUENT_ACCESS";
})(TableClass = exports.TableClass || (exports.TableClass = {}));
/**
 * An inline policy that is logically bound to the source table of a DynamoDB Global Tables
 * "cluster". This is here to ensure permissions are removed as part of (and not before) the
 * CleanUp phase of a stack update, when a replica is removed (or the entire "cluster" gets
 * replaced).
 *
 * If statements are added directly to the handler roles (as opposed to in a separate inline
 * policy resource), new permissions are in effect before clean up happens, and so replicas that
 * need to be dropped can no longer be due to lack of permissions.
 */
class SourceTableAttachedPolicy extends core_2.Construct {
    constructor(sourceTable, role) {
        super(sourceTable, `SourceTableAttachedManagedPolicy-${core_1.Names.nodeUniqueId(role.node)}`);
        const policy = new iam.ManagedPolicy(this, 'Resource', {
            // A CF update of the description property of a managed policy requires
            // a replacement. Use the table name in the description to force a managed
            // policy replacement when the table name changes. This way we preserve permissions
            // to delete old replicas in case of a table replacement.
            description: `DynamoDB replication managed policy for table ${sourceTable.tableName}`,
            roles: [role],
        });
        this.policy = policy;
        this.grantPrincipal = new SourceTableAttachedPrincipal(role, policy);
    }
}
/**
 * An `IPrincipal` entity that can be used as the target of `grant` calls, used by the
 * `SourceTableAttachedPolicy` class so it can act as an `IGrantable`.
 */
class SourceTableAttachedPrincipal extends iam.PrincipalBase {
    constructor(role, policy) {
        super();
        this.role = role;
        this.policy = policy;
    }
    get policyFragment() {
        return this.role.policyFragment;
    }
    addToPrincipalPolicy(statement) {
        this.policy.addStatements(statement);
        return {
            policyDependable: this.policy,
            statementAdded: true,
        };
    }
    dedupeString() {
        return undefined;
    }
}
//# sourceMappingURL=data:application/json;base64,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