# Apideck Python SDK

## Table of Contents

- [Table of Contents](#table-of-contents)
- [Installation](#installation--usage)
- [Getting started](#getting-started)
- [Example](#example)
- [Support](#support)
- [License](#license)

## Requirements.

Python >=3.6

## Installation & Usage
### pip install

If the python package is hosted on a repository, you can install directly using:

```sh
pip install git+https://github.com/apideck-libraries/python-sdk.git
```
(you may need to run `pip` with root permission: `sudo pip install git+https://github.com/apideck-libraries/python-sdk.git`)

Then import the package:
```python
import apideck
```

### Setuptools

Install via [Setuptools](http://pypi.python.org/pypi/setuptools).

```sh
python setup.py install --user
```
(or `sudo python setup.py install` to install the package for all users)

Then import the package:
```python
import apideck
```

## Getting started

The module supports all Apideck API endpoints. For complete information about the API, head
to the [docs][2].

## Example

Retrieving a list of all contacts and updating the first record with a new address.

```python
import apideck
from apideck.api import crm_api
from apideck.model.contacts_sort import ContactsSort
from apideck.model.sort_direction import SortDirection
from pprint import pprint

configuration = apideck.Configuration()

configuration.api_key['apiKey'] = '<insert-api-key-here>'
configuration.api_key_prefix['apiKey'] = 'Bearer'

with apideck.ApiClient(configuration) as api_client:
    # Create an instance of the API class
    api_instance = crm_api.CrmApi(api_client)
    raw = False
    consumer_id = '<insert-consumer-id-here>'
    app_id = '<insert-application-id-here>'
    service_id = '<insert-service-id-here>'
    limit = 20

    sort = ContactsSort(
        by="name",
        direction=SortDirection("asc"),
    ) 
    

    try:
        # List contacts
        api_response = api_instance.contacts_all(raw=raw, consumer_id=consumer_id, app_id=app_id, service_id=service_id, limit=limit, sort=sort)
        pprint(api_response)
    except apideck.ApiException as e:
        print("Exception when calling CrmApi->contacts_all: %s\n" % e)
```

## Apideck Unified Apis

The following Apis are currently available:

### AccountingApi

Read the full documentation of the AccountingApi [here](./docs/apis/AccountingApi.md).

### AtsApi

Read the full documentation of the AtsApi [here](./docs/apis/AtsApi.md).

### ConnectorApi

Read the full documentation of the ConnectorApi [here](./docs/apis/ConnectorApi.md).

### CrmApi

Read the full documentation of the CrmApi [here](./docs/apis/CrmApi.md).

### CustomerSupportApi

Read the full documentation of the CustomerSupportApi [here](./docs/apis/CustomerSupportApi.md).

### FileStorageApi

Read the full documentation of the FileStorageApi [here](./docs/apis/FileStorageApi.md).

### HrisApi

Read the full documentation of the HrisApi [here](./docs/apis/HrisApi.md).

### LeadApi

Read the full documentation of the LeadApi [here](./docs/apis/LeadApi.md).

### PosApi

Read the full documentation of the PosApi [here](./docs/apis/PosApi.md).

### SmsApi

Read the full documentation of the SmsApi [here](./docs/apis/SmsApi.md).

### VaultApi

Read the full documentation of the VaultApi [here](./docs/apis/VaultApi.md).

### WebhookApi

Read the full documentation of the WebhookApi [here](./docs/apis/WebhookApi.md).


### Utils

#### uploadFile

A utility for uploading files using the File Storage API. `uploadFile` will automatically use upload sessions for files larger than 4MB. Smaller files will be uploaded with a simple upload call.

**Example Usage**


## Support

Open an [issue][3]!

## License

[Apache-2.0][4]

[1]: https://apideck.com
[2]: https://developers.apideck.com/
[3]: https://github.com/apideck-libraries/python-sdk/issues/new
[4]: https://github.com/apideck-libraries/python-sdk/blob/master/LICENSE


```python

import time
import apideck
from pprint import pprint
from apideck.api import accounting_api
from apideck.model.accounting_customer import AccountingCustomer
from apideck.model.bad_request_response import BadRequestResponse
from apideck.model.balance_sheet_filter import BalanceSheetFilter
from apideck.model.bill import Bill
from apideck.model.create_bill_response import CreateBillResponse
from apideck.model.create_credit_note_response import CreateCreditNoteResponse
from apideck.model.create_customer_response import CreateCustomerResponse
from apideck.model.create_invoice_item_response import CreateInvoiceItemResponse
from apideck.model.create_invoice_response import CreateInvoiceResponse
from apideck.model.create_ledger_account_response import CreateLedgerAccountResponse
from apideck.model.create_payment_response import CreatePaymentResponse
from apideck.model.create_supplier_response import CreateSupplierResponse
from apideck.model.create_tax_rate_response import CreateTaxRateResponse
from apideck.model.credit_note import CreditNote
from apideck.model.customers_filter import CustomersFilter
from apideck.model.delete_bill_response import DeleteBillResponse
from apideck.model.delete_credit_note_response import DeleteCreditNoteResponse
from apideck.model.delete_customer_response import DeleteCustomerResponse
from apideck.model.delete_invoice_response import DeleteInvoiceResponse
from apideck.model.delete_ledger_account_response import DeleteLedgerAccountResponse
from apideck.model.delete_payment_response import DeletePaymentResponse
from apideck.model.delete_supplier_response import DeleteSupplierResponse
from apideck.model.delete_tax_rate_response import DeleteTaxRateResponse
from apideck.model.get_balance_sheet_response import GetBalanceSheetResponse
from apideck.model.get_bill_response import GetBillResponse
from apideck.model.get_bills_response import GetBillsResponse
from apideck.model.get_company_info_response import GetCompanyInfoResponse
from apideck.model.get_credit_note_response import GetCreditNoteResponse
from apideck.model.get_credit_notes_response import GetCreditNotesResponse
from apideck.model.get_customer_response import GetCustomerResponse
from apideck.model.get_customers_response import GetCustomersResponse
from apideck.model.get_invoice_item_response import GetInvoiceItemResponse
from apideck.model.get_invoice_items_response import GetInvoiceItemsResponse
from apideck.model.get_invoice_response import GetInvoiceResponse
from apideck.model.get_invoices_response import GetInvoicesResponse
from apideck.model.get_ledger_account_response import GetLedgerAccountResponse
from apideck.model.get_ledger_accounts_response import GetLedgerAccountsResponse
from apideck.model.get_payment_response import GetPaymentResponse
from apideck.model.get_payments_response import GetPaymentsResponse
from apideck.model.get_profit_and_loss_response import GetProfitAndLossResponse
from apideck.model.get_supplier_response import GetSupplierResponse
from apideck.model.get_suppliers_response import GetSuppliersResponse
from apideck.model.get_tax_rate_response import GetTaxRateResponse
from apideck.model.get_tax_rates_response import GetTaxRatesResponse
from apideck.model.invoice import Invoice
from apideck.model.invoice_item import InvoiceItem
from apideck.model.invoice_items_filter import InvoiceItemsFilter
from apideck.model.invoices_sort import InvoicesSort
from apideck.model.ledger_account import LedgerAccount
from apideck.model.not_found_response import NotFoundResponse
from apideck.model.passthrough import Passthrough
from apideck.model.payment import Payment
from apideck.model.payment_required_response import PaymentRequiredResponse
from apideck.model.profit_and_loss_filter import ProfitAndLossFilter
from apideck.model.supplier import Supplier
from apideck.model.tax_rate import TaxRate
from apideck.model.tax_rates_filter import TaxRatesFilter
from apideck.model.unauthorized_response import UnauthorizedResponse
from apideck.model.unexpected_error_response import UnexpectedErrorResponse
from apideck.model.unprocessable_response import UnprocessableResponse
from apideck.model.update_bill_response import UpdateBillResponse
from apideck.model.update_credit_note_response import UpdateCreditNoteResponse
from apideck.model.update_customer_response import UpdateCustomerResponse
from apideck.model.update_invoice_items_response import UpdateInvoiceItemsResponse
from apideck.model.update_invoice_response import UpdateInvoiceResponse
from apideck.model.update_ledger_account_response import UpdateLedgerAccountResponse
from apideck.model.update_payment_response import UpdatePaymentResponse
from apideck.model.update_supplier_response import UpdateSupplierResponse
from apideck.model.update_tax_rate_response import UpdateTaxRateResponse
# Defining the host is optional and defaults to https://unify.apideck.com
# See configuration.py for a list of all supported configuration parameters.
configuration = apideck.Configuration(
    host = "https://unify.apideck.com"
)

# The client must configure the authentication and authorization parameters
# in accordance with the API server security policy.
# Examples for each auth method are provided below, use the example that
# satisfies your auth use case.

# Configure API key authorization: apiKey
configuration.api_key['apiKey'] = 'YOUR_API_KEY'

# Uncomment below to setup prefix (e.g. Bearer) for API key, if needed
# configuration.api_key_prefix['apiKey'] = 'Bearer'


# Enter a context with an instance of the API client
with apideck.ApiClient(configuration) as api_client:
    # Create an instance of the API class
    api_instance = accounting_api.AccountingApi(api_client)
    x_apideck_consumer_id = "x-apideck-consumer-id_example" # str | ID of the consumer which you want to get or push data from (optional)
x_apideck_app_id = "x-apideck-app-id_example" # str | The ID of your Unify application (optional)
x_apideck_service_id = "x-apideck-service-id_example" # str | Provide the service id you want to call (e.g., pipedrive). Only needed when a consumer has activated multiple integrations for a Unified API. (optional)
pass_through = Passthrough() # Passthrough | Optional unmapped key/values that will be passed through to downstream as query parameters (optional)
filter = BalanceSheetFilter(
        start_date="2021-01-01",
        end_date="2021-12-31",
    ) # BalanceSheetFilter | Apply filters (optional)
raw = False # bool | Include raw response. Mostly used for debugging purposes (optional) (default to False)

    try:
        # Get BalanceSheet
        api_response = api_instance.balance_sheet_one(x_apideck_consumer_id=x_apideck_consumer_id, x_apideck_app_id=x_apideck_app_id, x_apideck_service_id=x_apideck_service_id, pass_through=pass_through, filter=filter, raw=raw)
        pprint(api_response)
    except apideck.ApiException as e:
        print("Exception when calling AccountingApi->balance_sheet_one: %s\n" % e)
```

## Documentation for API Endpoints

All URIs are relative to _https://unify.apideck.com_

| Class                                                             | Method                                                                         | HTTP request                | Description |
| ----------------------------------------------------------------- | ------------------------------------------------------------------------------ | --------------------------- | ----------- |
| _AccountingApi_ | [**balance_sheet_one**](docs/apis/AccountingApi.md#balance_sheet_one) | **GET** /accounting/balance-sheet | Get BalanceSheet |

_AccountingApi_ | [**bills_add**](docs/apis/AccountingApi.md#bills_add) | **POST** /accounting/bills | Create Bill |

_AccountingApi_ | [**bills_all**](docs/apis/AccountingApi.md#bills_all) | **GET** /accounting/bills | List Bills |

_AccountingApi_ | [**bills_delete**](docs/apis/AccountingApi.md#bills_delete) | **DELETE** /accounting/bills/{id} | Delete Bill |

_AccountingApi_ | [**bills_one**](docs/apis/AccountingApi.md#bills_one) | **GET** /accounting/bills/{id} | Get Bill |

_AccountingApi_ | [**bills_update**](docs/apis/AccountingApi.md#bills_update) | **PATCH** /accounting/bills/{id} | Update Bill |

_AccountingApi_ | [**company_info_one**](docs/apis/AccountingApi.md#company_info_one) | **GET** /accounting/company-info | Get company info |

_AccountingApi_ | [**credit_notes_add**](docs/apis/AccountingApi.md#credit_notes_add) | **POST** /accounting/credit-notes | Create Credit Note |

_AccountingApi_ | [**credit_notes_all**](docs/apis/AccountingApi.md#credit_notes_all) | **GET** /accounting/credit-notes | List Credit Notes |

_AccountingApi_ | [**credit_notes_delete**](docs/apis/AccountingApi.md#credit_notes_delete) | **DELETE** /accounting/credit-notes/{id} | Delete Credit Note |

_AccountingApi_ | [**credit_notes_one**](docs/apis/AccountingApi.md#credit_notes_one) | **GET** /accounting/credit-notes/{id} | Get Credit Note |

_AccountingApi_ | [**credit_notes_update**](docs/apis/AccountingApi.md#credit_notes_update) | **PATCH** /accounting/credit-notes/{id} | Update Credit Note |

_AccountingApi_ | [**customers_add**](docs/apis/AccountingApi.md#customers_add) | **POST** /accounting/customers | Create Customer |

_AccountingApi_ | [**customers_all**](docs/apis/AccountingApi.md#customers_all) | **GET** /accounting/customers | List Customers |

_AccountingApi_ | [**customers_delete**](docs/apis/AccountingApi.md#customers_delete) | **DELETE** /accounting/customers/{id} | Delete Customer |

_AccountingApi_ | [**customers_one**](docs/apis/AccountingApi.md#customers_one) | **GET** /accounting/customers/{id} | Get Customer |

_AccountingApi_ | [**customers_update**](docs/apis/AccountingApi.md#customers_update) | **PATCH** /accounting/customers/{id} | Update Customer |

_AccountingApi_ | [**invoice_items_add**](docs/apis/AccountingApi.md#invoice_items_add) | **POST** /accounting/invoice-items | Create Invoice Item |

_AccountingApi_ | [**invoice_items_all**](docs/apis/AccountingApi.md#invoice_items_all) | **GET** /accounting/invoice-items | List Invoice Items |

_AccountingApi_ | [**invoice_items_delete**](docs/apis/AccountingApi.md#invoice_items_delete) | **DELETE** /accounting/invoice-items/{id} | Delete Invoice Item |

_AccountingApi_ | [**invoice_items_one**](docs/apis/AccountingApi.md#invoice_items_one) | **GET** /accounting/invoice-items/{id} | Get Invoice Item |

_AccountingApi_ | [**invoice_items_update**](docs/apis/AccountingApi.md#invoice_items_update) | **PATCH** /accounting/invoice-items/{id} | Update Invoice Item |

_AccountingApi_ | [**invoices_add**](docs/apis/AccountingApi.md#invoices_add) | **POST** /accounting/invoices | Create Invoice |

_AccountingApi_ | [**invoices_all**](docs/apis/AccountingApi.md#invoices_all) | **GET** /accounting/invoices | List Invoices |

_AccountingApi_ | [**invoices_delete**](docs/apis/AccountingApi.md#invoices_delete) | **DELETE** /accounting/invoices/{id} | Delete Invoice |

_AccountingApi_ | [**invoices_one**](docs/apis/AccountingApi.md#invoices_one) | **GET** /accounting/invoices/{id} | Get Invoice |

_AccountingApi_ | [**invoices_update**](docs/apis/AccountingApi.md#invoices_update) | **PATCH** /accounting/invoices/{id} | Update Invoice |

_AccountingApi_ | [**ledger_accounts_add**](docs/apis/AccountingApi.md#ledger_accounts_add) | **POST** /accounting/ledger-accounts | Create Ledger Account |

_AccountingApi_ | [**ledger_accounts_all**](docs/apis/AccountingApi.md#ledger_accounts_all) | **GET** /accounting/ledger-accounts | List Ledger Accounts |

_AccountingApi_ | [**ledger_accounts_delete**](docs/apis/AccountingApi.md#ledger_accounts_delete) | **DELETE** /accounting/ledger-accounts/{id} | Delete Ledger Account |

_AccountingApi_ | [**ledger_accounts_one**](docs/apis/AccountingApi.md#ledger_accounts_one) | **GET** /accounting/ledger-accounts/{id} | Get Ledger Account |

_AccountingApi_ | [**ledger_accounts_update**](docs/apis/AccountingApi.md#ledger_accounts_update) | **PATCH** /accounting/ledger-accounts/{id} | Update Ledger Account |

_AccountingApi_ | [**payments_add**](docs/apis/AccountingApi.md#payments_add) | **POST** /accounting/payments | Create Payment |

_AccountingApi_ | [**payments_all**](docs/apis/AccountingApi.md#payments_all) | **GET** /accounting/payments | List Payments |

_AccountingApi_ | [**payments_delete**](docs/apis/AccountingApi.md#payments_delete) | **DELETE** /accounting/payments/{id} | Delete Payment |

_AccountingApi_ | [**payments_one**](docs/apis/AccountingApi.md#payments_one) | **GET** /accounting/payments/{id} | Get Payment |

_AccountingApi_ | [**payments_update**](docs/apis/AccountingApi.md#payments_update) | **PATCH** /accounting/payments/{id} | Update Payment |

_AccountingApi_ | [**profit_and_loss_one**](docs/apis/AccountingApi.md#profit_and_loss_one) | **GET** /accounting/profit-and-loss | Get Profit and Loss |

_AccountingApi_ | [**suppliers_add**](docs/apis/AccountingApi.md#suppliers_add) | **POST** /accounting/suppliers | Create Supplier |

_AccountingApi_ | [**suppliers_all**](docs/apis/AccountingApi.md#suppliers_all) | **GET** /accounting/suppliers | List Suppliers |

_AccountingApi_ | [**suppliers_delete**](docs/apis/AccountingApi.md#suppliers_delete) | **DELETE** /accounting/suppliers/{id} | Delete Supplier |

_AccountingApi_ | [**suppliers_one**](docs/apis/AccountingApi.md#suppliers_one) | **GET** /accounting/suppliers/{id} | Get Supplier |

_AccountingApi_ | [**suppliers_update**](docs/apis/AccountingApi.md#suppliers_update) | **PATCH** /accounting/suppliers/{id} | Update Supplier |

_AccountingApi_ | [**tax_rates_add**](docs/apis/AccountingApi.md#tax_rates_add) | **POST** /accounting/tax-rates | Create Tax Rate |

_AccountingApi_ | [**tax_rates_all**](docs/apis/AccountingApi.md#tax_rates_all) | **GET** /accounting/tax-rates | List Tax Rates |

_AccountingApi_ | [**tax_rates_delete**](docs/apis/AccountingApi.md#tax_rates_delete) | **DELETE** /accounting/tax-rates/{id} | Delete Tax Rate |

_AccountingApi_ | [**tax_rates_one**](docs/apis/AccountingApi.md#tax_rates_one) | **GET** /accounting/tax-rates/{id} | Get Tax Rate |

_AccountingApi_ | [**tax_rates_update**](docs/apis/AccountingApi.md#tax_rates_update) | **PATCH** /accounting/tax-rates/{id} | Update Tax Rate |

_AtsApi_ | [**applicants_add**](docs/apis/AtsApi.md#applicants_add) | **POST** /ats/applicants | Create applicant |

_AtsApi_ | [**applicants_all**](docs/apis/AtsApi.md#applicants_all) | **GET** /ats/applicants | List applicants |

_AtsApi_ | [**applicants_one**](docs/apis/AtsApi.md#applicants_one) | **GET** /ats/applicants/{id} | Get applicant |

_AtsApi_ | [**jobs_all**](docs/apis/AtsApi.md#jobs_all) | **GET** /ats/jobs | List Jobs |

_AtsApi_ | [**jobs_one**](docs/apis/AtsApi.md#jobs_one) | **GET** /ats/jobs/{id} | Get Job |

_ConnectorApi_ | [**api_resource_coverage_one**](docs/apis/ConnectorApi.md#api_resource_coverage_one) | **GET** /connector/apis/{id}/resources/{resource_id}/coverage | Get API Resource Coverage |

_ConnectorApi_ | [**api_resources_one**](docs/apis/ConnectorApi.md#api_resources_one) | **GET** /connector/apis/{id}/resources/{resource_id} | Get API Resource |

_ConnectorApi_ | [**apis_all**](docs/apis/ConnectorApi.md#apis_all) | **GET** /connector/apis | List APIs |

_ConnectorApi_ | [**apis_one**](docs/apis/ConnectorApi.md#apis_one) | **GET** /connector/apis/{id} | Get API |

_ConnectorApi_ | [**connector_docs_one**](docs/apis/ConnectorApi.md#connector_docs_one) | **GET** /connector/connectors/{id}/docs/{doc_id} | Get Connector Doc content |

_ConnectorApi_ | [**connector_resources_one**](docs/apis/ConnectorApi.md#connector_resources_one) | **GET** /connector/connectors/{id}/resources/{resource_id} | Get Connector Resource |

_ConnectorApi_ | [**connectors_all**](docs/apis/ConnectorApi.md#connectors_all) | **GET** /connector/connectors | List Connectors |

_ConnectorApi_ | [**connectors_one**](docs/apis/ConnectorApi.md#connectors_one) | **GET** /connector/connectors/{id} | Get Connector |

_CrmApi_ | [**activities_add**](docs/apis/CrmApi.md#activities_add) | **POST** /crm/activities | Create activity |

_CrmApi_ | [**activities_all**](docs/apis/CrmApi.md#activities_all) | **GET** /crm/activities | List activities |

_CrmApi_ | [**activities_delete**](docs/apis/CrmApi.md#activities_delete) | **DELETE** /crm/activities/{id} | Delete activity |

_CrmApi_ | [**activities_one**](docs/apis/CrmApi.md#activities_one) | **GET** /crm/activities/{id} | Get activity |

_CrmApi_ | [**activities_update**](docs/apis/CrmApi.md#activities_update) | **PATCH** /crm/activities/{id} | Update activity |

_CrmApi_ | [**companies_add**](docs/apis/CrmApi.md#companies_add) | **POST** /crm/companies | Create company |

_CrmApi_ | [**companies_all**](docs/apis/CrmApi.md#companies_all) | **GET** /crm/companies | List companies |

_CrmApi_ | [**companies_delete**](docs/apis/CrmApi.md#companies_delete) | **DELETE** /crm/companies/{id} | Delete company |

_CrmApi_ | [**companies_one**](docs/apis/CrmApi.md#companies_one) | **GET** /crm/companies/{id} | Get company |

_CrmApi_ | [**companies_update**](docs/apis/CrmApi.md#companies_update) | **PATCH** /crm/companies/{id} | Update company |

_CrmApi_ | [**contacts_add**](docs/apis/CrmApi.md#contacts_add) | **POST** /crm/contacts | Create contact |

_CrmApi_ | [**contacts_all**](docs/apis/CrmApi.md#contacts_all) | **GET** /crm/contacts | List contacts |

_CrmApi_ | [**contacts_delete**](docs/apis/CrmApi.md#contacts_delete) | **DELETE** /crm/contacts/{id} | Delete contact |

_CrmApi_ | [**contacts_one**](docs/apis/CrmApi.md#contacts_one) | **GET** /crm/contacts/{id} | Get contact |

_CrmApi_ | [**contacts_update**](docs/apis/CrmApi.md#contacts_update) | **PATCH** /crm/contacts/{id} | Update contact |

_CrmApi_ | [**leads_add**](docs/apis/CrmApi.md#leads_add) | **POST** /crm/leads | Create lead |

_CrmApi_ | [**leads_all**](docs/apis/CrmApi.md#leads_all) | **GET** /crm/leads | List leads |

_CrmApi_ | [**leads_delete**](docs/apis/CrmApi.md#leads_delete) | **DELETE** /crm/leads/{id} | Delete lead |

_CrmApi_ | [**leads_one**](docs/apis/CrmApi.md#leads_one) | **GET** /crm/leads/{id} | Get lead |

_CrmApi_ | [**leads_update**](docs/apis/CrmApi.md#leads_update) | **PATCH** /crm/leads/{id} | Update lead |

_CrmApi_ | [**notes_add**](docs/apis/CrmApi.md#notes_add) | **POST** /crm/notes | Create note |

_CrmApi_ | [**notes_all**](docs/apis/CrmApi.md#notes_all) | **GET** /crm/notes | List notes |

_CrmApi_ | [**notes_delete**](docs/apis/CrmApi.md#notes_delete) | **DELETE** /crm/notes/{id} | Delete note |

_CrmApi_ | [**notes_one**](docs/apis/CrmApi.md#notes_one) | **GET** /crm/notes/{id} | Get note |

_CrmApi_ | [**notes_update**](docs/apis/CrmApi.md#notes_update) | **PATCH** /crm/notes/{id} | Update note |

_CrmApi_ | [**opportunities_add**](docs/apis/CrmApi.md#opportunities_add) | **POST** /crm/opportunities | Create opportunity |

_CrmApi_ | [**opportunities_all**](docs/apis/CrmApi.md#opportunities_all) | **GET** /crm/opportunities | List opportunities |

_CrmApi_ | [**opportunities_delete**](docs/apis/CrmApi.md#opportunities_delete) | **DELETE** /crm/opportunities/{id} | Delete opportunity |

_CrmApi_ | [**opportunities_one**](docs/apis/CrmApi.md#opportunities_one) | **GET** /crm/opportunities/{id} | Get opportunity |

_CrmApi_ | [**opportunities_update**](docs/apis/CrmApi.md#opportunities_update) | **PATCH** /crm/opportunities/{id} | Update opportunity |

_CrmApi_ | [**pipelines_add**](docs/apis/CrmApi.md#pipelines_add) | **POST** /crm/pipelines | Create pipeline |

_CrmApi_ | [**pipelines_all**](docs/apis/CrmApi.md#pipelines_all) | **GET** /crm/pipelines | List pipelines |

_CrmApi_ | [**pipelines_delete**](docs/apis/CrmApi.md#pipelines_delete) | **DELETE** /crm/pipelines/{id} | Delete pipeline |

_CrmApi_ | [**pipelines_one**](docs/apis/CrmApi.md#pipelines_one) | **GET** /crm/pipelines/{id} | Get pipeline |

_CrmApi_ | [**pipelines_update**](docs/apis/CrmApi.md#pipelines_update) | **PATCH** /crm/pipelines/{id} | Update pipeline |

_CrmApi_ | [**users_add**](docs/apis/CrmApi.md#users_add) | **POST** /crm/users | Create user |

_CrmApi_ | [**users_all**](docs/apis/CrmApi.md#users_all) | **GET** /crm/users | List users |

_CrmApi_ | [**users_delete**](docs/apis/CrmApi.md#users_delete) | **DELETE** /crm/users/{id} | Delete user |

_CrmApi_ | [**users_one**](docs/apis/CrmApi.md#users_one) | **GET** /crm/users/{id} | Get user |

_CrmApi_ | [**users_update**](docs/apis/CrmApi.md#users_update) | **PATCH** /crm/users/{id} | Update user |

_CustomerSupportApi_ | [**customers_add**](docs/apis/CustomerSupportApi.md#customers_add) | **POST** /customer-support/customers | Create Customer Support Customer |

_CustomerSupportApi_ | [**customers_all**](docs/apis/CustomerSupportApi.md#customers_all) | **GET** /customer-support/customers | List Customer Support Customers |

_CustomerSupportApi_ | [**customers_delete**](docs/apis/CustomerSupportApi.md#customers_delete) | **DELETE** /customer-support/customers/{id} | Delete Customer Support Customer |

_CustomerSupportApi_ | [**customers_one**](docs/apis/CustomerSupportApi.md#customers_one) | **GET** /customer-support/customers/{id} | Get Customer Support Customer |

_CustomerSupportApi_ | [**customers_update**](docs/apis/CustomerSupportApi.md#customers_update) | **PATCH** /customer-support/customers/{id} | Update Customer Support Customer |

_FileStorageApi_ | [**drive_groups_add**](docs/apis/FileStorageApi.md#drive_groups_add) | **POST** /file-storage/drive-groups | Create DriveGroup |

_FileStorageApi_ | [**drive_groups_all**](docs/apis/FileStorageApi.md#drive_groups_all) | **GET** /file-storage/drive-groups | List DriveGroups |

_FileStorageApi_ | [**drive_groups_delete**](docs/apis/FileStorageApi.md#drive_groups_delete) | **DELETE** /file-storage/drive-groups/{id} | Delete DriveGroup |

_FileStorageApi_ | [**drive_groups_one**](docs/apis/FileStorageApi.md#drive_groups_one) | **GET** /file-storage/drive-groups/{id} | Get DriveGroup |

_FileStorageApi_ | [**drive_groups_update**](docs/apis/FileStorageApi.md#drive_groups_update) | **PATCH** /file-storage/drive-groups/{id} | Update DriveGroup |

_FileStorageApi_ | [**drives_add**](docs/apis/FileStorageApi.md#drives_add) | **POST** /file-storage/drives | Create Drive |

_FileStorageApi_ | [**drives_all**](docs/apis/FileStorageApi.md#drives_all) | **GET** /file-storage/drives | List Drives |

_FileStorageApi_ | [**drives_delete**](docs/apis/FileStorageApi.md#drives_delete) | **DELETE** /file-storage/drives/{id} | Delete Drive |

_FileStorageApi_ | [**drives_one**](docs/apis/FileStorageApi.md#drives_one) | **GET** /file-storage/drives/{id} | Get Drive |

_FileStorageApi_ | [**drives_update**](docs/apis/FileStorageApi.md#drives_update) | **PATCH** /file-storage/drives/{id} | Update Drive |

_FileStorageApi_ | [**files_all**](docs/apis/FileStorageApi.md#files_all) | **GET** /file-storage/files | List Files |

_FileStorageApi_ | [**files_delete**](docs/apis/FileStorageApi.md#files_delete) | **DELETE** /file-storage/files/{id} | Delete File |

_FileStorageApi_ | [**files_download**](docs/apis/FileStorageApi.md#files_download) | **GET** /file-storage/files/{id}/download | Download File |

_FileStorageApi_ | [**files_one**](docs/apis/FileStorageApi.md#files_one) | **GET** /file-storage/files/{id} | Get File |

_FileStorageApi_ | [**files_search**](docs/apis/FileStorageApi.md#files_search) | **POST** /file-storage/files/search | Search Files |

_FileStorageApi_ | [**folders_add**](docs/apis/FileStorageApi.md#folders_add) | **POST** /file-storage/folders | Create Folder |

_FileStorageApi_ | [**folders_copy**](docs/apis/FileStorageApi.md#folders_copy) | **POST** /file-storage/folders/{id}/copy | Copy Folder |

_FileStorageApi_ | [**folders_delete**](docs/apis/FileStorageApi.md#folders_delete) | **DELETE** /file-storage/folders/{id} | Delete Folder |

_FileStorageApi_ | [**folders_one**](docs/apis/FileStorageApi.md#folders_one) | **GET** /file-storage/folders/{id} | Get Folder |

_FileStorageApi_ | [**folders_update**](docs/apis/FileStorageApi.md#folders_update) | **PATCH** /file-storage/folders/{id} | Rename or move Folder |

_FileStorageApi_ | [**shared_links_add**](docs/apis/FileStorageApi.md#shared_links_add) | **POST** /file-storage/shared-links | Create Shared Link |

_FileStorageApi_ | [**shared_links_all**](docs/apis/FileStorageApi.md#shared_links_all) | **GET** /file-storage/shared-links | List SharedLinks |

_FileStorageApi_ | [**shared_links_delete**](docs/apis/FileStorageApi.md#shared_links_delete) | **DELETE** /file-storage/shared-links/{id} | Delete Shared Link |

_FileStorageApi_ | [**shared_links_one**](docs/apis/FileStorageApi.md#shared_links_one) | **GET** /file-storage/shared-links/{id} | Get Shared Link |

_FileStorageApi_ | [**shared_links_update**](docs/apis/FileStorageApi.md#shared_links_update) | **PATCH** /file-storage/shared-links/{id} | Update Shared Link |

_FileStorageApi_ | [**upload_sessions_add**](docs/apis/FileStorageApi.md#upload_sessions_add) | **POST** /file-storage/upload-sessions | Start Upload Session |

_FileStorageApi_ | [**upload_sessions_delete**](docs/apis/FileStorageApi.md#upload_sessions_delete) | **DELETE** /file-storage/upload-sessions/{id} | Abort Upload Session |

_FileStorageApi_ | [**upload_sessions_finish**](docs/apis/FileStorageApi.md#upload_sessions_finish) | **POST** /file-storage/upload-sessions/{id}/finish | Finish Upload Session |

_FileStorageApi_ | [**upload_sessions_one**](docs/apis/FileStorageApi.md#upload_sessions_one) | **GET** /file-storage/upload-sessions/{id} | Get Upload Session |

_HrisApi_ | [**companies_add**](docs/apis/HrisApi.md#companies_add) | **POST** /hris/companies | Create Company |

_HrisApi_ | [**companies_all**](docs/apis/HrisApi.md#companies_all) | **GET** /hris/companies | List Companies |

_HrisApi_ | [**companies_delete**](docs/apis/HrisApi.md#companies_delete) | **DELETE** /hris/companies/{id} | Delete Company |

_HrisApi_ | [**companies_one**](docs/apis/HrisApi.md#companies_one) | **GET** /hris/companies/{id} | Get Company |

_HrisApi_ | [**companies_update**](docs/apis/HrisApi.md#companies_update) | **PATCH** /hris/companies/{id} | Update Company |

_HrisApi_ | [**departments_add**](docs/apis/HrisApi.md#departments_add) | **POST** /hris/departments | Create Department |

_HrisApi_ | [**departments_all**](docs/apis/HrisApi.md#departments_all) | **GET** /hris/departments | List Departments |

_HrisApi_ | [**departments_delete**](docs/apis/HrisApi.md#departments_delete) | **DELETE** /hris/departments/{id} | Delete Department |

_HrisApi_ | [**departments_one**](docs/apis/HrisApi.md#departments_one) | **GET** /hris/departments/{id} | Get Department |

_HrisApi_ | [**departments_update**](docs/apis/HrisApi.md#departments_update) | **PATCH** /hris/departments/{id} | Update Department |

_HrisApi_ | [**employee_payrolls_all**](docs/apis/HrisApi.md#employee_payrolls_all) | **GET** /hris/payrolls/employees/{employee_id} | List Employee Payrolls |

_HrisApi_ | [**employee_payrolls_one**](docs/apis/HrisApi.md#employee_payrolls_one) | **GET** /hris/payrolls/employees/{employee_id}/payrolls/{payroll_id} | Get Employee Payroll |

_HrisApi_ | [**employee_schedules_all**](docs/apis/HrisApi.md#employee_schedules_all) | **GET** /hris/schedules/employees/{employee_id} | List Employee Schedules |

_HrisApi_ | [**employees_add**](docs/apis/HrisApi.md#employees_add) | **POST** /hris/employees | Create Employee |

_HrisApi_ | [**employees_all**](docs/apis/HrisApi.md#employees_all) | **GET** /hris/employees | List Employees |

_HrisApi_ | [**employees_delete**](docs/apis/HrisApi.md#employees_delete) | **DELETE** /hris/employees/{id} | Delete Employee |

_HrisApi_ | [**employees_one**](docs/apis/HrisApi.md#employees_one) | **GET** /hris/employees/{id} | Get Employee |

_HrisApi_ | [**employees_update**](docs/apis/HrisApi.md#employees_update) | **PATCH** /hris/employees/{id} | Update Employee |

_HrisApi_ | [**jobs_all**](docs/apis/HrisApi.md#jobs_all) | **GET** /hris/jobs/employees/{employee_id} | List Jobs |

_HrisApi_ | [**jobs_one**](docs/apis/HrisApi.md#jobs_one) | **GET** /hris/jobs/employees/{employee_id}/jobs/{job_id} | One Job |

_HrisApi_ | [**payrolls_all**](docs/apis/HrisApi.md#payrolls_all) | **GET** /hris/payrolls | List Payroll |

_HrisApi_ | [**payrolls_one**](docs/apis/HrisApi.md#payrolls_one) | **GET** /hris/payrolls/{payroll_id} | Get Payroll |

_HrisApi_ | [**time_off_requests_add**](docs/apis/HrisApi.md#time_off_requests_add) | **POST** /hris/time-off-requests | Create Time Off Request |

_HrisApi_ | [**time_off_requests_all**](docs/apis/HrisApi.md#time_off_requests_all) | **GET** /hris/time-off-requests | List Time Off Requests |

_HrisApi_ | [**time_off_requests_delete**](docs/apis/HrisApi.md#time_off_requests_delete) | **DELETE** /hris/time-off-requests/{id} | Delete Time Off Request |

_HrisApi_ | [**time_off_requests_one**](docs/apis/HrisApi.md#time_off_requests_one) | **GET** /hris/time-off-requests/{id} | Get Time Off Request |

_HrisApi_ | [**time_off_requests_update**](docs/apis/HrisApi.md#time_off_requests_update) | **PATCH** /hris/time-off-requests/{id} | Update Time Off Request |

_LeadApi_ | [**leads_add**](docs/apis/LeadApi.md#leads_add) | **POST** /lead/leads | Create lead |

_LeadApi_ | [**leads_all**](docs/apis/LeadApi.md#leads_all) | **GET** /lead/leads | List leads |

_LeadApi_ | [**leads_delete**](docs/apis/LeadApi.md#leads_delete) | **DELETE** /lead/leads/{id} | Delete lead |

_LeadApi_ | [**leads_one**](docs/apis/LeadApi.md#leads_one) | **GET** /lead/leads/{id} | Get lead |

_LeadApi_ | [**leads_update**](docs/apis/LeadApi.md#leads_update) | **PATCH** /lead/leads/{id} | Update lead |

_PosApi_ | [**items_add**](docs/apis/PosApi.md#items_add) | **POST** /pos/items | Create Item |

_PosApi_ | [**items_all**](docs/apis/PosApi.md#items_all) | **GET** /pos/items | List Items |

_PosApi_ | [**items_delete**](docs/apis/PosApi.md#items_delete) | **DELETE** /pos/items/{id} | Delete Item |

_PosApi_ | [**items_one**](docs/apis/PosApi.md#items_one) | **GET** /pos/items/{id} | Get Item |

_PosApi_ | [**items_update**](docs/apis/PosApi.md#items_update) | **PATCH** /pos/items/{id} | Update Item |

_PosApi_ | [**locations_add**](docs/apis/PosApi.md#locations_add) | **POST** /pos/locations | Create Location |

_PosApi_ | [**locations_all**](docs/apis/PosApi.md#locations_all) | **GET** /pos/locations | List Locations |

_PosApi_ | [**locations_delete**](docs/apis/PosApi.md#locations_delete) | **DELETE** /pos/locations/{id} | Delete Location |

_PosApi_ | [**locations_one**](docs/apis/PosApi.md#locations_one) | **GET** /pos/locations/{id} | Get Location |

_PosApi_ | [**locations_update**](docs/apis/PosApi.md#locations_update) | **PATCH** /pos/locations/{id} | Update Location |

_PosApi_ | [**merchants_add**](docs/apis/PosApi.md#merchants_add) | **POST** /pos/merchants | Create Merchant |

_PosApi_ | [**merchants_all**](docs/apis/PosApi.md#merchants_all) | **GET** /pos/merchants | List Merchants |

_PosApi_ | [**merchants_delete**](docs/apis/PosApi.md#merchants_delete) | **DELETE** /pos/merchants/{id} | Delete Merchant |

_PosApi_ | [**merchants_one**](docs/apis/PosApi.md#merchants_one) | **GET** /pos/merchants/{id} | Get Merchant |

_PosApi_ | [**merchants_update**](docs/apis/PosApi.md#merchants_update) | **PATCH** /pos/merchants/{id} | Update Merchant |

_PosApi_ | [**modifier_groups_add**](docs/apis/PosApi.md#modifier_groups_add) | **POST** /pos/modifier-groups | Create Modifier Group |

_PosApi_ | [**modifier_groups_all**](docs/apis/PosApi.md#modifier_groups_all) | **GET** /pos/modifier-groups | List Modifier Groups |

_PosApi_ | [**modifier_groups_delete**](docs/apis/PosApi.md#modifier_groups_delete) | **DELETE** /pos/modifier-groups/{id} | Delete Modifier Group |

_PosApi_ | [**modifier_groups_one**](docs/apis/PosApi.md#modifier_groups_one) | **GET** /pos/modifier-groups/{id} | Get Modifier Group |

_PosApi_ | [**modifier_groups_update**](docs/apis/PosApi.md#modifier_groups_update) | **PATCH** /pos/modifier-groups/{id} | Update Modifier Group |

_PosApi_ | [**modifiers_add**](docs/apis/PosApi.md#modifiers_add) | **POST** /pos/modifiers | Create Modifier |

_PosApi_ | [**modifiers_all**](docs/apis/PosApi.md#modifiers_all) | **GET** /pos/modifiers | List Modifiers |

_PosApi_ | [**modifiers_delete**](docs/apis/PosApi.md#modifiers_delete) | **DELETE** /pos/modifiers/{id} | Delete Modifier |

_PosApi_ | [**modifiers_one**](docs/apis/PosApi.md#modifiers_one) | **GET** /pos/modifiers/{id} | Get Modifier |

_PosApi_ | [**modifiers_update**](docs/apis/PosApi.md#modifiers_update) | **PATCH** /pos/modifiers/{id} | Update Modifier |

_PosApi_ | [**order_types_add**](docs/apis/PosApi.md#order_types_add) | **POST** /pos/order-types | Create Order Type |

_PosApi_ | [**order_types_all**](docs/apis/PosApi.md#order_types_all) | **GET** /pos/order-types | List Order Types |

_PosApi_ | [**order_types_delete**](docs/apis/PosApi.md#order_types_delete) | **DELETE** /pos/order-types/{id} | Delete Order Type |

_PosApi_ | [**order_types_one**](docs/apis/PosApi.md#order_types_one) | **GET** /pos/order-types/{id} | Get Order Type |

_PosApi_ | [**order_types_update**](docs/apis/PosApi.md#order_types_update) | **PATCH** /pos/order-types/{id} | Update Order Type |

_PosApi_ | [**orders_add**](docs/apis/PosApi.md#orders_add) | **POST** /pos/orders | Create Order |

_PosApi_ | [**orders_all**](docs/apis/PosApi.md#orders_all) | **GET** /pos/orders | List Orders |

_PosApi_ | [**orders_delete**](docs/apis/PosApi.md#orders_delete) | **DELETE** /pos/orders/{id} | Delete Order |

_PosApi_ | [**orders_one**](docs/apis/PosApi.md#orders_one) | **GET** /pos/orders/{id} | Get Order |

_PosApi_ | [**orders_pay**](docs/apis/PosApi.md#orders_pay) | **POST** /pos/orders/{id}/pay | Pay Order |

_PosApi_ | [**orders_update**](docs/apis/PosApi.md#orders_update) | **PATCH** /pos/orders/{id} | Update Order |

_PosApi_ | [**payments_add**](docs/apis/PosApi.md#payments_add) | **POST** /pos/payments | Create Payment |

_PosApi_ | [**payments_all**](docs/apis/PosApi.md#payments_all) | **GET** /pos/payments | List Payments |

_PosApi_ | [**payments_delete**](docs/apis/PosApi.md#payments_delete) | **DELETE** /pos/payments/{id} | Delete Payment |

_PosApi_ | [**payments_one**](docs/apis/PosApi.md#payments_one) | **GET** /pos/payments/{id} | Get Payment |

_PosApi_ | [**payments_update**](docs/apis/PosApi.md#payments_update) | **PATCH** /pos/payments/{id} | Update Payment |

_PosApi_ | [**tenders_add**](docs/apis/PosApi.md#tenders_add) | **POST** /pos/tenders | Create Tender |

_PosApi_ | [**tenders_all**](docs/apis/PosApi.md#tenders_all) | **GET** /pos/tenders | List Tenders |

_PosApi_ | [**tenders_delete**](docs/apis/PosApi.md#tenders_delete) | **DELETE** /pos/tenders/{id} | Delete Tender |

_PosApi_ | [**tenders_one**](docs/apis/PosApi.md#tenders_one) | **GET** /pos/tenders/{id} | Get Tender |

_PosApi_ | [**tenders_update**](docs/apis/PosApi.md#tenders_update) | **PATCH** /pos/tenders/{id} | Update Tender |

_SmsApi_ | [**messages_add**](docs/apis/SmsApi.md#messages_add) | **POST** /sms/messages | Create Message |

_SmsApi_ | [**messages_all**](docs/apis/SmsApi.md#messages_all) | **GET** /sms/messages | List Messages |

_SmsApi_ | [**messages_delete**](docs/apis/SmsApi.md#messages_delete) | **DELETE** /sms/messages/{id} | Delete Message |

_SmsApi_ | [**messages_one**](docs/apis/SmsApi.md#messages_one) | **GET** /sms/messages/{id} | Get Message |

_SmsApi_ | [**messages_update**](docs/apis/SmsApi.md#messages_update) | **PATCH** /sms/messages/{id} | Update Message |

_VaultApi_ | [**connection_settings_all**](docs/apis/VaultApi.md#connection_settings_all) | **GET** /vault/connections/{unified_api}/{service_id}/{resource}/config | Get resource settings |

_VaultApi_ | [**connection_settings_update**](docs/apis/VaultApi.md#connection_settings_update) | **PATCH** /vault/connections/{unified_api}/{service_id}/{resource}/config | Update settings |

_VaultApi_ | [**connections_all**](docs/apis/VaultApi.md#connections_all) | **GET** /vault/connections | Get all connections |

_VaultApi_ | [**connections_delete**](docs/apis/VaultApi.md#connections_delete) | **DELETE** /vault/connections/{unified_api}/{service_id} | Deletes a connection |

_VaultApi_ | [**connections_import**](docs/apis/VaultApi.md#connections_import) | **POST** /vault/connections/{unified_api}/{service_id}/import | Import connection |

_VaultApi_ | [**connections_one**](docs/apis/VaultApi.md#connections_one) | **GET** /vault/connections/{unified_api}/{service_id} | Get connection |

_VaultApi_ | [**connections_update**](docs/apis/VaultApi.md#connections_update) | **PATCH** /vault/connections/{unified_api}/{service_id} | Update connection |

_VaultApi_ | [**consumer_request_counts_all**](docs/apis/VaultApi.md#consumer_request_counts_all) | **GET** /vault/consumers/{consumer_id}/stats | Consumer request counts |

_VaultApi_ | [**consumers_all**](docs/apis/VaultApi.md#consumers_all) | **GET** /vault/consumers | Get all consumers |

_VaultApi_ | [**consumers_one**](docs/apis/VaultApi.md#consumers_one) | **GET** /vault/consumers/{consumer_id} | Get consumer |

_VaultApi_ | [**logs_all**](docs/apis/VaultApi.md#logs_all) | **GET** /vault/logs | Get all consumer request logs |

_VaultApi_ | [**sessions_create**](docs/apis/VaultApi.md#sessions_create) | **POST** /vault/sessions | Create Session |

_WebhookApi_ | [**event_logs_all**](docs/apis/WebhookApi.md#event_logs_all) | **GET** /webhook/logs | List event logs |

_WebhookApi_ | [**webhooks_add**](docs/apis/WebhookApi.md#webhooks_add) | **POST** /webhook/webhooks | Create webhook subscription |

_WebhookApi_ | [**webhooks_all**](docs/apis/WebhookApi.md#webhooks_all) | **GET** /webhook/webhooks | List webhook subscriptions |

_WebhookApi_ | [**webhooks_delete**](docs/apis/WebhookApi.md#webhooks_delete) | **DELETE** /webhook/webhooks/{id} | Delete webhook subscription |

_WebhookApi_ | [**webhooks_one**](docs/apis/WebhookApi.md#webhooks_one) | **GET** /webhook/webhooks/{id} | Get webhook subscription |

_WebhookApi_ | [**webhooks_update**](docs/apis/WebhookApi.md#webhooks_update) | **PATCH** /webhook/webhooks/{id} | Update webhook subscription |



## Documentation For Models

 - [AccountingCustomer](docs/models/AccountingCustomer.md)
 - [AccountingEventType](docs/models/AccountingEventType.md)
 - [Activity](docs/models/Activity.md)
 - [ActivityAttendee](docs/models/ActivityAttendee.md)
 - [Address](docs/models/Address.md)
 - [Api](docs/models/Api.md)
 - [ApiResource](docs/models/ApiResource.md)
 - [ApiResourceCoverage](docs/models/ApiResourceCoverage.md)
 - [ApiResourceCoverageCoverage](docs/models/ApiResourceCoverageCoverage.md)
 - [ApiResourceLinkedResources](docs/models/ApiResourceLinkedResources.md)
 - [ApiResources](docs/models/ApiResources.md)
 - [ApiStatus](docs/models/ApiStatus.md)
 - [ApisFilter](docs/models/ApisFilter.md)
 - [Applicant](docs/models/Applicant.md)
 - [ApplicantSocialLinks](docs/models/ApplicantSocialLinks.md)
 - [ApplicantWebsites](docs/models/ApplicantWebsites.md)
 - [ApplicantsFilter](docs/models/ApplicantsFilter.md)
 - [AtsActivity](docs/models/AtsActivity.md)
 - [AtsEventType](docs/models/AtsEventType.md)
 - [AuthType](docs/models/AuthType.md)
 - [BadRequestResponse](docs/models/BadRequestResponse.md)
 - [BalanceSheet](docs/models/BalanceSheet.md)
 - [BalanceSheetAssets](docs/models/BalanceSheetAssets.md)
 - [BalanceSheetAssetsCurrentAssets](docs/models/BalanceSheetAssetsCurrentAssets.md)
 - [BalanceSheetAssetsCurrentAssetsAccounts](docs/models/BalanceSheetAssetsCurrentAssetsAccounts.md)
 - [BalanceSheetAssetsFixedAssets](docs/models/BalanceSheetAssetsFixedAssets.md)
 - [BalanceSheetAssetsFixedAssetsAccounts](docs/models/BalanceSheetAssetsFixedAssetsAccounts.md)
 - [BalanceSheetEquity](docs/models/BalanceSheetEquity.md)
 - [BalanceSheetEquityItems](docs/models/BalanceSheetEquityItems.md)
 - [BalanceSheetFilter](docs/models/BalanceSheetFilter.md)
 - [BalanceSheetLiabilities](docs/models/BalanceSheetLiabilities.md)
 - [BalanceSheetLiabilitiesAccounts](docs/models/BalanceSheetLiabilitiesAccounts.md)
 - [BankAccount](docs/models/BankAccount.md)
 - [Benefit](docs/models/Benefit.md)
 - [Bill](docs/models/Bill.md)
 - [BillLineItem](docs/models/BillLineItem.md)
 - [Branch](docs/models/Branch.md)
 - [CashDetails](docs/models/CashDetails.md)
 - [CompaniesFilter](docs/models/CompaniesFilter.md)
 - [CompaniesSort](docs/models/CompaniesSort.md)
 - [Company](docs/models/Company.md)
 - [CompanyInfo](docs/models/CompanyInfo.md)
 - [CompanyRowType](docs/models/CompanyRowType.md)
 - [Compensation](docs/models/Compensation.md)
 - [Connection](docs/models/Connection.md)
 - [ConnectionConfiguration](docs/models/ConnectionConfiguration.md)
 - [ConnectionDefaults](docs/models/ConnectionDefaults.md)
 - [ConnectionImportData](docs/models/ConnectionImportData.md)
 - [ConnectionImportDataCredentials](docs/models/ConnectionImportDataCredentials.md)
 - [ConnectionMetadata](docs/models/ConnectionMetadata.md)
 - [ConnectionState](docs/models/ConnectionState.md)
 - [ConnectionWebhook](docs/models/ConnectionWebhook.md)
 - [Connector](docs/models/Connector.md)
 - [ConnectorDoc](docs/models/ConnectorDoc.md)
 - [ConnectorEvent](docs/models/ConnectorEvent.md)
 - [ConnectorOauthScopes](docs/models/ConnectorOauthScopes.md)
 - [ConnectorOauthScopes1](docs/models/ConnectorOauthScopes1.md)
 - [ConnectorResource](docs/models/ConnectorResource.md)
 - [ConnectorSetting](docs/models/ConnectorSetting.md)
 - [ConnectorStatus](docs/models/ConnectorStatus.md)
 - [ConnectorTlsSupport](docs/models/ConnectorTlsSupport.md)
 - [ConnectorUnifiedApis](docs/models/ConnectorUnifiedApis.md)
 - [ConnectorsFilter](docs/models/ConnectorsFilter.md)
 - [Consumer](docs/models/Consumer.md)
 - [ConsumerConnection](docs/models/ConsumerConnection.md)
 - [ConsumerMetadata](docs/models/ConsumerMetadata.md)
 - [ConsumerRequestCountsInDateRangeResponse](docs/models/ConsumerRequestCountsInDateRangeResponse.md)
 - [ConsumerRequestCountsInDateRangeResponseData](docs/models/ConsumerRequestCountsInDateRangeResponseData.md)
 - [Contact](docs/models/Contact.md)
 - [ContactsFilter](docs/models/ContactsFilter.md)
 - [ContactsSort](docs/models/ContactsSort.md)
 - [CopyFolderRequest](docs/models/CopyFolderRequest.md)
 - [CreateActivityResponse](docs/models/CreateActivityResponse.md)
 - [CreateApplicantResponse](docs/models/CreateApplicantResponse.md)
 - [CreateBillResponse](docs/models/CreateBillResponse.md)
 - [CreateCompanyResponse](docs/models/CreateCompanyResponse.md)
 - [CreateConnectionResponse](docs/models/CreateConnectionResponse.md)
 - [CreateContactResponse](docs/models/CreateContactResponse.md)
 - [CreateCreditNoteResponse](docs/models/CreateCreditNoteResponse.md)
 - [CreateCustomerResponse](docs/models/CreateCustomerResponse.md)
 - [CreateCustomerSupportCustomerResponse](docs/models/CreateCustomerSupportCustomerResponse.md)
 - [CreateDepartmentResponse](docs/models/CreateDepartmentResponse.md)
 - [CreateDriveGroupResponse](docs/models/CreateDriveGroupResponse.md)
 - [CreateDriveResponse](docs/models/CreateDriveResponse.md)
 - [CreateEmployeeResponse](docs/models/CreateEmployeeResponse.md)
 - [CreateFileRequest](docs/models/CreateFileRequest.md)
 - [CreateFileResponse](docs/models/CreateFileResponse.md)
 - [CreateFolderRequest](docs/models/CreateFolderRequest.md)
 - [CreateFolderResponse](docs/models/CreateFolderResponse.md)
 - [CreateHrisCompanyResponse](docs/models/CreateHrisCompanyResponse.md)
 - [CreateInvoiceItemResponse](docs/models/CreateInvoiceItemResponse.md)
 - [CreateInvoiceResponse](docs/models/CreateInvoiceResponse.md)
 - [CreateItemResponse](docs/models/CreateItemResponse.md)
 - [CreateJobResponse](docs/models/CreateJobResponse.md)
 - [CreateLeadResponse](docs/models/CreateLeadResponse.md)
 - [CreateLedgerAccountResponse](docs/models/CreateLedgerAccountResponse.md)
 - [CreateLocationResponse](docs/models/CreateLocationResponse.md)
 - [CreateMerchantResponse](docs/models/CreateMerchantResponse.md)
 - [CreateMessageResponse](docs/models/CreateMessageResponse.md)
 - [CreateModifierGroupResponse](docs/models/CreateModifierGroupResponse.md)
 - [CreateModifierResponse](docs/models/CreateModifierResponse.md)
 - [CreateNoteResponse](docs/models/CreateNoteResponse.md)
 - [CreateOpportunityResponse](docs/models/CreateOpportunityResponse.md)
 - [CreateOrderResponse](docs/models/CreateOrderResponse.md)
 - [CreateOrderTypeResponse](docs/models/CreateOrderTypeResponse.md)
 - [CreatePaymentResponse](docs/models/CreatePaymentResponse.md)
 - [CreatePipelineResponse](docs/models/CreatePipelineResponse.md)
 - [CreatePosPaymentResponse](docs/models/CreatePosPaymentResponse.md)
 - [CreateSessionResponse](docs/models/CreateSessionResponse.md)
 - [CreateSessionResponseData](docs/models/CreateSessionResponseData.md)
 - [CreateSharedLinkResponse](docs/models/CreateSharedLinkResponse.md)
 - [CreateSupplierResponse](docs/models/CreateSupplierResponse.md)
 - [CreateTaxRateResponse](docs/models/CreateTaxRateResponse.md)
 - [CreateTenderResponse](docs/models/CreateTenderResponse.md)
 - [CreateTimeOffRequestResponse](docs/models/CreateTimeOffRequestResponse.md)
 - [CreateUploadSessionRequest](docs/models/CreateUploadSessionRequest.md)
 - [CreateUploadSessionResponse](docs/models/CreateUploadSessionResponse.md)
 - [CreateUserResponse](docs/models/CreateUserResponse.md)
 - [CreateWebhookRequest](docs/models/CreateWebhookRequest.md)
 - [CreateWebhookResponse](docs/models/CreateWebhookResponse.md)
 - [CreditNote](docs/models/CreditNote.md)
 - [CrmEventType](docs/models/CrmEventType.md)
 - [Currency](docs/models/Currency.md)
 - [CustomField](docs/models/CustomField.md)
 - [CustomerSupportCustomer](docs/models/CustomerSupportCustomer.md)
 - [CustomersFilter](docs/models/CustomersFilter.md)
 - [Deduction](docs/models/Deduction.md)
 - [DeleteActivityResponse](docs/models/DeleteActivityResponse.md)
 - [DeleteBillResponse](docs/models/DeleteBillResponse.md)
 - [DeleteCompanyResponse](docs/models/DeleteCompanyResponse.md)
 - [DeleteContactResponse](docs/models/DeleteContactResponse.md)
 - [DeleteCreditNoteResponse](docs/models/DeleteCreditNoteResponse.md)
 - [DeleteCustomerResponse](docs/models/DeleteCustomerResponse.md)
 - [DeleteCustomerSupportCustomerResponse](docs/models/DeleteCustomerSupportCustomerResponse.md)
 - [DeleteDepartmentResponse](docs/models/DeleteDepartmentResponse.md)
 - [DeleteDriveGroupResponse](docs/models/DeleteDriveGroupResponse.md)
 - [DeleteDriveResponse](docs/models/DeleteDriveResponse.md)
 - [DeleteEmployeeResponse](docs/models/DeleteEmployeeResponse.md)
 - [DeleteFileResponse](docs/models/DeleteFileResponse.md)
 - [DeleteFolderResponse](docs/models/DeleteFolderResponse.md)
 - [DeleteHrisCompanyResponse](docs/models/DeleteHrisCompanyResponse.md)
 - [DeleteInvoiceItemResponse](docs/models/DeleteInvoiceItemResponse.md)
 - [DeleteInvoiceResponse](docs/models/DeleteInvoiceResponse.md)
 - [DeleteItemResponse](docs/models/DeleteItemResponse.md)
 - [DeleteJobResponse](docs/models/DeleteJobResponse.md)
 - [DeleteLeadResponse](docs/models/DeleteLeadResponse.md)
 - [DeleteLedgerAccountResponse](docs/models/DeleteLedgerAccountResponse.md)
 - [DeleteLocationResponse](docs/models/DeleteLocationResponse.md)
 - [DeleteMerchantResponse](docs/models/DeleteMerchantResponse.md)
 - [DeleteMessageResponse](docs/models/DeleteMessageResponse.md)
 - [DeleteModifierGroupResponse](docs/models/DeleteModifierGroupResponse.md)
 - [DeleteModifierResponse](docs/models/DeleteModifierResponse.md)
 - [DeleteNoteResponse](docs/models/DeleteNoteResponse.md)
 - [DeleteOpportunityResponse](docs/models/DeleteOpportunityResponse.md)
 - [DeleteOrderResponse](docs/models/DeleteOrderResponse.md)
 - [DeleteOrderTypeResponse](docs/models/DeleteOrderTypeResponse.md)
 - [DeletePaymentResponse](docs/models/DeletePaymentResponse.md)
 - [DeletePipelineResponse](docs/models/DeletePipelineResponse.md)
 - [DeletePosPaymentResponse](docs/models/DeletePosPaymentResponse.md)
 - [DeleteSharedLinkResponse](docs/models/DeleteSharedLinkResponse.md)
 - [DeleteSupplierResponse](docs/models/DeleteSupplierResponse.md)
 - [DeleteTaxRateResponse](docs/models/DeleteTaxRateResponse.md)
 - [DeleteTenderResponse](docs/models/DeleteTenderResponse.md)
 - [DeleteTimeOffRequestResponse](docs/models/DeleteTimeOffRequestResponse.md)
 - [DeleteUploadSessionResponse](docs/models/DeleteUploadSessionResponse.md)
 - [DeleteUserResponse](docs/models/DeleteUserResponse.md)
 - [DeleteWebhookResponse](docs/models/DeleteWebhookResponse.md)
 - [DeliveryUrl](docs/models/DeliveryUrl.md)
 - [Department](docs/models/Department.md)
 - [Drive](docs/models/Drive.md)
 - [DriveGroup](docs/models/DriveGroup.md)
 - [DriveGroupsFilter](docs/models/DriveGroupsFilter.md)
 - [DrivesFilter](docs/models/DrivesFilter.md)
 - [Email](docs/models/Email.md)
 - [Employee](docs/models/Employee.md)
 - [EmployeeCompensations](docs/models/EmployeeCompensations.md)
 - [EmployeeEmploymentRole](docs/models/EmployeeEmploymentRole.md)
 - [EmployeeJobs](docs/models/EmployeeJobs.md)
 - [EmployeeManager](docs/models/EmployeeManager.md)
 - [EmployeePartner](docs/models/EmployeePartner.md)
 - [EmployeePayroll](docs/models/EmployeePayroll.md)
 - [EmployeePayrolls](docs/models/EmployeePayrolls.md)
 - [EmployeeSchedules](docs/models/EmployeeSchedules.md)
 - [EmployeeTeam](docs/models/EmployeeTeam.md)
 - [EmployeesFilter](docs/models/EmployeesFilter.md)
 - [Error](docs/models/Error.md)
 - [ExecuteBaseUrl](docs/models/ExecuteBaseUrl.md)
 - [ExecuteWebhookEventRequest](docs/models/ExecuteWebhookEventRequest.md)
 - [ExecuteWebhookEventsRequest](docs/models/ExecuteWebhookEventsRequest.md)
 - [ExecuteWebhookResponse](docs/models/ExecuteWebhookResponse.md)
 - [FileStorageEventType](docs/models/FileStorageEventType.md)
 - [FileType](docs/models/FileType.md)
 - [FilesFilter](docs/models/FilesFilter.md)
 - [FilesSearch](docs/models/FilesSearch.md)
 - [FilesSort](docs/models/FilesSort.md)
 - [Folder](docs/models/Folder.md)
 - [FormField](docs/models/FormField.md)
 - [FormFieldOption](docs/models/FormFieldOption.md)
 - [FormFieldOptionGroup](docs/models/FormFieldOptionGroup.md)
 - [Gender](docs/models/Gender.md)
 - [GetActivitiesResponse](docs/models/GetActivitiesResponse.md)
 - [GetActivityResponse](docs/models/GetActivityResponse.md)
 - [GetApiResourceCoverageResponse](docs/models/GetApiResourceCoverageResponse.md)
 - [GetApiResourceResponse](docs/models/GetApiResourceResponse.md)
 - [GetApiResponse](docs/models/GetApiResponse.md)
 - [GetApisResponse](docs/models/GetApisResponse.md)
 - [GetApplicantResponse](docs/models/GetApplicantResponse.md)
 - [GetApplicantsResponse](docs/models/GetApplicantsResponse.md)
 - [GetBalanceSheetResponse](docs/models/GetBalanceSheetResponse.md)
 - [GetBillResponse](docs/models/GetBillResponse.md)
 - [GetBillsResponse](docs/models/GetBillsResponse.md)
 - [GetCompaniesResponse](docs/models/GetCompaniesResponse.md)
 - [GetCompanyInfoResponse](docs/models/GetCompanyInfoResponse.md)
 - [GetCompanyResponse](docs/models/GetCompanyResponse.md)
 - [GetConnectionResponse](docs/models/GetConnectionResponse.md)
 - [GetConnectionsResponse](docs/models/GetConnectionsResponse.md)
 - [GetConnectorResourceResponse](docs/models/GetConnectorResourceResponse.md)
 - [GetConnectorResponse](docs/models/GetConnectorResponse.md)
 - [GetConnectorsResponse](docs/models/GetConnectorsResponse.md)
 - [GetConsumerResponse](docs/models/GetConsumerResponse.md)
 - [GetConsumersResponse](docs/models/GetConsumersResponse.md)
 - [GetConsumersResponseData](docs/models/GetConsumersResponseData.md)
 - [GetContactResponse](docs/models/GetContactResponse.md)
 - [GetContactsResponse](docs/models/GetContactsResponse.md)
 - [GetCreditNoteResponse](docs/models/GetCreditNoteResponse.md)
 - [GetCreditNotesResponse](docs/models/GetCreditNotesResponse.md)
 - [GetCustomerResponse](docs/models/GetCustomerResponse.md)
 - [GetCustomerSupportCustomerResponse](docs/models/GetCustomerSupportCustomerResponse.md)
 - [GetCustomerSupportCustomersResponse](docs/models/GetCustomerSupportCustomersResponse.md)
 - [GetCustomersResponse](docs/models/GetCustomersResponse.md)
 - [GetDepartmentResponse](docs/models/GetDepartmentResponse.md)
 - [GetDepartmentsResponse](docs/models/GetDepartmentsResponse.md)
 - [GetDriveGroupResponse](docs/models/GetDriveGroupResponse.md)
 - [GetDriveGroupsResponse](docs/models/GetDriveGroupsResponse.md)
 - [GetDriveResponse](docs/models/GetDriveResponse.md)
 - [GetDrivesResponse](docs/models/GetDrivesResponse.md)
 - [GetEmployeePayrollResponse](docs/models/GetEmployeePayrollResponse.md)
 - [GetEmployeePayrollsResponse](docs/models/GetEmployeePayrollsResponse.md)
 - [GetEmployeeResponse](docs/models/GetEmployeeResponse.md)
 - [GetEmployeeSchedulesResponse](docs/models/GetEmployeeSchedulesResponse.md)
 - [GetEmployeesResponse](docs/models/GetEmployeesResponse.md)
 - [GetFileResponse](docs/models/GetFileResponse.md)
 - [GetFilesResponse](docs/models/GetFilesResponse.md)
 - [GetFolderResponse](docs/models/GetFolderResponse.md)
 - [GetFoldersResponse](docs/models/GetFoldersResponse.md)
 - [GetHrisCompaniesResponse](docs/models/GetHrisCompaniesResponse.md)
 - [GetHrisCompanyResponse](docs/models/GetHrisCompanyResponse.md)
 - [GetHrisJobResponse](docs/models/GetHrisJobResponse.md)
 - [GetHrisJobsResponse](docs/models/GetHrisJobsResponse.md)
 - [GetInvoiceItemResponse](docs/models/GetInvoiceItemResponse.md)
 - [GetInvoiceItemsResponse](docs/models/GetInvoiceItemsResponse.md)
 - [GetInvoiceResponse](docs/models/GetInvoiceResponse.md)
 - [GetInvoicesResponse](docs/models/GetInvoicesResponse.md)
 - [GetItemResponse](docs/models/GetItemResponse.md)
 - [GetItemsResponse](docs/models/GetItemsResponse.md)
 - [GetJobResponse](docs/models/GetJobResponse.md)
 - [GetJobsResponse](docs/models/GetJobsResponse.md)
 - [GetLeadResponse](docs/models/GetLeadResponse.md)
 - [GetLeadsResponse](docs/models/GetLeadsResponse.md)
 - [GetLedgerAccountResponse](docs/models/GetLedgerAccountResponse.md)
 - [GetLedgerAccountsResponse](docs/models/GetLedgerAccountsResponse.md)
 - [GetLocationResponse](docs/models/GetLocationResponse.md)
 - [GetLocationsResponse](docs/models/GetLocationsResponse.md)
 - [GetLogsResponse](docs/models/GetLogsResponse.md)
 - [GetMerchantResponse](docs/models/GetMerchantResponse.md)
 - [GetMerchantsResponse](docs/models/GetMerchantsResponse.md)
 - [GetMessageResponse](docs/models/GetMessageResponse.md)
 - [GetMessagesResponse](docs/models/GetMessagesResponse.md)
 - [GetModifierGroupResponse](docs/models/GetModifierGroupResponse.md)
 - [GetModifierGroupsResponse](docs/models/GetModifierGroupsResponse.md)
 - [GetModifierResponse](docs/models/GetModifierResponse.md)
 - [GetModifiersResponse](docs/models/GetModifiersResponse.md)
 - [GetNoteResponse](docs/models/GetNoteResponse.md)
 - [GetNotesResponse](docs/models/GetNotesResponse.md)
 - [GetOpportunitiesResponse](docs/models/GetOpportunitiesResponse.md)
 - [GetOpportunityResponse](docs/models/GetOpportunityResponse.md)
 - [GetOrderResponse](docs/models/GetOrderResponse.md)
 - [GetOrderTypeResponse](docs/models/GetOrderTypeResponse.md)
 - [GetOrderTypesResponse](docs/models/GetOrderTypesResponse.md)
 - [GetOrdersResponse](docs/models/GetOrdersResponse.md)
 - [GetPaymentResponse](docs/models/GetPaymentResponse.md)
 - [GetPaymentsResponse](docs/models/GetPaymentsResponse.md)
 - [GetPayrollResponse](docs/models/GetPayrollResponse.md)
 - [GetPayrollsResponse](docs/models/GetPayrollsResponse.md)
 - [GetPipelineResponse](docs/models/GetPipelineResponse.md)
 - [GetPipelinesResponse](docs/models/GetPipelinesResponse.md)
 - [GetPosPaymentResponse](docs/models/GetPosPaymentResponse.md)
 - [GetPosPaymentsResponse](docs/models/GetPosPaymentsResponse.md)
 - [GetProfitAndLossResponse](docs/models/GetProfitAndLossResponse.md)
 - [GetSharedLinkResponse](docs/models/GetSharedLinkResponse.md)
 - [GetSharedLinksResponse](docs/models/GetSharedLinksResponse.md)
 - [GetSupplierResponse](docs/models/GetSupplierResponse.md)
 - [GetSuppliersResponse](docs/models/GetSuppliersResponse.md)
 - [GetTaxRateResponse](docs/models/GetTaxRateResponse.md)
 - [GetTaxRatesResponse](docs/models/GetTaxRatesResponse.md)
 - [GetTenderResponse](docs/models/GetTenderResponse.md)
 - [GetTendersResponse](docs/models/GetTendersResponse.md)
 - [GetTimeOffRequestResponse](docs/models/GetTimeOffRequestResponse.md)
 - [GetTimeOffRequestsResponse](docs/models/GetTimeOffRequestsResponse.md)
 - [GetUploadSessionResponse](docs/models/GetUploadSessionResponse.md)
 - [GetUserResponse](docs/models/GetUserResponse.md)
 - [GetUsersResponse](docs/models/GetUsersResponse.md)
 - [GetWebhookEventLogsResponse](docs/models/GetWebhookEventLogsResponse.md)
 - [GetWebhookResponse](docs/models/GetWebhookResponse.md)
 - [GetWebhooksResponse](docs/models/GetWebhooksResponse.md)
 - [HrisCompany](docs/models/HrisCompany.md)
 - [HrisEventType](docs/models/HrisEventType.md)
 - [HrisJob](docs/models/HrisJob.md)
 - [HrisJobLocation](docs/models/HrisJobLocation.md)
 - [HrisJobs](docs/models/HrisJobs.md)
 - [IdempotencyKey](docs/models/IdempotencyKey.md)
 - [Invoice](docs/models/Invoice.md)
 - [InvoiceItem](docs/models/InvoiceItem.md)
 - [InvoiceItemAssetAccount](docs/models/InvoiceItemAssetAccount.md)
 - [InvoiceItemExpenseAccount](docs/models/InvoiceItemExpenseAccount.md)
 - [InvoiceItemIncomeAccount](docs/models/InvoiceItemIncomeAccount.md)
 - [InvoiceItemSalesDetails](docs/models/InvoiceItemSalesDetails.md)
 - [InvoiceItemsFilter](docs/models/InvoiceItemsFilter.md)
 - [InvoiceLineItem](docs/models/InvoiceLineItem.md)
 - [InvoiceResponse](docs/models/InvoiceResponse.md)
 - [InvoicesSort](docs/models/InvoicesSort.md)
 - [Item](docs/models/Item.md)
 - [Job](docs/models/Job.md)
 - [JobSalary](docs/models/JobSalary.md)
 - [JobStatus](docs/models/JobStatus.md)
 - [JobsFilter](docs/models/JobsFilter.md)
 - [Lead](docs/models/Lead.md)
 - [LeadEventType](docs/models/LeadEventType.md)
 - [LeadsFilter](docs/models/LeadsFilter.md)
 - [LeadsSort](docs/models/LeadsSort.md)
 - [LedgerAccount](docs/models/LedgerAccount.md)
 - [LedgerAccountCategories](docs/models/LedgerAccountCategories.md)
 - [LedgerAccountParentAccount](docs/models/LedgerAccountParentAccount.md)
 - [LedgerAccounts](docs/models/LedgerAccounts.md)
 - [LinkedConnectorResource](docs/models/LinkedConnectorResource.md)
 - [LinkedCustomer](docs/models/LinkedCustomer.md)
 - [LinkedFolder](docs/models/LinkedFolder.md)
 - [LinkedInvoiceItem](docs/models/LinkedInvoiceItem.md)
 - [LinkedLedgerAccount](docs/models/LinkedLedgerAccount.md)
 - [LinkedSupplier](docs/models/LinkedSupplier.md)
 - [LinkedTaxRate](docs/models/LinkedTaxRate.md)
 - [Links](docs/models/Links.md)
 - [Location](docs/models/Location.md)
 - [Log](docs/models/Log.md)
 - [LogOperation](docs/models/LogOperation.md)
 - [LogService](docs/models/LogService.md)
 - [LogsFilter](docs/models/LogsFilter.md)
 - [Merchant](docs/models/Merchant.md)
 - [Message](docs/models/Message.md)
 - [Meta](docs/models/Meta.md)
 - [MetaCursors](docs/models/MetaCursors.md)
 - [Modifier](docs/models/Modifier.md)
 - [ModifierGroup](docs/models/ModifierGroup.md)
 - [ModifierGroupFilter](docs/models/ModifierGroupFilter.md)
 - [NotFoundResponse](docs/models/NotFoundResponse.md)
 - [NotImplementedResponse](docs/models/NotImplementedResponse.md)
 - [Note](docs/models/Note.md)
 - [OAuthGrantType](docs/models/OAuthGrantType.md)
 - [Offer](docs/models/Offer.md)
 - [OpportunitiesFilter](docs/models/OpportunitiesFilter.md)
 - [OpportunitiesSort](docs/models/OpportunitiesSort.md)
 - [Opportunity](docs/models/Opportunity.md)
 - [Order](docs/models/Order.md)
 - [OrderCustomers](docs/models/OrderCustomers.md)
 - [OrderDiscounts](docs/models/OrderDiscounts.md)
 - [OrderFulfillments](docs/models/OrderFulfillments.md)
 - [OrderLineItems](docs/models/OrderLineItems.md)
 - [OrderPayments](docs/models/OrderPayments.md)
 - [OrderPickupDetails](docs/models/OrderPickupDetails.md)
 - [OrderPickupDetailsCurbsidePickupDetails](docs/models/OrderPickupDetailsCurbsidePickupDetails.md)
 - [OrderPickupDetailsRecipient](docs/models/OrderPickupDetailsRecipient.md)
 - [OrderRefunds](docs/models/OrderRefunds.md)
 - [OrderTenders](docs/models/OrderTenders.md)
 - [OrderType](docs/models/OrderType.md)
 - [Owner](docs/models/Owner.md)
 - [PaginationCoverage](docs/models/PaginationCoverage.md)
 - [Passthrough](docs/models/Passthrough.md)
 - [Payment](docs/models/Payment.md)
 - [PaymentAllocations](docs/models/PaymentAllocations.md)
 - [PaymentCard](docs/models/PaymentCard.md)
 - [PaymentRequiredResponse](docs/models/PaymentRequiredResponse.md)
 - [PaymentUnit](docs/models/PaymentUnit.md)
 - [Payroll](docs/models/Payroll.md)
 - [PayrollTotals](docs/models/PayrollTotals.md)
 - [PayrollsFilter](docs/models/PayrollsFilter.md)
 - [PhoneNumber](docs/models/PhoneNumber.md)
 - [Pipeline](docs/models/Pipeline.md)
 - [PipelineStages](docs/models/PipelineStages.md)
 - [PosBankAccount](docs/models/PosBankAccount.md)
 - [PosBankAccountAchDetails](docs/models/PosBankAccountAchDetails.md)
 - [PosPayment](docs/models/PosPayment.md)
 - [PosPaymentCardDetails](docs/models/PosPaymentCardDetails.md)
 - [PosPaymentExternalDetails](docs/models/PosPaymentExternalDetails.md)
 - [Price](docs/models/Price.md)
 - [ProfitAndLoss](docs/models/ProfitAndLoss.md)
 - [ProfitAndLossExpenses](docs/models/ProfitAndLossExpenses.md)
 - [ProfitAndLossFilter](docs/models/ProfitAndLossFilter.md)
 - [ProfitAndLossGrossProfit](docs/models/ProfitAndLossGrossProfit.md)
 - [ProfitAndLossIncome](docs/models/ProfitAndLossIncome.md)
 - [ProfitAndLossNetIncome](docs/models/ProfitAndLossNetIncome.md)
 - [ProfitAndLossNetOperatingIncome](docs/models/ProfitAndLossNetOperatingIncome.md)
 - [ProfitAndLossRecord](docs/models/ProfitAndLossRecord.md)
 - [ProfitAndLossRecords](docs/models/ProfitAndLossRecords.md)
 - [ProfitAndLossSection](docs/models/ProfitAndLossSection.md)
 - [RequestCountAllocation](docs/models/RequestCountAllocation.md)
 - [ResolveWebhookEventRequest](docs/models/ResolveWebhookEventRequest.md)
 - [ResolveWebhookEventsRequest](docs/models/ResolveWebhookEventsRequest.md)
 - [ResolveWebhookResponse](docs/models/ResolveWebhookResponse.md)
 - [ResourceStatus](docs/models/ResourceStatus.md)
 - [Schedule](docs/models/Schedule.md)
 - [ScheduleWorkPattern](docs/models/ScheduleWorkPattern.md)
 - [ScheduleWorkPatternOddWeeks](docs/models/ScheduleWorkPatternOddWeeks.md)
 - [ServiceCharge](docs/models/ServiceCharge.md)
 - [ServiceCharges](docs/models/ServiceCharges.md)
 - [Session](docs/models/Session.md)
 - [SessionSettings](docs/models/SessionSettings.md)
 - [SessionTheme](docs/models/SessionTheme.md)
 - [SharedLink](docs/models/SharedLink.md)
 - [SharedLinkTarget](docs/models/SharedLinkTarget.md)
 - [SimpleFormFieldOption](docs/models/SimpleFormFieldOption.md)
 - [SocialLink](docs/models/SocialLink.md)
 - [SortDirection](docs/models/SortDirection.md)
 - [Status](docs/models/Status.md)
 - [Supplier](docs/models/Supplier.md)
 - [SupportedProperty](docs/models/SupportedProperty.md)
 - [SupportedPropertyChildProperties](docs/models/SupportedPropertyChildProperties.md)
 - [Tags](docs/models/Tags.md)
 - [Tax](docs/models/Tax.md)
 - [TaxRate](docs/models/TaxRate.md)
 - [TaxRatesFilter](docs/models/TaxRatesFilter.md)
 - [Tender](docs/models/Tender.md)
 - [TimeOffRequest](docs/models/TimeOffRequest.md)
 - [TimeOffRequestNotes](docs/models/TimeOffRequestNotes.md)
 - [TimeOffRequestsFilter](docs/models/TimeOffRequestsFilter.md)
 - [TooManyRequestsResponse](docs/models/TooManyRequestsResponse.md)
 - [TooManyRequestsResponseDetail](docs/models/TooManyRequestsResponseDetail.md)
 - [UnauthorizedResponse](docs/models/UnauthorizedResponse.md)
 - [UnexpectedErrorResponse](docs/models/UnexpectedErrorResponse.md)
 - [UnifiedApiId](docs/models/UnifiedApiId.md)
 - [UnifiedFile](docs/models/UnifiedFile.md)
 - [UnifiedId](docs/models/UnifiedId.md)
 - [UnprocessableResponse](docs/models/UnprocessableResponse.md)
 - [UpdateActivityResponse](docs/models/UpdateActivityResponse.md)
 - [UpdateBillResponse](docs/models/UpdateBillResponse.md)
 - [UpdateCompanyResponse](docs/models/UpdateCompanyResponse.md)
 - [UpdateConnectionResponse](docs/models/UpdateConnectionResponse.md)
 - [UpdateContactResponse](docs/models/UpdateContactResponse.md)
 - [UpdateCreditNoteResponse](docs/models/UpdateCreditNoteResponse.md)
 - [UpdateCustomerResponse](docs/models/UpdateCustomerResponse.md)
 - [UpdateCustomerSupportCustomerResponse](docs/models/UpdateCustomerSupportCustomerResponse.md)
 - [UpdateDepartmentResponse](docs/models/UpdateDepartmentResponse.md)
 - [UpdateDriveGroupResponse](docs/models/UpdateDriveGroupResponse.md)
 - [UpdateDriveResponse](docs/models/UpdateDriveResponse.md)
 - [UpdateEmployeeResponse](docs/models/UpdateEmployeeResponse.md)
 - [UpdateFileResponse](docs/models/UpdateFileResponse.md)
 - [UpdateFolderRequest](docs/models/UpdateFolderRequest.md)
 - [UpdateFolderResponse](docs/models/UpdateFolderResponse.md)
 - [UpdateHrisCompanyResponse](docs/models/UpdateHrisCompanyResponse.md)
 - [UpdateInvoiceItemsResponse](docs/models/UpdateInvoiceItemsResponse.md)
 - [UpdateInvoiceResponse](docs/models/UpdateInvoiceResponse.md)
 - [UpdateItemResponse](docs/models/UpdateItemResponse.md)
 - [UpdateJobResponse](docs/models/UpdateJobResponse.md)
 - [UpdateLeadResponse](docs/models/UpdateLeadResponse.md)
 - [UpdateLedgerAccountResponse](docs/models/UpdateLedgerAccountResponse.md)
 - [UpdateLocationResponse](docs/models/UpdateLocationResponse.md)
 - [UpdateMerchantResponse](docs/models/UpdateMerchantResponse.md)
 - [UpdateMessageResponse](docs/models/UpdateMessageResponse.md)
 - [UpdateModifierGroupResponse](docs/models/UpdateModifierGroupResponse.md)
 - [UpdateModifierResponse](docs/models/UpdateModifierResponse.md)
 - [UpdateNoteResponse](docs/models/UpdateNoteResponse.md)
 - [UpdateOpportunityResponse](docs/models/UpdateOpportunityResponse.md)
 - [UpdateOrderResponse](docs/models/UpdateOrderResponse.md)
 - [UpdateOrderTypeResponse](docs/models/UpdateOrderTypeResponse.md)
 - [UpdatePaymentResponse](docs/models/UpdatePaymentResponse.md)
 - [UpdatePipelineResponse](docs/models/UpdatePipelineResponse.md)
 - [UpdatePosPaymentResponse](docs/models/UpdatePosPaymentResponse.md)
 - [UpdateSharedLinkResponse](docs/models/UpdateSharedLinkResponse.md)
 - [UpdateSupplierResponse](docs/models/UpdateSupplierResponse.md)
 - [UpdateTaxRateResponse](docs/models/UpdateTaxRateResponse.md)
 - [UpdateTenderResponse](docs/models/UpdateTenderResponse.md)
 - [UpdateTimeOffRequestResponse](docs/models/UpdateTimeOffRequestResponse.md)
 - [UpdateUploadSessionResponse](docs/models/UpdateUploadSessionResponse.md)
 - [UpdateUserResponse](docs/models/UpdateUserResponse.md)
 - [UpdateWebhookRequest](docs/models/UpdateWebhookRequest.md)
 - [UpdateWebhookResponse](docs/models/UpdateWebhookResponse.md)
 - [UploadSession](docs/models/UploadSession.md)
 - [Url](docs/models/Url.md)
 - [User](docs/models/User.md)
 - [VaultEventType](docs/models/VaultEventType.md)
 - [WalletDetails](docs/models/WalletDetails.md)
 - [Webhook](docs/models/Webhook.md)
 - [WebhookEventLog](docs/models/WebhookEventLog.md)
 - [WebhookEventLogAttempts](docs/models/WebhookEventLogAttempts.md)
 - [WebhookEventLogService](docs/models/WebhookEventLogService.md)
 - [WebhookEventLogsFilter](docs/models/WebhookEventLogsFilter.md)
 - [WebhookEventLogsFilterService](docs/models/WebhookEventLogsFilterService.md)
 - [WebhookEventType](docs/models/WebhookEventType.md)
 - [WebhookSubscription](docs/models/WebhookSubscription.md)
 - [WebhookSupport](docs/models/WebhookSupport.md)
 - [Website](docs/models/Website.md)


## Documentation For Authorization



## apiKey


- **Type**: API key
- **API key parameter name**: Authorization
- **Location**: HTTP header



## applicationId


- **Type**: API key
- **API key parameter name**: x-apideck-app-id
- **Location**: HTTP header



## consumerId


- **Type**: API key
- **API key parameter name**: x-apideck-consumer-id
- **Location**: HTTP header


## Author




## Notes for Large OpenAPI documents

If the OpenAPI document is large, imports in apideck.apis and apideck.models may fail with a
RecursionError indicating the maximum recursion limit has been exceeded. In that case, there are a couple of solutions:

Solution 1:
Use specific imports for apis and models like:

- `from apideck.api.default_api import DefaultApi`
- `from apideck.model.pet import Pet`

Solution 2:
Before importing the package, adjust the maximum recursion limit as shown below:

```
import sys
sys.setrecursionlimit(1500)
import apideck
from apideck.apis import *
from apideck.models import *
```

